// OpenSSL replacement

#include "crypt.h"

#include <gcrypt.h>

static _Bool DES_ecbn_encrypt(const_DES_cblock *input, DES_cblock *output,
	const void *k, int enc, int algo);
static _Bool DES_encrypt(const unsigned char *input, unsigned char *output,
	long length, const void *k, int enc, int algo, int mode, ...);
static size_t gcry_mpi_get_nbytes(gcry_mpi_t a);

BIGNUM *BN_bin2bn(const unsigned char *s, int len, BIGNUM *ret) {
	gcry_mpi_t r_mpi;
	if (gcry_mpi_scan(&r_mpi, GCRYMPI_FMT_USG, s, len, NULL))
		return NULL;
	if (ret) {
		gcry_mpi_set(ret, r_mpi);
		gcry_mpi_release(r_mpi);
		return ret;
	}
	return r_mpi;
}

int BN_bn2bin(const BIGNUM *a, unsigned char *to) {
	gcry_mpi_t a_copy = gcry_mpi_copy((gcry_mpi_t) a);
	size_t buflen = gcry_mpi_get_nbytes(a_copy);
	size_t nwritten;
	gcry_error_t e = gcry_mpi_print(GCRYMPI_FMT_USG, to, buflen, &nwritten,
		a_copy);
	gcry_mpi_release(a_copy);
	if (e)
		return -1;
	return nwritten;
}

int BN_cmp(BIGNUM *a, BIGNUM *b) {
	int cmp = gcry_mpi_cmp(a, b);
	return cmp < 0 ? -1 : cmp > 0 ? 1 : 0;
}

void BN_free(BIGNUM *a) {
	gcry_mpi_release(a);
}

BIGNUM *BN_new(void) {
	return gcry_mpi_new(0);
}

int BN_num_bytes(const BIGNUM *a) {
	// gcry_mpi_copy should not alter a
	// TODO: fix libgcrypt prototype
	gcry_mpi_t a_copy = gcry_mpi_copy((gcry_mpi_t) a);
	int num_bytes = gcry_mpi_get_nbytes(a_copy);
	gcry_mpi_release(a_copy);
	return num_bytes;
}

int BN_sub(BIGNUM *r, const BIGNUM *a, const BIGNUM *b) {
	gcry_mpi_t a_copy = gcry_mpi_copy((gcry_mpi_t) a);
	gcry_mpi_t b_copy = gcry_mpi_copy((gcry_mpi_t) b);
	gcry_mpi_sub(r, a_copy, b_copy);
	gcry_mpi_release(a_copy);
	gcry_mpi_release(b_copy);
	return 1;
}

_Bool DES_ecb_encrypt(const_DES_cblock *input, DES_cblock *output,
	DES_key_schedule *ks, int enc) {
	return DES_ecbn_encrypt(input, output, ks, enc, GCRY_CIPHER_DES);
}

_Bool DES_ecb2_encrypt(const_DES_cblock *input, DES_cblock *output,
	DES_key_schedule *ks1, DES_key_schedule *ks2, int enc) {
	unsigned char k[3 * 8];
	memcpy(k, ks1, 8);
	memcpy(k + 8, ks2, 8);
	memcpy(k + 2 * 8, ks1, 8);
	return DES_ecbn_encrypt(input, output, k, enc, GCRY_CIPHER_3DES);
}

static _Bool DES_ecbn_encrypt(const_DES_cblock *input, DES_cblock *output,
	const void *k, int enc, int algo) {
	return DES_encrypt(*input, *output, 8, k, enc, algo,
		GCRY_CIPHER_MODE_ECB);
}

_Bool DES_ede3_cbc_encrypt(const unsigned char *input, unsigned char *output,
	long length, DES_key_schedule *ks1, DES_key_schedule *ks2,
	DES_key_schedule *ks3, DES_cblock *ivec, int enc) {
	unsigned char k[3 * 8];
	memcpy(k, ks1, 8);
	memcpy(k + 8, ks2, 8);
	memcpy(k + 2 * 8, ks3, 8);
	return DES_encrypt(input, output, length, k, enc, GCRY_CIPHER_3DES,
		GCRY_CIPHER_MODE_CBC, ivec);
}

static _Bool DES_encrypt(const unsigned char *input, unsigned char *output,
	long length, const void *k, int enc, int algo, int mode, ...) {
	gcry_cipher_hd_t h;
	if (gcry_cipher_open(&h, algo, mode, GCRY_CIPHER_SECURE))
		return 1;
	_Bool error = 1;
	do {
		size_t l = algo == GCRY_CIPHER_DES ? 8 : 3 * 8;
		if (gcry_cipher_setkey(h, k, l))
			break;
		if (mode == GCRY_CIPHER_MODE_CBC) {
			DES_cblock *ivec;
			va_list ap;
			va_start(ap, mode);
			ivec = va_arg(ap, DES_cblock *);
			va_end(ap);
			if (gcry_cipher_setiv(h, ivec, 8))
				break;
		}
		size_t outsize = length;
		// libgcrypt does not pad last block with 0, a compatibility
		// flag in gcry_cipher_open could be implemented
		if (outsize % 8)
			break;
		const unsigned char *in;
		size_t inlen;
		if (input == (void *) output) {
			in = NULL;
			inlen = 0;
		}
		else {
			in = input;
			inlen = length;
		}
		typeof(gcry_cipher_encrypt) *f = enc ? gcry_cipher_encrypt
			: gcry_cipher_decrypt;
		if (f(h, output, outsize, in, inlen))
			break;
		error = 0;
	} while (0);
	gcry_cipher_close(h);
	return error;
}

void DES_set_key_unchecked(const_DES_cblock *key, DES_key_schedule *schedule) {
	memcpy(schedule, key, sizeof(const_DES_cblock));
}

int RAND_bytes(unsigned char *buf, int num) {
	gcry_randomize(buf, num, GCRY_STRONG_RANDOM);
	return 1;
}

void RSA_free(RSA *rsa) {
	gcry_mpi_release(rsa->n);
	gcry_mpi_release(rsa->e);
	gcry_mpi_release(rsa->d);
	free(rsa);
}

RSA *RSA_new(void) {
	RSA *rsa = malloc(sizeof(RSA));
	rsa->n = NULL;
	rsa->e = NULL;
	rsa->d = gcry_mpi_snew(0);
	return rsa;
}

int RSA_private_decrypt(int flen, unsigned char *from, unsigned char *to,
	RSA *rsa, int padding) {
	gcry_sexp_t skey;
	if (gcry_sexp_build(&skey, NULL, "(private-key(rsa(n %m)(e %m)(d %m)))",
		rsa->n, rsa->e, rsa->d))
		return -1;
	if (padding != RSA_NO_PADDING) {
		gcry_sexp_release(skey);
		return -1;
	}
	gcry_mpi_t from_mpi;
	if (gcry_mpi_scan(&from_mpi, GCRYMPI_FMT_USG, from, flen, NULL)) {
		gcry_sexp_release(skey);
		return -1;
	}
	gcry_sexp_t data;
	gcry_error_t e = gcry_sexp_build(&data, NULL,
		"(enc-val(flags)(rsa(a %m)))", from_mpi);
	gcry_mpi_release(from_mpi);
	if (e) {
		gcry_sexp_release(skey);
		return -1;
	}
	gcry_sexp_t r_plain;
	gcry_pk_decrypt(&r_plain, data, skey);
	gcry_sexp_release(skey);
	gcry_sexp_release(data);
	if (!r_plain)
		return -1;
	gcry_mpi_t a_mpi = gcry_sexp_nth_mpi(r_plain, 1, GCRYMPI_FMT_USG);
	gcry_sexp_release(r_plain);
	if (!a_mpi)
		return -1;
	size_t n_size = gcry_mpi_get_nbytes(rsa->n);
	size_t a_size = gcry_mpi_get_nbytes(a_mpi);
	size_t offset = n_size - a_size;
	memset(to, 0, offset);
	e = gcry_mpi_print(GCRYMPI_FMT_USG, to + offset, a_size, NULL, a_mpi);
	gcry_mpi_release(a_mpi);
	if (e)
		return -1;
	return n_size;
}

int RSA_public_encrypt(int flen, unsigned char *from, unsigned char *to,
	RSA *rsa, int padding) {
	gcry_sexp_t pkey;
	if (gcry_sexp_build(&pkey, NULL, "(public-key(rsa(n %m)(e %m)))",
		rsa->n, rsa->e))
		return -1;
	if (padding != RSA_NO_PADDING) {
		gcry_sexp_release(pkey);
		return -1;
	}
	gcry_mpi_t from_mpi;
	if (gcry_mpi_scan(&from_mpi, GCRYMPI_FMT_USG, from, flen, NULL)) {
		gcry_sexp_release(pkey);
		return -1;
	}
	gcry_sexp_t data;
	gcry_error_t e = gcry_sexp_build(&data, NULL,
		"(data(flags raw)(value %m))", from_mpi);
	gcry_mpi_release(from_mpi);
	if (e) {
		gcry_sexp_release(pkey);
		return -1;
	}
	gcry_sexp_t r_ciph;
	gcry_pk_encrypt(&r_ciph, data, pkey);
	gcry_sexp_release(pkey);
	gcry_sexp_release(data);
	if (!r_ciph)
		return -1;
	gcry_sexp_t a_sexp;
	a_sexp = gcry_sexp_find_token(r_ciph, "a", 0);
	gcry_sexp_release(r_ciph);
	if (!a_sexp)
		return -1;
	gcry_mpi_t a_mpi = gcry_sexp_nth_mpi(a_sexp, 1, GCRYMPI_FMT_USG);
	gcry_sexp_release(a_sexp);
	if (!a_mpi)
		return -1;
	size_t n_size = gcry_mpi_get_nbytes(rsa->n);
	size_t a_size = gcry_mpi_get_nbytes(a_mpi);
	size_t offset = n_size - a_size;
	memset(to, 0, offset);
	e = gcry_mpi_print(GCRYMPI_FMT_USG, to + offset, a_size, NULL, a_mpi);
	gcry_mpi_release(a_mpi);
	if (e)
		return -1;
	return n_size;
}

unsigned char *SHA1(const unsigned char *d, unsigned long n,
	unsigned char *md) {
	gcry_md_hash_buffer(GCRY_MD_SHA1, md, d, n);
	return md;
}

static size_t gcry_mpi_get_nbytes(gcry_mpi_t a) {
	unsigned nbits = gcry_mpi_get_nbits(a);
	size_t nbytes = nbits >> 3;
	if (nbits % 8)
		nbytes++;
	return nbytes;
}
