// Text

#include "stdproc/text.h"

#include <err.h>
#include <bestiola/utf8.h>

#include "control_points.h"
#include "core.h"
#include "font.h"
#include "graphic.h"
#include "list.h"

typedef struct {
	const struct font *fnt;
	h_align halign;
	v_align valign;
} text_private;

static size_t private_offset;

static void utf8_parse_action(unsigned code, void *data);

static text_private *get_private(text *t) {
	void *p = t;
	return p + private_offset;
}

static void on_kill(process *p) {
	graphic_destroy(p->graph);
}

void set_graph_text(text *t_p, const char *value) {
	text_private *t = get_private(t_p);

	list *codes = ocell_calloc(1, sizeof(list));
	if (utf8_parse(value, utf8_parse_action, codes)) {
		warnx("Invalid UTF-8 sequence");
		fatal_error();
	}

	int width = 0;
	list_entry *entry;
	for (entry = codes->first; entry; entry = entry->next) {
		unsigned code = entry->u;
		const SDL_Surface *glyph = font_glyph(t->fnt, code);
		if (!glyph) {
			warnx("Missing glyph %x", code);
			fatal_error();
		}
		width += glyph->w;
	}

	int height = font_glyph(t->fnt, codes->first->u)->h;
	SDL_Surface *text_surface = new_map(width, height);

	width = 0;
	for (entry = codes->first; entry; entry = entry->next) {
		SDL_Rect target;
		target.x = width;
		target.y = 0;
		SDL_Surface *glyph = font_glyph(t->fnt, entry->u);
		SDL_BlitSurface(glyph, NULL, text_surface, &target);
		width += glyph->w;
	}
	list_destroy(codes);

	process *p = TEXT_CAST(t_p, process);
	p->graph = graphic_from_surface(text_surface);

	const control_point *center = get_center(p->graph);
	control_point new_center;
	switch (t->halign) {
	case LEFT: ;
		new_center.x = 0;
		break;
	case CENTER:
		new_center.x = center->x;
		break;
	case RIGHT:
		new_center.x = width - 1;
	}
	switch (t->valign) {
	case TOP:
		new_center.y = 0;
		break;
	case MIDDLE:
		new_center.y = center->y;
		break;
	case BOTTOM:
		new_center.y = height - 1;
	}
	set_center(p->graph, &new_center);
}

text *text_create(process *father, const font *fnt, int x, int y,
	h_align halign, v_align valign, const char *value) {
	text *t = alloc(text_sizeof());
	text_init(t, father, fnt, x, y, halign, valign, value);
	process *p = TEXT_CAST(t, process);
	p->run(p);
	return t;
}

void text_init(text *t_p, process *father, const struct font *fnt, int x, int y,
	h_align halign, v_align valign, const char *value) {
	process *p = TEXT_CAST(t_p, process);
	process_init(p, father);
	p->x = x;
	p->y = y;
	p->z = -256;
	p->on_kill = on_kill;
	text_private *t = get_private(t_p);
	t->fnt = fnt;
	t->halign = halign;
	t->valign = valign;
	if (value)
		set_graph_text(t_p, value);
}

size_t text_sizeof(void) {
	static size_t size;
	if (!size) {
		private_offset = process_sizeof();
		size = private_offset + sizeof(text_private);
	}
	return size;
}

static void utf8_parse_action(unsigned code, void *data) {
	list *codes = data;
	list_create_entry(codes)->u = code;
}
