// Core

#define _GNU_SOURCE

#include "core.h"

#include <err.h>
#include <math.h>
#include <time.h>
#include <bestiola/compare.h>
#include <bestiola/runtime.h>
#include <GL/gl.h>

#include "atomic.h"
#include "control_points.h"
#include "graphic.h"
#include "graphic_internal.h"
#include "image.h"
#include "int_set.h"
#include "list.h"
#include "process.h"

typedef struct {
	Uint8 *data;
	int bytes;
} audio_chunk;

typedef struct {
	process *p;
	unsigned order;
} process_entry;

struct sample_channel {
	Sound_Sample *sample;
	int decoded_pos;
	int decoded_len;
	int volume;
	int loops;
	atomic_pointer *channel;
	list_entry *entry;
};

int fps;
bool full_screen;
struct mouse mouse;
bool quit_requested;
void (*resize_callback)(int w, int h);

static list channels;
static unsigned chunk_samples = 2 * 44100 * 40 / 1000;
static unsigned frame_interval = 40;
static unsigned frame_max_skip;
static unsigned frame_skip;
static bool initialized_SDL;
static int_set *keys_down;
static Uint8 *keystate;
static unsigned last_time;
static SDL_AudioSpec obtained;
static list processes;
static bool restart_run_loop;
static SDL_Surface *screen;

static void advance_coord(int angle, int distance, int coord[2]);
void audio_callback(void *userdata, Uint8 *stream, int len);
static audio_chunk *next_audio_chunk(void);
static bool play_channel(sample_channel *channel, int *audio_buffer, int len);
static void process_on_kill(process *p);
static void process_render(process *p);
static void setup_opengl(void);
static void update_fps(void);
static void update_mouse_state(void);
static void wait_next_frame(void);

list_entry *add_reference(process **reference) {
	process_private *p = get_private(*reference);
	if (!p->references)
		p->references = ocell_calloc(1, sizeof(list));
	list_entry *entry = list_create_entry(p->references);
	entry->p = reference;
	return entry;
}

void advance(process *p, int distance) {
	advance_coord(p->angle, distance, &p->x);
}

void advance_angle(process *p, int distance, int coord[2]) {
	advance_vector(p, distance, coord, p->angle);
}

static void advance_coord(int angle, int distance, int coord[2]) {
	float radians = angle / 1000.0f / 180 * M_PI;
	coord[0] += distance * cosf(radians);
	coord[1] -= distance * sinf(radians);
}

void advance_vector(process *p, int distance, int coord[2], int angle) {
	advance_coord(angle, distance, coord);
	unsigned char resolution = get_private(p)->resolution;
	p->x = coord[0] >> resolution;
	p->y = coord[1] >> resolution;
}

void *alloc(size_t size) {
	if (!size)
		return NULL;
	void *mem = malloc(size);
	if (!mem)
		fatal_error();
	return mem;
}

void audio_callback(__attribute__((unused)) void *userdata, Uint8 *stream,
	int len) {
	static audio_chunk *current_chunk;
	static int remaining;

	while (len) {
		if (!current_chunk) {
			current_chunk = next_audio_chunk();
			remaining = current_chunk->bytes;
		}
		if (len >= remaining) {
			memcpy(stream, current_chunk->data
				+ current_chunk->bytes - remaining, remaining);
			free(current_chunk->data);
			free(current_chunk);
			current_chunk = NULL;
			stream += remaining;
			len -= remaining;
		}
		else {
			memcpy(stream, current_chunk->data
				+ current_chunk->bytes - remaining, len);
			remaining -= len;
			return;
		}
	}

	__attribute__((destructor)) void destructor(void) {
		if (current_chunk) {
			free(current_chunk->data);
			free(current_chunk);
		}
	}
}

static audio_chunk *audio_chunk_create(void *data, int length) {
	audio_chunk *ac = alloc(sizeof(audio_chunk));
	ac->data = data;
	ac->bytes = sizeof(short) * length;
	return ac;
}

static int compare_depth(const void *p1, const void *p2) {
	const process_entry *e1 = p1;
	const process_entry *e2 = p2;
	int result = intcmp(e1->p->z, e2->p->z);
	if (result)
		return -result;
	return uintcmp(e1->order, e2->order);
}

static int compare_priority(const void *p1, const void *p2) {
	const process_entry *e1 = p1;
	const process_entry *e2 = p2;
	int result = intcmp(e1->p->priority, e2->p->priority);
	if (result)
		return -result;
	return uintcmp(e1->order, e2->order);
}

static void destroy_framework(void) {
	Sound_Quit();
	SDL_Quit();
	initialized_SDL = false;
	free(keys_down);
}

void fatal_error(void) {
	if (initialized_SDL)
		SDL_Quit();
	exit(EXIT_FAILURE);
}

int fget_angle(int x1, int y1, int x2, int y2) {
	return atan2(y1 - y2, x2 - x1) / M_PI * 180 * 1000;
}

bool frame(process *p, void *state, int percent) {
	p->state = state;
	process_private *p_priv = get_private(p);
	p_priv->frame_percent += percent;
	return p_priv->frame_percent >= 100 || p_priv->status != S_WAKEUP;
}

void frame_done(process *p) {
	process_private *p_priv = get_private(p);
	p_priv->frame_percent += 100;
}

char *full_name(const char *file_name) {
	return ocell_asprintf("%s/%s", runtime_share_dir(), file_name);
}

process *get_collided(process *p, list *ids) {
	list_entry *entry;
	for (entry = ids->first; entry; entry = entry->next) {
		process *id = entry->p;
		if (collision(p, id))
			return id;
	}
	return NULL;
}

const SDL_Surface *get_screen(void) {
	return screen;
}

static void init_audio(void) {
	SDL_AudioSpec desired;
	desired.freq = 44100;
	desired.format = AUDIO_S16SYS;
	desired.channels = 2;
	desired.samples = 512;
	desired.callback = audio_callback;
	desired.userdata = NULL;

	if (SDL_OpenAudio(&desired, &obtained) != 0) {
		warnx("SDL_OpenAudio: %s", SDL_GetError());
		fatal_error();
	}

	if (!Sound_Init()) {
		warnx("Sound_Init: %s", Sound_GetError());
		fatal_error();
	}

	SDL_PauseAudio(0);
}

static void init_framework(void) {
	if (SDL_Init(SDL_INIT_TIMER | SDL_INIT_AUDIO | SDL_INIT_VIDEO
		| SDL_INIT_JOYSTICK)) {
		warnx("SDL_Init: %s", SDL_GetError());
		exit(EXIT_FAILURE);
	}
	initialized_SDL = true;

	init_audio();
	keys_down = int_set_create();
	keystate = SDL_GetKeyState(NULL);
	SDL_ShowCursor(SDL_DISABLE);
	srandom(time(NULL));

	last_time = SDL_GetTicks();
}

bool key(int k) {
	return keystate[k];
}

bool key_down(int k) {
	return int_set_find(keys_down, k);
}

void let_me_alone(process *p) {
	list_entry *entry;
	for (entry = processes.first; entry; entry = entry->next) {
		process *p2 = entry->p;
		if (p2 == p)
			continue;
		send_signal(p2, S_KILL);
	}
}

Sound_Sample *load_sample(const char *_file_name) {
	char *file_name = full_name(_file_name);
	Sound_AudioInfo desired;
	desired.format = obtained.format;
	desired.channels = obtained.channels;
	desired.rate = obtained.freq;
	Sound_Sample *sample = Sound_NewSampleFromFile(file_name, &desired,
		obtained.size);
	if (!sample)
		warnx("Cannot load %s\nSound_NewSampleFromFile: %s", _file_name,
			Sound_GetError());
	free(file_name);
	return sample;
}

static unsigned min(unsigned a, unsigned b) {
	return a < b? a: b;
}

int near_angle(int src_angle, int dst_angle, int max_inc) {
	src_angle %= 360000;
	if (src_angle < 0)
		src_angle += 360000;
	dst_angle %= 360000;
	if (dst_angle < 0)
		dst_angle += 360000;
	int diff_angle = abs(dst_angle - src_angle);
	if (diff_angle <= max_inc)
		return dst_angle;
	if (dst_angle > src_angle) {
		if (diff_angle <= 180000)
			return src_angle + max_inc;
		src_angle -= max_inc;
		if (src_angle < 0)
			src_angle += 360000;
	}
	else {
		if (diff_angle <= 180000)
			return src_angle - max_inc;
		src_angle += max_inc;
		if (src_angle >= 360000)
			src_angle -= 360000;
	}
	return src_angle;
}

SDL_Surface *new_map(int width, int height) {
	Uint32 rmask;
	Uint32 gmask;
	Uint32 bmask;
	Uint32 amask;
	if (BYTE_ORDER == BIG_ENDIAN) {
		rmask = 0xff000000;
		gmask = 0x00ff0000;
		bmask = 0x0000ff00;
		amask = 0x000000ff;
	}
	else {
		rmask = 0x000000ff;
		gmask = 0x0000ff00;
		bmask = 0x00ff0000;
		amask = 0xff000000;
	}
	SDL_Surface *surface = SDL_CreateRGBSurface(0, width, height, 32,
		rmask, gmask, bmask, amask);
	SDL_FillRect(surface, NULL, 0);
	return surface;
}

static audio_chunk *next_audio_chunk(void) {
	int *audio_buffer = calloc(chunk_samples, sizeof(int));
	if (!audio_buffer)
		fatal_error();

	list_entry *entry = channels.first;
	while (entry) {
		sample_channel *channel = entry->p;
		if (play_channel(channel, audio_buffer, chunk_samples)) {
			if (channel->channel)
				set(channel->channel, NULL);
			free(channel);
			list_entry *next = entry->next;
			erase_entry(&channels, entry);
			entry = next;
		}
		else
			entry = entry->next;
	}

	short *clipped_buffer = alloc(chunk_samples * sizeof(short));
	unsigned i;
	for (i = 0; i < chunk_samples; i++) {
		clipped_buffer[i] = audio_buffer[i] >= 32767? 32767:
			audio_buffer[i] <= -32768? -32768:
			audio_buffer[i];
	}
	free(audio_buffer);
	return audio_chunk_create(clipped_buffer, chunk_samples);
}

char *ocell_asprintf(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		fatal_error();
	return str;
}

void *ocell_calloc(size_t nmemb, size_t size) {
	if (!nmemb || !size)
		return NULL;
	void *mem = calloc(nmemb, size);
	if (!mem)
		fatal_error();
	return mem;
}

void *ocell_realloc(void *ptr, size_t size) {
	void *new = realloc(ptr, size);
	if (!new)
		fatal_error();
	return new;
}

static bool play_channel(sample_channel *channel, int *audio_buffer, int len) {
	Sound_Sample *sample = channel->sample;
	if (!sample) {
		warnx("Empty sample");
		return true;
	}

	short *buffer = (short *) sample->buffer;
	int decoded_pos = channel->decoded_pos;
	int decoded_len = channel->decoded_len;
	int i;
	for (i = 0; i < len; i++) {
		while (decoded_pos >= decoded_len) {
			if (sample->flags & SOUND_SAMPLEFLAG_EOF) {
				Sound_Rewind(sample);
				int loops = channel->loops;
				if (loops != -1) {
					if (!loops)
						return true;
					channel->loops = loops - 1;
				}
			}
			Uint32 audio_decoded = Sound_Decode(sample);
			if (audio_decoded < sample->buffer_size) {
				if (sample->flags & SOUND_SAMPLEFLAG_ERROR) {
					warnx("Sound_Decode: %s",
						Sound_GetError());
					fatal_error();
				}
				if (!audio_decoded && !decoded_len
					&& sample->flags
					& SOUND_SAMPLEFLAG_EOF) {
					warnx("Empty sound file");
					return true;
				}
			}
			decoded_len = audio_decoded >> 1;
			decoded_pos = 0;
		}
		audio_buffer[i] += buffer[decoded_pos++] * channel->volume
			/ 100;
	}
	channel->decoded_pos = decoded_pos;
	channel->decoded_len = decoded_len;
	return false;
}

void play_sample(Sound_Sample *sample, int loops, atomic_pointer *_channel) {
	sample_channel *channel = alloc(sizeof(sample_channel));
	channel->sample = sample;
	channel->decoded_pos = 0;
	channel->decoded_len = 0;
	channel->volume = 100;
	channel->loops = loops;
	channel->channel = _channel;
	if (_channel)
		set(_channel, channel);
	SDL_LockAudio();
	list_entry *entry = list_create_entry(&channels);
	entry->p = channel;
	SDL_UnlockAudio();
	channel->entry = entry;
}

void process_init(process *p, process *father) {
	p->father = father;
	p->son = NULL;
	p->smallbro = NULL;
	p->bigbro = father->son;
	father->son = p;
	if (p->bigbro)
		p->bigbro->smallbro = p;
	p->priority = 0;
	p->angle = 0;
	p->size = 100;
	p->size_x = 100;
	p->size_y = 100;
	p->alpha = OPAQUE;
	p->z = 0;
	p->graph = NULL;
	p->state = NULL;
	p->on_kill = process_on_kill;
	p->render = process_render;
	p->run = frame_done;

	process_private *p_priv = get_private(p);
	p_priv->status = S_WAKEUP;
	p_priv->frame_percent = 0;
	p_priv->resolution = 0;
	p_priv->references = NULL;

	list_create_entry(&processes)->p = p;
}

static void process_on_kill(__attribute__((unused)) process *p) {
}

static void process_render(process *p) {
	glPushMatrix();
	glTranslatef(p->x, p->y, 0);
	if (p->size != 100 || p->size_x != 100 || p->size_y != 100)
		glScalef((p->size / 100.0f) * (p->size_x / 100.0f),
			(p->size / 100.0f) * (p->size_y / 100.0f), 1);
	if (p->angle)
		glRotatef(p->angle / 1000.0f, 0, 0,
			(p->size_x >= 0 && p->size_y >= 0)
			|| (p->size_x < 0 && p->size_y < 0)? -1: 1);
	glColor4f(1, 1, 1, p->alpha != OPAQUE? p->alpha / (float) OPAQUE: 1);
	graphic_render(p->graph);
	glPopMatrix();
}

void process_set_resolution(process *p, unsigned char resolution) {
	get_private(p)->resolution = resolution;
}

size_t process_sizeof(void) {
	return sizeof(process_private);
}

void program_init(process *p) {
	struct {
		process *father;
		process *son;
	} dummy;
	dummy.son = NULL;
	process_init(p, (process *) &dummy);
	p->father = NULL;
}

static void rearrange_family(process *p) {
	if (p->father && p->father->son == p)
		p->father->son = p->bigbro;
	if (p->smallbro)
		p->smallbro->bigbro = p->bigbro;
	if (p->bigbro)
		p->bigbro->smallbro = p->smallbro;
	process *orphan = p->son;
	while (orphan) {
		orphan->father = NULL;
		orphan = orphan->bigbro;
	}
}

void remove_all_channels(void) {
	SDL_LockAudio();
	list_entry *entry;
	for (entry = channels.first; entry; entry = entry->next) {
		sample_channel *channel = entry->p;
		if (channel->channel)
			set(channel->channel, NULL);
		free(channel);
	}
	list_clear(&channels);
	SDL_UnlockAudio();
}

void remove_channel(atomic_pointer *_channel) {
	SDL_LockAudio();
	sample_channel *channel = get(_channel);
	if (channel)
		erase_entry(&channels, channel->entry);
	SDL_UnlockAudio();
	if (channel) {
		Sound_Rewind(channel->sample);
		free(channel);
		set(_channel, NULL);
	}
}

void remove_reference(process *p, list_entry *entry) {
	erase_entry(get_private(p)->references, entry);
}

void run_program(void) {
	if (runtime_init_failed())
		fatal_error();

	init_framework();
	while (processes.size) {
		SDL_Event event;
		while (SDL_PollEvent(&event)) {
			switch (event.type) {
			case SDL_KEYDOWN:
				int_set_push_back(keys_down,
					event.key.keysym.sym);
				break;
			case SDL_QUIT:
				quit_requested = true;
				break;
			case SDL_VIDEORESIZE:
				if (resize_callback)
					resize_callback(event.resize.w,
						event.resize.h);
				break;
			}
		}
		sort(keys_down);
		update_mouse_state();

		unsigned size = processes.size;
		process_entry *prioritized
			= alloc(size * sizeof(process_entry));
		process_entry *prioritized_entry = prioritized;
		unsigned order = 0;
		list_entry *entry;
		for (entry = processes.first; entry; entry = entry->next) {
			prioritized_entry->p = entry->p;
			prioritized_entry->order = order++;
			prioritized_entry++;
		}
		qsort(prioritized, size, sizeof(process_entry),
			compare_priority);
		prioritized_entry = prioritized;
		for (order = 0; order < size;) {
			process *p = prioritized_entry->p;
			process_private *p_priv = get_private(p);
			if (p_priv->status == S_WAKEUP
				&& p_priv->frame_percent < 100)
				p->run(p);
			if (restart_run_loop) {
				restart_run_loop = false;
				prioritized_entry = prioritized;
				order = 0;
				continue;
			}
			prioritized_entry++;
			order++;
		}
		free(prioritized);
		clear(keys_down);

		size = processes.size;
		process_entry *visible = alloc(size * sizeof(process_entry));
		process_entry *visible_entry = visible;
		order = 0;
		for (entry = processes.first; entry;) {
			process *p = entry->p;
			process_private *p_priv = get_private(p);
			if (p_priv->status == S_KILL) {
				free(p);
				list_entry *next = entry->next;
				erase_entry(&processes, entry);
				entry = next;
				continue;
			}
			if (p_priv->status != S_SLEEP && p->graph) {
				visible_entry->p = entry->p;
				visible_entry->order = order++;
				visible_entry++;
			}
			if (p_priv->frame_percent >= 100)
				p_priv->frame_percent -= 100;
			entry = entry->next;
		}
		if (frame_skip) {
			if (!--frame_skip)
				last_time = SDL_GetTicks();
			free(visible);
			continue;
		}
		if (screen) {
			glClear(GL_COLOR_BUFFER_BIT);
			qsort(visible, order, sizeof(process_entry),
				compare_depth);
			visible_entry = visible;
			unsigned u;
			for (u = 0; u < order; u++) {
				process *p = visible_entry++->p;
				p->render(p);
			}
			SDL_GL_SwapBuffers();
		}
		free(visible);

		update_fps();
		wait_next_frame();
	}
	destroy_framework();
}

void sample_channel_set_volume(sample_channel *channel, int volume) {
	channel->volume = volume;
}

int sample_channel_volume(const sample_channel *channel) {
	return channel->volume;
}

void send_signal(process *p, signal_t s) {
	send_signal_tree(p, s, false);
}

void send_signal_tree(process *p, signal_t s, bool tree) {
	process_private *p_priv = get_private(p);
	if (p_priv->status == S_KILL)
		return;

	if (tree) {
		process *son = p->son;
		while (son) {
			send_signal_tree(son, s, true);
			son = son->bigbro;
		}
	}

	switch (s) {
	case S_KILL:
		p->on_kill(p);
		rearrange_family(p);
		if (p_priv->references) {
			list_entry *entry;
			for (entry = p_priv->references->first; entry;
				entry = entry->next) {
				void **p_ref = entry->p;
				*p_ref = NULL;
			}
			list_destroy(p_priv->references);
		}
		break;
	case S_WAKEUP:
		if (p_priv->status != S_WAKEUP)
			restart_run_loop = true;
		break;
	default: ;
	}
	p_priv->status = s;
}

void set_fps(unsigned fps, unsigned skip) {
	frame_interval = fps? 1000 / fps: 0;
	frame_max_skip = skip;
	SDL_LockAudio();
	chunk_samples = 2 * 44100 * frame_interval / 1000;
	SDL_UnlockAudio();
}

void set_mode(int width, int height) {
	unsigned flags = SDL_OPENGL;
	if (full_screen)
		flags |= SDL_FULLSCREEN;
	else if (resize_callback)
		flags |= SDL_RESIZABLE;

	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

	screen = SDL_SetVideoMode(width, height, 0, flags);
	if (!screen) {
		warnx("Couldn't set GL mode: %s", SDL_GetError());
		fatal_error();
	}

	setup_opengl();
}

void set_title(const char *title) {
	SDL_WM_SetCaption(title, title);
	SDL_Surface *icon = load_image("icon.xpm");
	SDL_WM_SetIcon(icon, NULL);
	SDL_FreeSurface(icon);
}

static void setup_opengl(void) {
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glEnable(GL_TEXTURE_RECTANGLE_ARB);

	glClearColor(0, 0, 0, 0);
	glViewport(0, 0, screen->w, screen->h);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, screen->w, screen->h, 0, -1, 1);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glClear(GL_COLOR_BUFFER_BIT);
}

static void update_fps(void) {
	static int frames;
	static unsigned last_check;
	unsigned check = SDL_GetTicks();
	if (check / 1000 > last_check / 1000) {
		fps = frames;
		frames = 1;
	}
	else
		frames++;
	last_check = check;
}

static void update_mouse_state(void) {
	Uint8 state = SDL_GetMouseState(&mouse.x, &mouse.y);
	mouse.left = state & SDL_BUTTON(1);
	mouse.middle = state & SDL_BUTTON(2);
	mouse.right = state & SDL_BUTTON(3);
}

static void wait_next_frame(void) {
	unsigned current_time = SDL_GetTicks();
	unsigned elapsed = current_time - last_time;
	if (frame_interval > elapsed) {
		SDL_Delay(frame_interval - elapsed);
		last_time += frame_interval;
	}
	else {
		frame_skip = min(elapsed / frame_interval - 1, frame_max_skip);
		if (frame_skip)
			return;
		last_time = current_time;
	}
}
