// Get control points

#include "control_points.h"

#include <err.h>
#include <png.h>

#include "core.h"
#include "pointer_map.h"

static inline control_info *to_info(pointer_map *map);
static inline pointer_map *to_map(control_info *info);

control_info *control_info_create(void) {
	return to_info(pointer_map_create());
}

void control_info_destroy(control_info *info) {
	pointer_map_destroy(to_map(info), free);
}

control_info *control_info_from_file(const char *file_name) {
	pointer_map *map = pointer_map_create();

	FILE *file = fopen(file_name, "rb");

	do {
		if (!file)
			break;

		static unsigned PNG_BYTES_TO_CHECK = 8;
		png_byte buf[PNG_BYTES_TO_CHECK];

		if (fread(buf, PNG_BYTES_TO_CHECK, 1, file) != 1)
			break;

		if (!png_check_sig(buf, PNG_BYTES_TO_CHECK))
			break;

		png_structp png_ptr = png_create_read_struct(
			PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
		if (!png_ptr)
			break;

		png_set_sig_bytes(png_ptr, PNG_BYTES_TO_CHECK);

		png_infop info_ptr = png_create_info_struct(png_ptr);
		if (!info_ptr) {
			png_destroy_read_struct(&png_ptr, NULL, NULL);
			break;
		}

		if (setjmp(png_jmpbuf(png_ptr))) {
			png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
			warnx("libpng error");
			fatal_error();
		}

		png_init_io(png_ptr, file);
		png_read_info(png_ptr, info_ptr);

		png_textp text_ptr;
		int num_text;
		if (png_get_text(png_ptr, info_ptr, &text_ptr, &num_text)) {
			int i;
			for (i = 0; i < num_text; i++) {
				static const char TAG_START[] = "CPoint ";
				static size_t TAG_LEN = sizeof(TAG_START) - 1;
				if (strncmp(text_ptr[i].key, TAG_START,
					TAG_LEN))
					continue;
				unsigned num;
				if (sscanf(text_ptr[i].key + TAG_LEN, "%u",
					&num) != 1)
					continue;
				int x;
				int y;
				if (sscanf(text_ptr[i].text, "%d,%d", &x, &y)
					!= 2)
					continue;
				control_point *p = alloc(sizeof(control_point));
				p->x = x;
				p->y = y;
				set_entry(map, num, p);
			}
		}

		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
	} while (0);

	fclose(file);

	return to_info(map);
}

control_point *control_info_get_point(const control_info *info, unsigned n) {
	return get_entry((const pointer_map *) info, n);
}

void control_info_set_point(control_info *info, unsigned n,
	control_point *point) {
	set_entry(to_map(info), n, point);
}

static inline control_info *to_info(pointer_map *map) {
        return (control_info *) map;
}

static inline pointer_map *to_map(control_info *info) {
        return (pointer_map *) info;
}
