// Detect graphic collisions

#include "core.h"

#include <err.h>
#include <math.h>

#include "control_points.h"
#include "graphic.h"
#include "process.h"
#include "super_cast.h"

typedef struct pixel_check pixel_check;
struct pixel_check {
	float coefs[6];
	const control_point *center;
	int w;
	int h;
	bool (*is_opaque)(const pixel_check *c, float x, float y);
};

#define AMASK_CHECK_CAST(p, t) SUPER_CAST(p, t, amask_check, pixel_check, p)

typedef struct amask_check {
	pixel_check super;
	Uint16 scanline;
	Uint32 *pixels;
	Uint32 Amask;
} amask_check;

typedef struct {
	float left;
	float right;
	float top;
	float bottom;
} boundaries;

#define COLORKEY_CHECK_CAST(p, t) \
	SUPER_CAST(p, t, colorkey_check, pixel_check, p)

typedef struct colorkey_check {
	pixel_check super;
	Uint16 scanline;
	Uint8 *pixels;
	Uint32 colorkey;
} colorkey_check;

static void get_boundaries(process *p, boundaries *b);
static int get_offset_x(const pixel_check *c, float x, float y);
static int get_offset_y(const pixel_check *c, float x, float y);
static pixel_check *get_pixel_check(process *p);
static float max_element(float list[4]);
static float min_element(float list[4]);
static bool outside(const pixel_check *c, int x, int y);
static bool pixel_is_opaque(const pixel_check *c, float x, float y);

static bool amask_is_opaque(const pixel_check *pc, float x, float y) {
	const amask_check *c = (const amask_check *) pc;
	int offset_x = get_offset_x(pc, x, y);
	int offset_y = get_offset_y(pc, x, y);
	if (outside(pc, offset_x, offset_y))
		return false;
	return c->pixels[offset_y * c->scanline + offset_x] & c->Amask;
}

static void calculate_coefs(process *p, float *coefs) {
	if (p->angle) {
		float radians = -p->angle / 1000.0f / 180 * M_PI;
		float cos = cosf(radians);
		float sin = sinf(radians);
		coefs[0] = cos;
		coefs[1] = sin;
		coefs[3] = -sin;
		coefs[4] = cos;
	}
	else {
		coefs[0] = 1;
		coefs[1] = 0;
		coefs[3] = 0;
		coefs[4] = 1;
	}
	if (p->size != 100 || p->size_x != 100 || p->size_y != 100) {
		float scale_x = (100.0f / p->size) * (100.0f / p->size_x);
		float scale_y = (100.0f / p->size) * (100.0f / p->size_y);
		coefs[0] *= scale_x;
		coefs[1] *= scale_y;
		coefs[3] *= scale_x;
		coefs[4] *= scale_y;
	}
	coefs[2] = -(coefs[0] * p->x + coefs[1] * p->y);
	coefs[5] = -(coefs[3] * p->x + coefs[4] * p->y);
}

bool collision(process *p, process *p2) {
	if (!p->graph || !p2->graph || get_private(p2)->status == S_SLEEP)
		return false;

	boundaries b;
	get_boundaries(p, &b);

	boundaries b2;
	get_boundaries(p2, &b2);

	if (b2.right < b.left || b.right < b2.left || b2.bottom < b.top
		|| b.bottom < b2.top)
		return false;

	if (b.left < b2.left)
		b.left = b2.left;
	if (b.right > b2.right)
		b.right = b2.right;
	if (b.top < b2.top)
		b.top = b2.top;
	if (b.bottom > b2.bottom)
		b.bottom = b2.bottom;

	pixel_check *check = get_pixel_check(p);
	pixel_check *check2 = get_pixel_check(p2);

	float i;
	for (i = b.top; i <= b.bottom; i++) {
		float j;
		for (j = b.left; j <= b.right; j++)
			if (check->is_opaque(check, j, i)
				&& check2->is_opaque(check2, j, i)) {
				free(check);
				free(check2);
				return true;
			}
	}
	free(check);
	free(check2);
	return false;
}

static bool colorkey_is_opaque(const pixel_check *pc, float x, float y) {
	const colorkey_check *c = (const colorkey_check *) pc;
	int offset_x = get_offset_x(pc, x, y);
	int offset_y = get_offset_y(pc, x, y);
	if (outside(pc, offset_x, offset_y))
		return false;
	return c->pixels[offset_y * c->scanline + offset_x] != c->colorkey;
}

static void get_boundaries(process *p, boundaries *b) {
	float coefs[6];
	coefs[2] = p->x;
	coefs[5] = p->y;
	if (p->size != 100 || p->size_x != 100 || p->size_y != 100) {
		coefs[0] = (p->size / 100.0f) * (p->size_x / 100.0f);
		coefs[4] = (p->size / 100.0f) * (p->size_y / 100.0f);
	}
	else {
		coefs[0] = 1;
		coefs[4] = 1;
	}
	if (p->angle) {
		float radians = p->angle / 1000.0f / 180 * M_PI;
		float cos = cosf(radians);
		float sin = sinf(radians);
		coefs[1] = coefs[0] * sin;
		coefs[3] = coefs[4] * -sin;
		coefs[0] *= cos;
		coefs[4] *= cos;
	}
	else {
		coefs[1] = 0;
		coefs[3] = 0;
	}

	const control_point *center = get_center(p->graph);
	const SDL_Surface *surface = get_surface(p->graph);

	float left = -center->x;
	float right = surface->w - center->x;
	float top = -center->y;
	float bottom = surface->h - center->y;

	float left_top_x = left * coefs[0] + top * coefs[1] + coefs[2];
	float left_top_y = left * coefs[3] + top * coefs[4] + coefs[5];
	float right_top_x = right * coefs[0] + top * coefs[1] + coefs[2];
	float right_top_y = right * coefs[3] + top * coefs[4] + coefs[5];
	float left_bottom_x = left * coefs[0] + bottom * coefs[1] + coefs[2];
	float left_bottom_y = left * coefs[3] + bottom * coefs[4] + coefs[5];
	float right_bottom_x = right * coefs[0] + bottom * coefs[1] + coefs[2];
	float right_bottom_y = right * coefs[3] + bottom * coefs[4] + coefs[5];

	float corners_x[] = { left_top_x, right_top_x, left_bottom_x,
		right_bottom_x };
	float corners_y[] = { left_top_y, right_top_y, left_bottom_y,
		right_bottom_y };

	b->left = min_element(corners_x);
	b->right = max_element(corners_x);
	b->top = min_element(corners_y);
	b->bottom = max_element(corners_y);
}

static int get_offset_x(const pixel_check *c, float x, float y) {
	return x * c->coefs[0] + y * c->coefs[1] + c->coefs[2] + c->center->x;
}

static int get_offset_y(const pixel_check *c, float x, float y) {
	return x * c->coefs[3] + y * c->coefs[4] + c->coefs[5] + c->center->y;
}

static pixel_check *get_pixel_check(process *p) {
	const SDL_Surface *surface = get_surface(p->graph);
	Uint32 Amask = surface->format->Amask;
	if (Amask) {
		if (surface->format->BitsPerPixel != 32)
			warnx("collision: Unexpected format");
		amask_check *acheck = alloc(sizeof(amask_check));
		pixel_check *check = AMASK_CHECK_CAST(acheck, pixel_check);
		calculate_coefs(p, check->coefs);
		check->center = get_center(p->graph);
		check->w = surface->w;
		check->h = surface->h;
		check->is_opaque = amask_is_opaque;
		acheck->scanline = surface->pitch / sizeof(Uint32);
		acheck->pixels = (Uint32 *) surface->pixels;
		acheck->Amask = Amask;
		return check;
	}
	if (surface->flags & SDL_SRCCOLORKEY) {
		if (surface->format->BitsPerPixel != 8)
			warnx("collision: Unexpected format");
		colorkey_check *ccheck = alloc(sizeof(colorkey_check));
		pixel_check *check = COLORKEY_CHECK_CAST(ccheck, pixel_check);
		calculate_coefs(p, check->coefs);
		check->center = get_center(p->graph);
		check->w = surface->w;
		check->h = surface->h;
		check->is_opaque = colorkey_is_opaque;
		ccheck->scanline = surface->pitch;
		ccheck->pixels = (Uint8 *) surface->pixels;
		ccheck->colorkey = surface->format->colorkey;
		return check;
	}

	pixel_check *check = alloc(sizeof(pixel_check));
	calculate_coefs(p, check->coefs);
	check->center = get_center(p->graph);
	check->w = surface->w;
	check->h = surface->h;
	check->is_opaque = pixel_is_opaque;
	return check;
}

static float max_element(float list[4]) {
	float result = list[0];
	size_t i;
	for (i = 1; i < 4; i++)
		if (list[i] > result)
			result = list[i];
	return result;
}

static float min_element(float list[4]) {
	float result = list[0];
	size_t i;
	for (i = 1; i < 4; i++)
		if (list[i] < result)
			result = list[i];
	return result;
}

static bool outside(const pixel_check *c, int x, int y) {
	return x < 0 || x >= c->w || y < 0 || y >= c->h;
}

static bool pixel_is_opaque(const pixel_check *c, float x, float y) {
	int offset_x = get_offset_x(c, x, y);
	int offset_y = get_offset_y(c, x, y);
	return !outside(c, offset_x, offset_y);
}
