// Core

#ifndef OCELL_CORE_H
#define OCELL_CORE_H

#include <stdbool.h>
#include <SDL/SDL.h>
#include <SDL/SDL_sound.h>

struct atomic_pointer;
struct list;

typedef enum {
	S_KILL,
	S_WAKEUP,
	S_SLEEP,
	S_FREEZE
} signal_t;

typedef struct process process;
struct process {
	process *father;
	process *son;
	process *smallbro;
	process *bigbro;

	int priority;

	int x;
	int y;
	int angle;
	int size;
	int size_x;
	int size_y;

	unsigned char alpha;

	int z;
	struct graphic *graph;

	void *state;

	void (*on_kill)(process *p);
	void (*render)(process *p);
	void (*run)(process *p);
};

static const unsigned char TRANSPARENT = 0;
static const unsigned char OPAQUE = 255;

enum {
	_TAB = SDLK_TAB,
	_ENTER = SDLK_RETURN,
	_ESC = SDLK_ESCAPE,
	_SPACE = SDLK_SPACE,
	_1 = SDLK_1,
	_2 = SDLK_2,
	_3 = SDLK_3,
	_4 = SDLK_4,
	_5 = SDLK_5,
	_6 = SDLK_6,
	_7 = SDLK_7,
	_8 = SDLK_8,
	_A = SDLK_a,
	_C = SDLK_c,
	_D = SDLK_d,
	_E = SDLK_e,
	_I = SDLK_i,
	_L = SDLK_l,
	_O = SDLK_o,
	_P = SDLK_p,
	_R = SDLK_r,
	_Q = SDLK_q,
	_S = SDLK_s,
	_T = SDLK_t,
	_X = SDLK_x,
	_Y = SDLK_y,
	_Z = SDLK_z,
	_UP = SDLK_UP,
	_DOWN = SDLK_DOWN,
	_RIGHT = SDLK_RIGHT,
	_LEFT = SDLK_LEFT,
	_END = SDLK_END,
	_F2 = SDLK_F2,
	_F4 = SDLK_F4,
	_F9 = SDLK_F9,
	_CONTROL = SDLK_LCTRL,
	_ALT = SDLK_LALT
};

struct mouse {
	int x;
	int y;
	bool left;
	bool middle;
	bool right;
};

typedef struct sample_channel sample_channel;

extern int fps;
extern bool full_screen;
extern struct mouse mouse;
extern bool quit_requested;
extern void (*resize_callback)(int w, int h);

struct list_entry *add_reference(process **reference);
void advance(process *p, int distance);
void advance_angle(process *p, int distance, int coord[2]);
void advance_vector(process *p, int distance, int coord[2], int angle);
void *alloc(size_t size);
bool collision(process *p, process *p2);
void fatal_error(void);
int fget_angle(int x1, int y1, int x2, int y2);
void first_run(process *p, process *father);

#define STATE_LABEL(s) state ## s
#define FRAME_FUNCTION(p, n, percent) \
	do { \
		if (frame(p, &&STATE_LABEL(n), percent)) \
			return; \
STATE_LABEL(n): ; \
	} while (0)
#define FRAME_PERCENT(p, percent) FRAME_FUNCTION(p, __COUNTER__, percent)
#define FRAME(p) FRAME_PERCENT(p, 100)

bool frame(process *p, void *state, int percent);
void frame_done(process *p);
process *get_collided(process *p, struct list *ids);
char *full_name(const char *file_name);
const SDL_Surface *get_screen(void);
bool key(int k);
bool key_down(int k);
void let_me_alone(process *p);
Sound_Sample *load_sample(const char *_file_name);
int near_angle(int src_angle, int dst_angle, int max_inc);
SDL_Surface *new_map(int width, int height);
char *ocell_asprintf(const char *fmt, ...)
	__attribute__((format(printf, 1, 2)));
void *ocell_calloc(size_t nmemb, size_t size);
void *ocell_realloc(void *ptr, size_t size);
void play_sample(Sound_Sample *sample, int loops,
	struct atomic_pointer *_channel);
void process_init(process *p, process *father);
void process_set_resolution(process *p, unsigned char resolution);
size_t process_sizeof(void);
void program_init(process *p);

static inline int random_between(int min, int max) {
	return min + random() % (max - min + 1);
}

void remove_all_channels(void);
void remove_channel(struct atomic_pointer *_channel);
void remove_reference(process *p, struct list_entry *entry);
void run_program(void);
void sample_channel_set_volume(sample_channel *channel, int volume);
int sample_channel_volume(const sample_channel *channel);
void send_signal(process *p, signal_t s);
void send_signal_tree(process *p, signal_t s, bool tree);
void set_fps(unsigned fps, unsigned skip);
void set_mode(int width, int height);
void set_title(const char *title);

#endif
