// pointer_map.c - Map of pointers

#define _GNU_SOURCE

#include "pointer_map.h"

#include <search.h>
#include <bestiola/compare.h>

#include "core.h"

typedef struct {
	unsigned key;
	void *value;
} entry;

struct pointer_map {
	void *root;
};

static void (*cleanup_value)(void *);

static int compare_entries(const void *p1, const void *p2) {
	const entry *e1 = p1;
	const entry *e2 = p2;
	return uintcmp(e1->key, e2->key);
}

static void free_node(void *datap) {
	entry *e = datap;
	cleanup_value(e->value);
	free(e);
}

void *get_entry(const pointer_map *map, unsigned key) {
	entry e;
	e.key = key;
	entry **result = tfind(&e, &map->root, compare_entries);
	return result ? (*result)->value : NULL;
}

pointer_map *pointer_map_create(void) {
	return ocell_calloc(1, sizeof(pointer_map));
}

void pointer_map_destroy(pointer_map *map, void (*cleanup)(void *)) {
	void (*old_value)(void *) = cleanup_value;
	cleanup_value = cleanup;
	tdestroy(map->root, free_node);
	free(map);
	cleanup_value = old_value;
}

void set_entry(pointer_map *map, unsigned key, void *value) {
	entry *e = alloc(sizeof(entry));
	e->key = key;
	e->value = value;

	if (!value) {
		tdelete(e, &map->root, compare_entries);
		free(e);
		return;
	}

	entry **result = tsearch(e, &map->root, compare_entries);
	if (!result)
		fatal_error();
	entry *found = *result;
	if (found != e) {
		found->value = value;
		free(e);
	}
}
