// Graphic

#include "graphic.h"

#include <err.h>
#include <GL/gl.h>

#include "control_points.h"
#include "core.h"
#include "image.h"

struct graphic {
	SDL_Surface *surface;
	GLuint texture;
	control_info *points;
};

static void init(graphic *g);
static void set_texture(graphic *g, const SDL_Surface *surface);

void convert_palette(graphic *g, const SDL_Color *colors) {
	SDL_Surface *surface = g->surface;
	SDL_Palette *palette = surface->format->palette;
	if (!palette)
		return;
	if (!SDL_SetColors(surface, colors, 0, palette->ncolors))
		return;

	SDL_Surface *rgba = new_map(surface->w, surface->h);
	SDL_BlitSurface(surface, NULL, rgba, NULL);
	set_texture(g, rgba);
	SDL_FreeSurface(rgba);
}

const control_point *get_center(const graphic *g) {
	return control_info_get_point(g->points, 0);
}

const SDL_Surface *get_surface(const graphic *g) {
	return g->surface;
}

void graphic_destroy(graphic *g) {
	SDL_FreeSurface(g->surface);
	glDeleteTextures(1, &g->texture);
	control_info_destroy(g->points);
	free(g);
}

graphic *graphic_from_file(const char *file_name) {
	graphic *g = alloc(sizeof(graphic));
	g->surface = load_image(file_name);
	if (file_name[0] == '/')
		g->points = control_info_from_file(file_name);
	else {
		char *full_file_name = full_name(file_name);
		g->points = control_info_from_file(full_file_name);
		free(full_file_name);
	}
	init(g);
	return g;
}

graphic *graphic_from_surface(SDL_Surface *surface) {
	graphic *g = alloc(sizeof(graphic));
	g->surface = surface;
	g->points = control_info_create();
	init(g);
	return g;
}

void graphic_render(const graphic *g) {
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, g->texture);
	int w = g->surface->w;
	int h = g->surface->h;
	const control_point *center = get_center(g);
	glBegin(GL_QUADS);
	glTexCoord2i(0, 0);
	glVertex2i(-center->x, -center->y);
	glTexCoord2i(w, 0);
	glVertex2i(w - center->x, -center->y);
	glTexCoord2i(w, h);
	glVertex2i(w - center->x, h - center->y);
	glTexCoord2i(0, h);
	glVertex2i(-center->x, h - center->y);
	glEnd();
}

static void init(graphic *g) {
	SDL_Surface *surface = g->surface;

	if (!get_center(g)) {
		control_point *center = alloc(sizeof(control_point));
		center->x = surface->w / 2;
		center->y = surface->h / 2;
		control_info_set_point(g->points, 0, center);
	}

	glGenTextures(1, &g->texture);
	if (surface->format->BytesPerPixel < 3) {
		SDL_Surface *rgba = new_map(surface->w, surface->h);
		SDL_BlitSurface(surface, NULL, rgba, NULL);
		set_texture(g, rgba);
		SDL_FreeSurface(rgba);
	}
	else
		set_texture(g, surface);
}

void set_center(graphic *g, const control_point *new_center) {
	control_point *center = control_info_get_point(g->points, 0);
	center->x = new_center->x;
	center->y = new_center->y;
}

static void set_texture(graphic *g, const SDL_Surface *surface) {
	GLenum pixel_format;
	GLint colors = surface->format->BytesPerPixel;
	if (colors == 4)
		pixel_format = BYTE_ORDER == BIG_ENDIAN ? GL_ABGR_EXT : GL_RGBA;
	else
		pixel_format = BYTE_ORDER == BIG_ENDIAN ? GL_BGR : GL_RGB;

	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, g->texture);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER,
		GL_LINEAR);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MAG_FILTER,
		GL_LINEAR);
	glTexImage2D(GL_TEXTURE_RECTANGLE_ARB, 0, colors, surface->w,
		surface->h, 0, pixel_format, GL_UNSIGNED_BYTE, surface->pixels);

	GLenum error = glGetError();
	while (error != GL_NO_ERROR) {
		warnx("OpenGL error %d", error);
		error = glGetError();
	}
}
