// db_connection.c - Operacions amb una base de dades PostgreSQL

#define _GNU_SOURCE

#include "db_connection.h"

#include <stdarg.h>
#include <postgresql/libpq-fe.h>

#include "core.h"

static moix_db_result *alloc_db_result(PGresult *res) {
	return (moix_db_result *) moix_alloc_resource(res,
		(void (*)(void *)) PQclear);
}

static PGresult *assert_PQexec(PGconn *conn, const char *query) {
	PGresult *res = PQexec(conn, query);
	if (!res)
		moix_error();
	return res;
}

/**
 * Executa la sentència arbitrària query.
 * @param query
 */
int moix_db_exec(const char *query) {
	moix_db_result *res = moix_db_exec_nocheck(query);
	if (PQresultStatus(moix_pg_result(res)) != PGRES_COMMAND_OK)
		moix_error();
	int tuples = moix_db_cmd_tuples(res);
	moix_db_free(res);
	return tuples;
}

int moix_db_exec_fmt(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	moix_mem *query = moix_alloc_memory(str);
	int tuples = moix_db_exec(str);
	moix_free(query);
	return tuples;
}

moix_db_result *moix_db_exec_nocheck(const char *query) {
	PGresult *res = assert_PQexec(moix_get_db_connection(), query);
	moix_db_result *m_res = alloc_db_result(res);
	switch (PQresultStatus(res)) {
	case PGRES_COMMAND_OK:
	case PGRES_FATAL_ERROR:
		break;
	default:
		moix_error();
	}
	return m_res;
}

moix_db_result *moix_db_exec_nocheck_fmt(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	moix_mem *query = moix_alloc_memory(str);
	moix_db_result *res = moix_db_exec_nocheck(str);
	moix_free(query);
	return res;
}

/**
 * Fa una consulta SELECT EXISTS (query).
 * @param query
 * @return
 */
bool moix_db_exists(const char *query) {
	moix_mem *exists_query = moix_asprintf("SELECT EXISTS(%s)", query);
	PGresult *res = assert_PQexec(moix_get_db_connection(),
		moix_string(exists_query));
	moix_db_result *m_res = alloc_db_result(res);
	moix_free(exists_query);
	if (PQresultStatus(res) != PGRES_TUPLES_OK || !PQntuples(res))
		moix_error();
	bool exists = PQgetvalue(res, 0, 0)[0] == 't';
	moix_db_free(m_res);
	return exists;
}

bool moix_db_exists_fmt(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	moix_mem *query = moix_alloc_memory(str);
	bool exists = moix_db_exists(str);
	moix_free(query);
	return exists;
}

	/**
	 * Avança el cursor de la llista de resultats.
	 * Retorna true si s'apunta a un registre vàlid. 
	 * @return
	 * @throws SQLException
	 */
//	public boolean next() throws SQLException {
//		return result_cursor.next();
//	}

/**
 * Obté el valor del camp textual cname. 
 * @param cname
 * @return
 */
const char *moix_db_get(const moix_db_result *res, int ntuple,
	const char *cname) {
	const PGresult *pgres = moix_pg_result(res);
	if (ntuple < 0 || ntuple >= PQntuples(pgres))
		moix_error();
	int fnumber = PQfnumber(pgres, cname);
	if (fnumber == -1)
		moix_error();
	if (PQgetisnull(pgres, ntuple, fnumber))
		return NULL;
	return PQgetvalue(pgres, ntuple, fnumber);
}

/**
 * Obté el valor del camp booleà cname. 
 * @param cname
 * @return
 */
bool moix_db_get_bool(const moix_db_result *res, int ntuple,
	const char *cname) {
	const char *value = moix_db_get(res, ntuple, cname);
	return value && value[0] == 't';
}

/**
 * Obté el valor del camp numèric cname. 
 * @param cname
 * @return
 */
int moix_db_get_int(const moix_db_result *res, int ntuple, const char *cname) {
	const char *value = moix_db_get(res, ntuple, cname);
	return value ? moix_atoi(value) : 0;
}

int moix_db_ntuples(const struct moix_db_result *res) {
	return PQntuples(moix_pg_result(res));
}

/**
 * Realitza una consulta arbitrària query.
 * @param query
 */
moix_db_result *moix_db_query(const char *query) {
	moix_db_result *res = moix_db_query_nocheck(query);
	if (PQresultStatus(moix_pg_result(res)) != PGRES_TUPLES_OK)
		moix_error();
	return res;
}

moix_db_result *moix_db_query_fmt(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	moix_mem *query = moix_alloc_memory(str);
	moix_db_result *res = moix_db_query(str);
	moix_free(query);
	return res;
}

moix_db_result *moix_db_query_nocheck(const char *query) {
	PGresult *res = assert_PQexec(moix_get_db_connection(), query);
	moix_db_result *m_res = alloc_db_result(res);
	switch (PQresultStatus(res)) {
	case PGRES_TUPLES_OK:
	case PGRES_FATAL_ERROR:
		break;
	default:
		moix_error();
	}
	return m_res;
}

moix_db_result *moix_db_query_nocheck_fmt(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	moix_mem *query = moix_alloc_memory(str);
	moix_db_result *res = moix_db_query_nocheck(str);
	moix_free(query);
	return res;
}
