// Moixtomeu daemon

#define _GNU_SOURCE

#include "daemon.h"

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <moixproto.h>
#include <poll.h>
#include <pthread.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert_ops/libc.h>
#include <bestiola/compare.h>
#include <bestiola/runtime.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>

#include "core.h"
#include "log_message.h"

typedef struct {
	const char *name;
	const int val;
} cmd_option;

static const char *conninfo;
static pthread_cond_t no_active_requests = PTHREAD_COND_INITIALIZER;
static bool is_daemon;
static bool listening = true;
static int request_count;
static pthread_mutex_t request_count_mutex = PTHREAD_MUTEX_INITIALIZER;
static void (*select_moixervlet)(void);
static const char *stderr_file;

static void process_args(int argc, char **argv);
static int server_open(const char *socket_file);
static void sigterm_handler(int signum, siginfo_t *info, void *uc);
static void *start_routine(void *arg);

static void daemonize(const int daemon_fd[2]) {
	switch (fork()) {
	case -1:
		err(EXIT_FAILURE, "fork");
	case 0:
		assert_close(daemon_fd[0]);
		break;
	default:
		assert_close(daemon_fd[1]);
		unsigned char daemon_status;
		exit(read(daemon_fd[0], &daemon_status, 1) == 1 ? EXIT_SUCCESS
			: EXIT_FAILURE);
	}

	if (setsid() == -1)
		err(EXIT_FAILURE, "setsid");
}

static bool lock_file(const char *path) {
	int fd = open(path, O_RDWR | O_CREAT | O_NOFOLLOW, 0600);
	if (fd == -1)
		return false;

	struct flock fl;
	fl.l_type = F_WRLCK;
	fl.l_whence = SEEK_SET;
	fl.l_start = 0;
	fl.l_len = 0;

	return fcntl(fd, F_SETLK, &fl) != -1;
}

int moix_run_daemon(int argc, char **argv, const moix_daemon_options *options) {
	conninfo = options->conninfo;
	select_moixervlet = options->select_moixervlet;

	process_args(argc, argv);

	if (stderr_file && !freopen(stderr_file, "a", stderr))
		errx(EXIT_FAILURE, "Server initialization failed.");

	if (!freopen("/dev/null", "r", stdin)
		|| !freopen("/dev/null", "a", stdout))
		errx(EXIT_FAILURE, "Server initialization failed.");

	if (moix_init_failed())
		errx(EXIT_FAILURE, "Moixtomeu initialization failed.");

	if (runtime_init_failed())
		errx(EXIT_FAILURE, "Moixtomeu initialization failed.");

	char *socket_file = assert_aprintf("%s/socket", runtime_run_dir());

	if (moix_set_conn_info(conninfo))
		errx(EXIT_FAILURE, "Moixtomeu initialization failed.");

	int daemon_fd[2];
	if (is_daemon) {
		if (pipe(daemon_fd))
			err(EXIT_FAILURE, "pipe");
		daemonize(daemon_fd);
	}

	char *lock_filename;
	if (asprintf(&lock_filename, "%s.lock", socket_file) == -1
		|| !lock_file(lock_filename))
		errx(EXIT_FAILURE, "Could not acquire lock.");

	int server = server_open(socket_file);
	if (server == -1)
		errx(EXIT_FAILURE, "Server initialization failed.");

	struct sigaction sa;
	sa.sa_handler = SIG_IGN;
	sa.sa_flags = SA_SIGINFO;
	if (sigemptyset(&sa.sa_mask))
		errx(EXIT_FAILURE, "Server initialization failed.");
	if (sigaction(SIGPIPE, &sa, NULL))
		errx(EXIT_FAILURE, "Server initialization failed.");

	sa.sa_sigaction = sigterm_handler;
	if (sigaction(SIGTERM, &sa, NULL))
		errx(EXIT_FAILURE, "Server initialization failed.");

	sigset_t oldmask;
	sigset_t sigterm_mask;
	if (sigemptyset(&sigterm_mask) || sigaddset(&sigterm_mask, SIGTERM)
		|| pthread_sigmask(SIG_BLOCK, &sigterm_mask, &oldmask))
		errx(EXIT_FAILURE, "Server initialization failed.");

	if (is_daemon) {
		unsigned char daemon_status = EXIT_SUCCESS;
		if (write(daemon_fd[1], &daemon_status, 1) != 1)
			err(EXIT_FAILURE, "write");
		assert_close(daemon_fd[1]);
	}

	pthread_attr_t attr;
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	while (listening) {
		struct pollfd pollin;
		pollin.fd = server;
		pollin.events = POLLIN;

		if (ppoll(&pollin, 1, NULL, &oldmask) == -1) {
			if (errno == EINTR)
				continue;
			errx(EXIT_FAILURE, "Unexpected poll error.");
		}
		if (!(pollin.revents & POLLIN))
			errx(EXIT_FAILURE, "Unexpected poll error.");

		int sd = accept(server, NULL, NULL);
		if (sd == -1)
			errx(EXIT_FAILURE, "Unexpected socket error.");
		FILE *stream = assert_fdopen(sd, "r+");

		pthread_mutex_lock(&request_count_mutex);
		request_count++;
		pthread_mutex_unlock(&request_count_mutex);

		pthread_t thread;
		if (pthread_create(&thread, &attr, start_routine, stream))
			errx(EXIT_FAILURE, "Could not create request thread.");
	}

	pthread_mutex_lock(&request_count_mutex);
	if (request_count)
		pthread_cond_wait(&no_active_requests, &request_count_mutex);
	pthread_mutex_unlock(&request_count_mutex);

	pthread_attr_destroy(&attr);

	assert_close(server);
	remove(socket_file);
	free(socket_file);
	remove(lock_filename);
	free(lock_filename);

	return EXIT_SUCCESS;
}

static void process_args(int argc, char **argv) {
	enum {
		CONNINFO,
		DAEMON,
		STDERR
	};
	static const cmd_option options[] = {
		{"--conninfo", CONNINFO},
		{"--daemon", DAEMON},
		{"--stderr", STDERR}
	};

	int i;
	for (i = 1; i < argc; i++) {
		cmd_option option;
		option.name = argv[i];

		cmd_option *found = bsearch(&option, options,
			sizeof(options) / sizeof(options[0]),
			sizeof(options[0]), compar_strcmp);
		if (!found)
			break;
		switch (found->val) {
		case CONNINFO:
			if (++i < argc)
				conninfo = argv[i];
			break;
		case DAEMON:
			is_daemon = true;
			break;
		case STDERR:
			if (++i < argc)
				stderr_file = argv[i];
			break;
		}
	}
}

static int server_open(const char *socket_file) {
	struct sockaddr_un sa;
	sa.sun_family = AF_LOCAL;

	if (strlen(socket_file) >= sizeof(sa.sun_path))
		return -1;
	strcpy(sa.sun_path, socket_file);
	int s = socket(PF_LOCAL, SOCK_STREAM | SOCK_NONBLOCK, 0);
	if (s == -1)
		return -1;
	remove(socket_file);
	mode_t oldu = umask(0);
	int ret = bind(s, (struct sockaddr *) &sa, sizeof(sa));
	umask(oldu);
	const int MAXBACKLOG = 5;
	if (ret == -1 || listen(s, MAXBACKLOG) == -1) {
		assert_close(s);
		return -1;
	}
	return s;
}

static void sigterm_handler(__attribute__((unused)) int signum,
	__attribute__((unused)) siginfo_t *info,
	__attribute__((unused)) void *uc) {
	listening = false;
}

static void start_moixervlet(void) {
	sigjmp_buf env;
	if (sigsetjmp(env, 1))
		return;
	moix_setjmp_env(env);

	if (conninfo)
		moix_get_session_key(0);
	select_moixervlet();
}

static void *start_routine(void *arg) {
	FILE *stream = arg;

	char *uri;
	if (!moix_get_string(stream, &uri))
		goto close_stream;

	char result;
	if (moix_create_request(stream, uri)) {
		start_moixervlet();
		result = moix_free_request();
	}
	else
		result = MOIX_ERROR;
	free(uri);
	if (result != MOIX_SEND_ERROR)
		fputc(result, stream);

	if (fflush(stream))
		log_message("WARNING: Stream was not shut down properly");
close_stream:
	assert_fclose(stream);

	pthread_mutex_lock(&request_count_mutex);
	if (!--request_count)
		pthread_cond_signal(&no_active_requests);
	pthread_mutex_unlock(&request_count_mutex);

	return NULL;
}
