// Support functions for moixervlets

#define _GNU_SOURCE

#include "core.h"

#include <ctype.h>
#include <errno.h>
#include <moixproto.h>
#include <pthread.h>
#include <search.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <bestiola/compare.h>
#include <bestiola/utf8.h>
#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <postgresql/libpq-fe.h>

#include "db_connection.h"
#include "http_codes.h"
#include "int_ops.h"
#include "log_message.h"
#include "random_hexchar.h"
#include "xstring.h"

struct moix_parameter {
	const char *name;
	unsigned order;
	const char *value;
};

typedef struct {
	char result;
	bool checked_certificate;
	bool checked_content_length;
	bool checked_query_string;
	bool checked_request_session_id;
	bool checked_user;
	bool user_authenticated;
	char is_secure;
	FILE *stream;
	const char *uri;
	struct moix_resource *last_resource;
	const char *context_path;
	const char *context_uri;
	const char *host;
	const char *method;
	struct moix_parameter *params;
	const char *password;
	const char *query_string;
	const unsigned char *remote_address;
	const char *user;
	const char *document_root;
	void *attribute_tree;
	gnutls_x509_crt_t certificate;
	PGconn *db_connection;
	int authenticated_session;
	int content_length;
	int num_params;
	int session_key;
	int user_id;
	void *setjmp_env;
} moix_request;

typedef struct {
	const void *key;
	moix_mem *value;
} moix_request_attr;

struct moix_resource {
	void *value;
	struct moix_resource *prev;
	struct moix_resource *next;
	void (*free)(void *value);
};

static char *db_conninfo;
static bool init_failed;
static __thread moix_request *request;

static void free_resources(void);
static moix_mem *get_pair_value(const char *pairs, const char *name,
	bool insensitive, char pair_sep);
static void moix_flush(void);
static moix_mem *moix_get_request_body(void);
static char *moix_gets(void);
static void moix_putc(int c);
static void moix_puts(const char *s);
static void moix_set_cookie(const char *name, const char *value);
static void parse_body(const char *body);
static int touch_session_from_request(const char *session_id);
static int touch_session_from_scratch(const char *session_id);
static int touch_session_from_user(moix_mem **session_id);
static moix_mem *unescaped_url_string(const char *escaped, bool form);
static void utf8_check_valid(const char *s);

__attribute__((constructor)) static void module_init(void) {
	if (gnutls_global_init())
		moix_set_init_failed();
}

__attribute__((destructor)) static void module_destroy(void) {
	free(db_conninfo);
	gnutls_global_deinit();
}

static int compare_attributes(const void *p1, const void *p2) {
	const moix_request_attr *attr1 = p1;
	const moix_request_attr *attr2 = p2;
	if (attr1->key < attr2->key)
		return -1;
	if (attr1->key > attr2->key)
		return 1;
	return 0;
}

static int compare_parameters(const void *p1, const void *p2) {
	const moix_parameter *par1 = p1;
	const moix_parameter *par2 = p2;
	int result = strcmp(par1->name, par2->name);
	if (result)
		return result;
	return intcmp(par1->order, par2->order);
}

static void free_attribute_node(void *datap) {
	free(datap);
}

static void free_resources(void) {
	moix_resource *resource = request->last_resource;
	while (resource) {
		resource->free(resource->value);
		moix_resource *prev = resource->prev;
		free(resource);
		resource = prev;
	}
}

static void get_authorization_data(void) {
	if (request->password)
		return;
	moix_mem *header = moix_get_header("Authorization");
	if (!header)
		return;

	const char *header_str = moix_string(header);
	static const char BASIC[] = "Basic ";
	static const size_t BASIC_LEN = sizeof(BASIC) - 1;
	if (strncasecmp(header_str, BASIC, BASIC_LEN))
		moix_send_error(HTTP_BAD_REQUEST);
	header_str += BASIC_LEN;

	moix_mem *data = moix_xstring_decode_base64(header_str);
	if (!data)
		moix_send_error(HTTP_BAD_REQUEST);
	moix_free(header);
	const char *data_str = moix_string(data);
	const char *delimiter = strchr(data_str, ':');
	if (!delimiter)
		moix_send_error(HTTP_BAD_REQUEST);

	if (strncmp(data_str, " :", 2)) {
		if (!request->user) {
			const char *user = moix_string(moix_slice(data_str,
				delimiter));
			utf8_check_valid(user);
			request->user = user;
		}
		const char *password = moix_string(moix_strdup(delimiter + 1));
		utf8_check_valid(password);
		request->password = password;
	}
	moix_free(data);
}

static moix_mem *get_cookie_value(const char *cookie, const char *name) {
//Cookie: NAME1=OPAQUE_STRING1; NAME2=OPAQUE_STRING2 ...
	return get_pair_value(cookie, name, true, ';');
}

static moix_mem *get_item(int type) {
	moix_putc(type);
	moix_flush();
	return moix_alloc_memory(moix_gets());
}

static moix_mem *get_pair_value(const char *pairs, const char *name,
	bool insensitive, char pair_sep) {
// should the session cookie be the only one supported?
	if (!pairs)
		return NULL;
	const char *pname = name;
	while (*pairs) {
		bool equal = insensitive ? tolower(*pname) == tolower(*pairs)
			: *pname == *pairs;
		if (equal) {
			pname++;
			pairs++;
			if (*pname)
				continue;
			if (*pairs == '=') {
				pairs++;
				const char *value_start = pairs;
				while (*pairs && *pairs != pair_sep)
					pairs++;
				return moix_slice(value_start, pairs);
			}
		}
		while (*pairs && *pairs != pair_sep)
			pairs++;
		if (*pairs) {
			pairs++;
			pname = name;
		}
	}
	return NULL;
}

static moix_mem *get_required_item(int type) {
	moix_mem *item = get_item(type);
	if (!item)
		moix_error();
	return item;
}

static char hexvalue(char c) {
	c = tolower(c);
	return '0' <= c && c <= '9' ? c - '0' : c - 'a' + 10;
}

void *moix_alloc(size_t size) {
	if (!size)
		return NULL;
	void *mem = malloc(size);
	if (!mem)
		moix_error();
	return mem;
}

moix_mem *moix_alloc_memory(void *mem) {
	return mem ? (moix_mem *) moix_alloc_resource(mem, free) : NULL;
}

moix_resource *moix_alloc_resource(void *value, void (*cleanup)(void *)) {
	moix_resource *resource = malloc(sizeof(moix_resource));
	if (!resource) {
		cleanup(value);
		moix_error();
	}
	resource->value = value;
	moix_resource *last_resource = request->last_resource;
	resource->prev = last_resource;
	if (last_resource)
		last_resource->next = resource;
	request->last_resource = resource;
	resource->next = NULL;
	resource->free = cleanup;
	return resource;
}

void moix_append_string(moix_mem *buffer, const char *s, size_t *length) {
	size_t b_length = *length;
	size_t s_length = strlen(s);
	char *buffer_ptr = moix_realloc(buffer, b_length + s_length + 1);
	memcpy(buffer_ptr + b_length, s, s_length + 1);
	*length = b_length + s_length;
}

moix_mem *moix_asprintf(const char *fmt, ...) {
	char *str;
	va_list ap;
	va_start(ap, fmt);
	int ret = vasprintf(&str, fmt, ap);
	va_end(ap);
	if (ret == -1)
		moix_error();
	return moix_alloc_memory(str);
}

int moix_atoi(const char *nptr) {
	int value;
	if (moix_strtoi(nptr, &value))
		moix_error();
	return value;
}

bool moix_authenticated_session(void) {
	int session_key = moix_get_session_key(0);
	if (!session_key)
		return false;

	int authenticated_session = request->authenticated_session;
	if (authenticated_session != -1)
		return authenticated_session;

	authenticated_session = moix_db_exists_fmt(
		"SELECT 1 FROM moixtomeu.session"
		" WHERE id = %d AND \"user\" IS NOT NULL", session_key);
	request->authenticated_session = authenticated_session;
	return authenticated_session;
}

void *moix_calloc(size_t nmemb, size_t size) {
	if (!nmemb || !size)
		return NULL;
	void *mem = calloc(nmemb, size);
	if (!mem)
		moix_error();
	return mem;
}

gnutls_x509_crt_t moix_client_cert(void) {
	if (request->checked_certificate)
		return request->certificate;

	moix_putc(MOIX_CERTIFICATE);
	moix_flush();
	char *pem_str = moix_gets();
	if (!pem_str) {
		request->checked_certificate = true;
		return NULL;
	}
	moix_mem *pem = moix_alloc_memory(pem_str);
	gnutls_datum_t data;
	data.data = (unsigned char *) pem_str;
	data.size = strlen(pem_str);
	gnutls_x509_crt_t cert;
	if (gnutls_x509_crt_init(&cert))
		moix_error();
	request->certificate = cert;
	if (gnutls_x509_crt_import(cert, &data, GNUTLS_X509_FMT_PEM))
		moix_error();
	moix_free(pem);
	request->checked_certificate = true;
	return cert;
}

const char *moix_context_path(void) {
	if (request->context_path)
		return request->context_path;

	moix_mem *item = get_item(MOIX_CONTEXT_PATH);
	const char *context_path = item? moix_string(item): "";
	request->context_path = context_path;
	return context_path;
}

const char *moix_context_uri(void) {
	if (request->context_uri)
		return request->context_uri;

	const char *context_uri = moix_uri() + strlen(moix_context_path());
	request->context_uri = context_uri;
	return context_uri;
}

bool moix_create_request(FILE *stream, const char *uri) {
	request = malloc(sizeof(moix_request));
	if (!request)
		return false;
	request->result = MOIX_END;
	request->checked_certificate = false;
	request->checked_content_length = false;
	request->checked_query_string = false;
	request->checked_request_session_id = false;
	request->checked_user = false;
	request->user_authenticated = false;
	request->is_secure = -1;
	request->stream = stream;
	request->uri = uri;
	request->last_resource = NULL;
	request->context_path = NULL;
	request->host = NULL;
	request->method = NULL;
	request->password = NULL;
	request->query_string = NULL;
	request->remote_address = NULL;
	request->user = NULL;
	request->document_root = NULL;
	request->attribute_tree = NULL;
	request->certificate = NULL;
	request->db_connection = NULL;
	request->authenticated_session = -1;
	request->num_params = -1;
	request->session_key = 0;
	request->user_id = 0;
	request->setjmp_env = NULL;
	return true;
}

int moix_db_cmd_tuples(const moix_db_result *result) {
	const moix_resource *resource = (const moix_resource *) result;
	const char *tuples_str = PQcmdTuples(resource->value);
	return *tuples_str ? moix_atoi(tuples_str) : 0;
}

void moix_decline(void) {
	request->result = MOIX_DECLINED;
	siglongjmp(request->setjmp_env, 1);
}

bool moix_do_get(void) {
	const char *method = moix_method();
	return !strcmp(method, "GET") || !strcmp(method, "HEAD");
}

bool moix_do_post(void) {
	return !strcmp(moix_method(), "POST");
}

void moix_error(void) {
	request->result = MOIX_ERROR;
	log_message("WARNING: Application exception");
	siglongjmp(request->setjmp_env, 1);
}

static void moix_flush(void) {
	if (fflush(request->stream))
		moix_error();
}

void moix_free_error(void) {
	request->result = MOIX_ERROR;
	log_message("WARNING: Application exception");
	if (request->setjmp_env)
		siglongjmp(request->setjmp_env, 1);
}

char moix_free_request(void) {
	request->setjmp_env = NULL;
	free_resources();
	tdestroy(request->attribute_tree, free_attribute_node);
	gnutls_x509_crt_deinit(request->certificate);
	PQfinish(request->db_connection);
	char result = request->result;
	free(request);
	return result;
}

void moix_free_resource(moix_resource *resource) {
	if (!resource)
		return;
	resource->free(resource->value);
	moix_resource *prev = resource->prev;
	moix_resource *next = resource->next;
	if (prev)
		prev->next = next;
	if (next)
		next->prev = prev;
	free(resource);
	if (request->last_resource == resource)
		request->last_resource = prev;
}

int moix_get_content_length(void) {
	if (request->checked_content_length)
		return request->content_length;

	moix_mem *header = moix_get_header("Content-Length");
	if (!header) {
		request->content_length = -1;
		request->checked_content_length = true;
		return -1;
	}

	int length = moix_atoi(moix_string(header));
	moix_free(header);
	request->content_length = length;
	request->checked_content_length = true;
	return length;
}

/**
 * Obre una connexió amb la base PostgreSQL dbname a la màquina host
 * identificant-se amb user i password.
 */
PGconn *moix_get_db_connection(void) {
	PGconn *conn = request->db_connection;
	if (conn)
		return conn;
	request->db_connection = conn = PQconnectdb(db_conninfo);
	if (PQstatus(conn) != CONNECTION_OK)
		moix_error();
	return conn;
}

const moix_parameter *moix_get_first_parameter(const char *name) {
	switch (request->num_params) {
	case -1:
	{
		moix_mem *body = moix_get_request_body();
		if (!body) {
			request->num_params = 0;
			return NULL;
		}
		parse_body(moix_string(body));
		moix_free(body);
		if (request->num_params)
			break;
	}
	case 0:
		return NULL;
	}

	moix_parameter key;
	key.name = name;
	key.order = 0;

	return bsearch(&key, request->params, request->num_params,
		sizeof(moix_parameter), compare_parameters);
}

moix_mem *moix_get_header(const char *name) {
	moix_putc(MOIX_HEADER);
	moix_puts(name);
	moix_flush();
	return moix_alloc_memory(moix_gets());
}

const moix_parameter *moix_get_next_parameter(const moix_parameter *param) {
	const moix_parameter *p = param + 1;
	int next_index = p - request->params;
	return next_index < request->num_params && p->order ? p : NULL;
}

const char *moix_get_parameter(const char *name) {
	return moix_get_parameter_value(moix_get_first_parameter(name));
}

const char *moix_get_parameter_value(const moix_parameter *param) {
	return param ? param->value : NULL;
}

unsigned moix_get_parameters_length(const moix_parameter *param) {
	unsigned length = param->order + 1;
	const moix_parameter *p = param + 1;
	unsigned next_index = p - request->params;
	unsigned remaining = request->num_params - next_index;
	unsigned i;
	for (i = 0; i < remaining && p->order; i++, p++, length++) ;
	return length;
}

moix_mem *moix_get_real_path(const char *uri) {
	if (!request->document_root) {
		moix_mem *item = get_required_item(MOIX_DOCUMENT_ROOT);
		const char *document_root = moix_string(item);
		size_t length = strlen(document_root);
		int lastpos = length - 1;
		if (document_root[lastpos] == '/') {
			char *new_str = moix_realloc(item, length);
			new_str[lastpos] = '\0';
			document_root = new_str;
		}
		request->document_root = document_root;
	}

	if (uri[0] == '/')
		uri++;
	return moix_asprintf("%s/%s", request->document_root, uri);
}

const moix_mem *moix_get_request_attr(const void *key) {
	moix_request_attr attr;
	attr.key = key;

	moix_request_attr **result = tfind(&attr, &request->attribute_tree,
		compare_attributes);
	return result ? (*result)->value : NULL;
}

static moix_mem *moix_get_request_body(void) {
	int length = moix_get_content_length();
	if (length == -1)
		return NULL;
	moix_putc(MOIX_REQUEST_BODY);
	moix_flush();

	char *body_str = moix_alloc(length + 1);
	moix_mem *body = moix_alloc_memory(body_str);
	body_str[length] = '\0';
	char *boffset = body_str;
	size_t remaining = length;
	while (remaining) {
		char *chunk = moix_gets();
		size_t chunklen = strlen(chunk);
		if (chunklen > remaining) {
			free(chunk);
			moix_error();
		}
		memcpy(boffset, chunk, chunklen);
		free(chunk);
		boffset += chunklen;
		remaining -= chunklen;
	}
	return body;
}

moix_mem *moix_get_session_attr(const char *key) {
	int session_key = moix_get_session_key(0);
	if (!session_key)
		return NULL;
	moix_mem *db_key = moix_xstring_to_db(key);
	moix_db_result *res = moix_db_query_fmt(
		"SELECT value FROM moixtomeu.session_attr WHERE session = %d"
		" AND name = %s", session_key, moix_string(db_key));
	moix_free(db_key);
	if (!PQntuples(moix_pg_result(res))) {
		moix_db_free(res);
		return NULL;
	}
	const char *value_str = moix_db_get(res, 0, "value");
	moix_mem *value = value_str ? moix_strdup(value_str) : NULL;
	moix_db_free(res);
	return value;
}

int moix_get_session_key(bool create) {
	int session_key = request->session_key;
	if (session_key)
		return session_key;

	if (!request->checked_request_session_id) {
		request->checked_request_session_id = true;

		moix_db_exec(
			"DELETE FROM moixtomeu.session"
			" WHERE CURRENT_TIMESTAMP - last_access > '00:15:00'"
			" AND id NOT IN"
			" (SELECT session FROM moixtomeu.user_session)");

		moix_mem *header = moix_get_header("Cookie");
		moix_mem *session_id = NULL;
		if (header) {
			session_id = get_cookie_value(moix_string(header),
				"SESSIONID");
			moix_free(header);
		}

		if (session_id)
			session_key = touch_session_from_request(
				moix_string(session_id));
		else {
			if (moix_user_id())
				session_key
					= touch_session_from_user(&session_id);
			moix_set_cookie("SESSIONID",
				session_id ? moix_string(session_id) : "");
		}
		moix_free(session_id);
	}

	if (!session_key) {
		if (!create)
			return 0;
		moix_mem *session_id = moix_random_hexchar(32);
		const char *session_id_str = moix_string(session_id);
		session_key = touch_session_from_scratch(session_id_str);
		moix_set_cookie("SESSIONID", session_id_str);
		moix_free(session_id);
	}

	request->session_key = session_key;
	return session_key;
}

static char *moix_gets(void) {
	char *s;
	if (!moix_get_string(request->stream, &s))
		moix_error();
	return s;
}

const char *moix_host(void) {
	const char *host = request->host;
	if (!host) {
		host = moix_string(get_required_item(MOIX_HOST));
		request->host = host;
	}
	return host;
}

bool moix_init_failed(void) {
	return init_failed;
}

void moix_invalidate_session(void) {
	int session_key = moix_get_session_key(0);
	if (!session_key)
		return;
	moix_db_exec_fmt("DELETE FROM moixtomeu.session WHERE id = %d",
		session_key);
	request->session_key = 0;
	moix_set_cookie("SESSIONID", "");
}

bool moix_is_secure(void) {
	int is_secure = request->is_secure;
	if (is_secure != -1)
		return is_secure;

	moix_putc(MOIX_SECURE);
	moix_flush();
	is_secure = fgetc(request->stream);
	if (is_secure == EOF)
		moix_error();
	if (is_secure)
		is_secure = 1;
	request->is_secure = is_secure;
	return is_secure;
}

const char *moix_method(void) {
	const char *method = request->method;
	if (!method) {
		method = moix_string(get_required_item(MOIX_METHOD));
		request->method = method;
	}
	return method;
}

static void moix_putc(int c) {
	if (fputc(c, request->stream) == EOF)
		moix_error();
}

static void moix_puts(const char *s) {
	if (!moix_put_string(request->stream, s))
		moix_error();
}

const char *moix_query_string(void) {
	if (request->checked_query_string)
		return request->query_string;

	moix_mem *escaped = get_item(MOIX_QUERY_STRING);
	const char *query_string = NULL;
	if (escaped) {
		query_string = moix_string(unescaped_url_string(moix_string(
			escaped), false));
		moix_free(escaped);
	}

	request->checked_query_string = true;
	request->query_string = query_string;
	return query_string;
}

void *moix_realloc(moix_mem *mem, size_t size) {
	moix_resource *resource = (moix_resource *) mem;
	void *new = realloc(resource->value, size);
	if (!new)
		moix_error();
	resource->value = new;
	return new;
}

void moix_redirect(const char *url) {
	moix_putc(MOIX_SEND_ERROR);
	if (!moix_put_short(request->stream, HTTP_SEE_OTHER))
		moix_error();
	moix_puts(url);
	request->result = MOIX_SEND_ERROR;
	siglongjmp(request->setjmp_env, 1);
}

const unsigned char *moix_remote_address(void) {
	if (request->remote_address)
		return request->remote_address;

	moix_putc(MOIX_REMOTE_ADDRESS);
	moix_flush();

	unsigned char *remote_address = malloc(16);
	if (!remote_address)
		moix_error();
	moix_alloc_memory(remote_address);

	if (!fread(remote_address, 16, 1, request->stream))
		moix_error();

	request->remote_address = remote_address;
	return remote_address;
}

void moix_send_content(const char *content) {
	moix_putc(MOIX_CONTENT);
	moix_puts(content);
}

void moix_send_error(short code) {
	moix_putc(MOIX_SEND_ERROR);
	if (!moix_put_short(request->stream, code))
		moix_error();
	request->result = MOIX_SEND_ERROR;
	siglongjmp(request->setjmp_env, 1);
}

void moix_set_cache_control(const char *directives) {
	moix_putc(MOIX_CACHE_CONTROL);
	moix_puts(directives);
}

bool moix_set_conn_info(const char *conninfo) {
	free(db_conninfo);
	if (conninfo) {
		db_conninfo = strdup(conninfo);
		return !db_conninfo;
	}
	db_conninfo = NULL;
	return false;
}

void moix_set_content_language(const char *language) {
	moix_putc(MOIX_CONTENT_LANGUAGE);
	moix_puts(language);
}

void moix_set_content_type(const char *type) {
	moix_putc(MOIX_CONTENT_TYPE);
	moix_puts(type);
}

static void moix_set_cookie(const char *name, const char *value) {
	moix_putc(MOIX_COOKIE);
	moix_puts(name);
	moix_puts(value);
}

void moix_set_init_failed(void) {
	init_failed = true;
}

void moix_set_request_attr(const void *key, moix_mem *value) {
	moix_request_attr *attr = moix_alloc(sizeof(moix_request_attr));
	attr->key = key;
	attr->value = value;

	if (!value) {
		moix_request_attr **result = tfind(attr,
			&request->attribute_tree, compare_attributes);
		if (result) {
			moix_free((*result)->value);
			tdelete(attr, &request->attribute_tree,
				compare_attributes);
		}
		free(attr);
		return;
	}

	moix_request_attr **result = tsearch(attr, &request->attribute_tree,
		compare_attributes);
	if (!result) {
		free(attr);
		moix_error();
	}
	moix_request_attr *found = *result;
	if (found != attr) {
		moix_free(found->value);
		found->value = value;
		free(attr);
	}
}

void moix_set_session_attr(const char *key, const char *value) {
	int session_key = moix_get_session_key(1);
	moix_mem *db_key = moix_xstring_to_db(key);
	moix_mem *delete_query = moix_asprintf(
		"DELETE FROM moixtomeu.session_attr WHERE session = %d"
		" AND name = %s", session_key, moix_string(db_key));
	if (value) {
		moix_mem *db_value = moix_xstring_to_db(value);
		moix_db_exec_fmt(
			"%s; "
			"INSERT INTO moixtomeu.session_attr"
			" (session, name, value) SELECT %d, %s, %s"
			" WHERE (%d, %s) NOT IN"
			" (SELECT session, name FROM moixtomeu.session_attr)",
			moix_string(delete_query), session_key,
			moix_string(db_key), moix_string(db_value), session_key,
			moix_string(db_key));
		moix_free(db_value);
	}
	else
		moix_db_exec(moix_string(delete_query));
	moix_free(db_key);
	moix_free(delete_query);
}

void moix_setjmp_env(sigjmp_buf env) {
	request->setjmp_env = env;
}

moix_mem *moix_slice(const char *start, const char *end) {
	size_t length = end - start;
	char *slice = moix_alloc(length + 1);
	char *end_slice = mempcpy(slice, start, length);
	*end_slice = '\0';
	return moix_alloc_memory(slice);
}

moix_mem *moix_strdup(const char *s) {
	char *str = strdup(s);
	if (!str)
		moix_error();
	return moix_alloc_memory(str);
}

const char *moix_uri(void) {
	return request->uri;
}

bool moix_uri_starts_path(const char *uri, const char *path) {
	int pathlen = strlen(path);
	if (strncmp(uri, path, pathlen))
		return false;
	switch (uri[pathlen]) {
	case '\0':
	case '/':
		return true;
	}
	return false;
}

const char *moix_user(bool authentic) {
	const char *user = request->user;
	if (!authentic || request->user_authenticated) {
		if (user)
			return user;
		if (!authentic && request->checked_user)
			return NULL;
	}

	moix_putc(MOIX_USER);
	moix_putc(authentic);
	moix_flush();
	char *received = moix_gets();
	if (user)
		free(received);
	else {
		moix_alloc_memory(received);
		user = received;
	}

	if (received) {
		request->user = user;
		request->user_authenticated = true;
		return user;
	}
	if (authentic) {
		request->result = MOIX_SEND_ERROR;
		siglongjmp(request->setjmp_env, 1);
	}

	get_authorization_data();
	user = request->user;
	if (user)
		moix_user(1);
	else
		request->checked_user = true;
	return user;
}

int moix_user_id(void) {
	int user_id = request->user_id;
	if (user_id)
		return user_id;
	const char *user = moix_user(0);
	if (!user)
		return 0;

	moix_mem *db_user = moix_xstring_to_db(user);
	get_authorization_data();
	const char *password = request->password;
	moix_mem *db_password = password ? moix_xstring_to_db(password) : NULL;
	moix_db_result *res = moix_db_query_fmt("SELECT get_user_id(%s, %s)",
		moix_string(db_user),
		db_password ? moix_string(db_password) : "NULL");
	moix_free(db_user);
	moix_free(db_password);
	user_id = moix_db_get_int(res, 0, "get_user_id");
	moix_db_free(res);

	if (!user_id)
		moix_send_error(HTTP_BAD_REQUEST);
	request->user_id = user_id;
	return user_id;
}

static void parse_body(const char *body) {
	int size_params = 1;
	int num_params = 0;
	int order = 0;
	moix_parameter *params_ptr = moix_alloc(sizeof(moix_parameter));
	moix_mem *params = moix_alloc_memory(params_ptr);
	const char *name_start = body;
	while (*body) {
		if (*body != '=') {
			body++;
			continue;
		}
		if (num_params == size_params) {
			size_params <<= 1;
			params_ptr = moix_realloc(params,
				size_params * sizeof(moix_parameter));
		}
		moix_parameter *param = params_ptr + num_params++;
		moix_mem *name = moix_slice(name_start, body);
		param->name = moix_string(unescaped_url_string(moix_string(
			name), true));
		moix_free(name);
		param->order = order++;
		const char *value_start = ++body;
		while (*body && *body != '&')
			body++;
		moix_mem *value = moix_slice(value_start, body);
		param->value = moix_string(unescaped_url_string(moix_string(
			value), true));
		moix_free(value);
		if (*body)
			name_start = ++body;
	}

	if (!num_params) {
		moix_free(params);
		request->num_params = 0;
		return;
	}

	params_ptr = moix_realloc(params, num_params * sizeof(moix_parameter));
	qsort(params_ptr, num_params, sizeof(moix_parameter),
		compare_parameters);
	int i;
	const char *last_name = NULL;
	for (i = 0; i < num_params; i++) {
		moix_parameter *param = params_ptr + i;
		if (!last_name || strcmp(param->name, last_name)) {
			last_name = param->name;
			param->order = order = 0;
		}
		else
			param->order = ++order;
	}
	request->params = params_ptr;
	request->num_params = num_params;
}

static int touch_session_from_request(const char *session_id) {
	int user_id = moix_user_id();
	moix_db_result *res = moix_db_query_fmt(
		"UPDATE moixtomeu.session SET last_access = CURRENT_TIMESTAMP"
		" WHERE session_id = '%s'"
		" AND (\"user\" IS NULL OR \"user\" = %d) RETURNING id",
		session_id, user_id);
	if (!PQntuples(moix_pg_result(res))) {
		moix_db_free(res);
		return 0;
	}
	int id = moix_db_get_int(res, 0, "id");
	moix_db_free(res);
	return id;
}

static int touch_session_from_scratch(const char *session_id) {
	int user_id = moix_user_id();
	moix_mem *user_id_str = user_id ? moix_asprintf("%d", user_id) : NULL;
	moix_db_result *res = moix_db_query_fmt(
		"INSERT INTO moixtomeu.session (session_id, \"user\")"
		" VALUES ('%s', %s) RETURNING id", session_id,
		user_id_str ? moix_string(user_id_str) : "NULL");
	moix_free(user_id_str);

	if (!PQntuples(moix_pg_result(res))) {
		moix_db_free(res);
		return 0;
	}
	int id = moix_db_get_int(res, 0, "id");
	moix_db_free(res);

	if (user_id)
		moix_db_exec_fmt(
			"DELETE FROM moixtomeu.user_session"
			" WHERE \"user\" = %d; "
			"INSERT INTO moixtomeu.user_session (\"user\", session)"
			" SELECT %d, %d WHERE %d NOT IN"
			" (SELECT \"user\" FROM moixtomeu.user_session)",
			user_id, user_id, id, user_id);

	return id;
}

static int touch_session_from_user(moix_mem **session_id) {
	int user_id = moix_user_id();
	moix_db_result *res = moix_db_query_fmt(
		"UPDATE moixtomeu.session SET last_access = CURRENT_TIMESTAMP"
		" WHERE (\"user\", id)"
		" = (SELECT \"user\", session"
		" FROM moixtomeu.user_session WHERE \"user\" = %d)"
		" RETURNING id, session_id", user_id);
	if (!PQntuples(moix_pg_result(res))) {
		*session_id = NULL;
		moix_db_free(res);
		return 0;
	}
	int id = moix_db_get_int(res, 0, "id");
	const char *str = moix_db_get(res, 0, "session_id");
	*session_id = str ? moix_strdup(str) : NULL;
	moix_db_free(res);
	return id;
}

static moix_mem *unescaped_url_string(const char *escaped, bool form) {
	if (!escaped)
		return NULL;
	size_t escaped_size = strlen(escaped) + 1;
	char *unescaped_str = moix_alloc(escaped_size);
	moix_mem *unescaped = moix_alloc_memory(unescaped_str);
	const char *ec = escaped;
	char *uc = unescaped_str;
	while (*ec) {
		if (ec[0] == '%' && isxdigit(ec[1]) && isxdigit(ec[2])) {
			ec++;
			char value = hexvalue(*ec++);
			*uc++ = (value << 4) | hexvalue(*ec++);
		}
		else {
			char c = *ec++;
			*uc++ = form && c == '+' ? ' ' : c;
		}
	}
	*uc++ = '\0';
	moix_realloc(unescaped, uc - unescaped_str);
	return unescaped;
}

static void utf8_check_valid(const char *s) {
	if (!utf8_valid(s))
		moix_send_error(HTTP_BAD_REQUEST);
}
