// XML functions

#include "xml.h"

#include <libxml/parser.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include "core.h"

static inline xmlDoc *to_xmlDoc(moix_xml_document *document);

__attribute__((constructor)) static void unit_constructor(void) {
	xmlInitParser();
	xsltInit();
}

__attribute__((destructor)) static void unit_destructor(void) {
	xsltCleanupGlobals();
}

moix_node *moix_xml_get_root(moix_xml_document *document) {
	return xmlDocGetRootElement(to_xmlDoc(document));
}

moix_node *moix_xml_new_child(moix_node *parent, const char *name) {
	return moix_xml_new_text_child(parent, name, NULL);
}

moix_xml_document *moix_xml_new_document(void) {
	xmlDoc *document = xmlNewDoc(xmlCast("1.0"));
	if (!document)
		moix_error();
	return (moix_xml_document *) moix_alloc_resource(document,
		(void (*)(void *)) xmlFreeDoc);
}

moix_node *moix_xml_new_int_child(moix_node *parent, const char *name,
	int content) {
	moix_mem *string = moix_asprintf("%d", content);
	moix_node *node = moix_xml_new_text_child(parent, name,
		moix_string(string));
	moix_free(string);
	return node;
}

moix_node *moix_xml_new_root(moix_xml_document *document, const char *name) {
	moix_node *root = xmlNewNode(NULL, xmlCast(name));
	if (!root)
		moix_error();
	xmlDocSetRootElement(to_xmlDoc(document), root);
	return root;
}

moix_node *moix_xml_new_text_child(moix_node *parent, const char *name,
	const char *content) {
	moix_node *node = xmlNewChild(parent, NULL, xmlCast(name),
		xmlCast(content));
	if (!node)
		moix_error();
	return node;
}

void moix_xml_set_attr(moix_node *node, const char *name, const char *value) {
	if (value) {
		if (!*value)
			value = NULL;
		if (!xmlSetNsProp(node, NULL, xmlCast(name), xmlCast(value)))
			moix_error();
	}
	else
		xmlUnsetProp(node, xmlCast(name));
}

void moix_xml_set_int_attr(moix_node *node, const char *name, int value) {
	moix_mem *string = moix_asprintf("%d", value);
	moix_xml_set_attr(node, name, moix_string(string));
	moix_free(string);
}

void moix_xml_set_text(moix_node *node, const char *content) {
	xmlNodeSetContent(node, xmlCast(content));
}

moix_mem *moix_xsl_transform(const char *xslt_file,
	moix_xml_document *document) {
	xsltStylesheet *style = xsltParseStylesheetFile(xmlCast(xslt_file));
	static const char *params[] = { NULL };
	xmlDoc *result = xsltApplyStylesheet(style, to_xmlDoc(document),
		params);
	xmlChar *doc_txt;
	_Bool ok = 0;
	if (result) {
		int doc_txt_len;
		ok = !xsltSaveResultToString(&doc_txt, &doc_txt_len, result,
			style);
		xmlFreeDoc(result);
	}
	xsltFreeStylesheet(style);
	if (!ok)
		moix_error();
	return moix_alloc_memory(doc_txt);
}

static inline xmlDoc *to_xmlDoc(moix_xml_document *document) {
	return *((xmlDoc **) document);
}
