// Fast pseudo random byte generator

#include "random.h"

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>

#include "core.h"

#if RAND_MAX != 0x7fffffff
#error RAND_MAX is not 0x7fffffff
#endif

static unsigned bit_buffer;
static unsigned bit_buffer_length;

static void fill_bit_buffer(void);
static void shift_bit_buffer(unsigned n);

__attribute__((constructor)) static void urandom_init(void) {
	static FILE *urandom;
	urandom = fopen("/dev/urandom", "r");
	if (!urandom) {
		moix_set_init_failed();
		return;
	}
	unsigned seed;
	if (!fread(&seed, sizeof(unsigned), 1, urandom))
		moix_set_init_failed();
	fclose(urandom);
	static char state[256];
	if (!initstate(seed, state, sizeof(state)))
		moix_set_init_failed();
	fill_bit_buffer();
}

static void fill_bit_buffer(void) {
	bit_buffer = random();
	bit_buffer_length = 31;
}

void random_buffer(char *buffer, unsigned length) {
	static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_mutex_lock(&mutex);
	while (length) {
		if (bit_buffer_length >= 8) {
			*buffer++ = bit_buffer;
			length--;
			shift_bit_buffer(8);
			continue;
		}
		char remainder = bit_buffer;
		unsigned remainder_length = bit_buffer_length;
		fill_bit_buffer();
		*buffer++ = remainder | (bit_buffer << remainder_length);
		length--;
		shift_bit_buffer(8 - remainder_length);
	}
	pthread_mutex_unlock(&mutex);
}

static void shift_bit_buffer(unsigned n) {
	bit_buffer >>= n;
	bit_buffer_length -= n;
}
