// Select the preferred supported language

#define _ISOC99_SOURCE

#include "language.h"

#include <stdlib.h>
#include <string.h>
#include <bestiola/compare.h>

#include "core.h"
#include "http_codes.h"

static const char LAST;

typedef struct {
	const char *language;
	moix_mem *match;
	float quality;
} preference;

static int compare_quality(const void *p1, const void *p2);

static moix_mem *acceptable_language(const char *header,
	const char * const *supported, size_t nmemb) {
	preference *prefs = moix_alloc(nmemb * sizeof(preference));
	moix_mem *prefs_mem = moix_alloc_memory(prefs);
	size_t i;
	for (i = 0; i < nmemb; i++) {
		preference *p = prefs + i;
		p->language = supported[i];
		p->match = NULL;
		p->quality = 0;
	}

	for (;;) {
		moix_mem *language_token_mem = NULL;
		const char *language_token;
		const char *comma = strchr(header, ',');
		if (comma) {
			language_token_mem = moix_slice(header, comma);
			language_token = moix_string(language_token_mem);
			header = comma + 1;
		}
		else
			language_token = header;

		moix_mem *language_range_mem = NULL;
		const char *language_range;
		float quality = 1;
		const char *semicolon = strchr(language_token, ';');
		if (semicolon) {
			language_range_mem = moix_slice(language_token,
				semicolon);
			language_range = moix_string(language_range_mem);
			const char *equal = strchr(semicolon + 1, '=');
			if (!equal)
				moix_send_error(HTTP_BAD_REQUEST);
			char *endptr;
			quality = strtof(equal + 1, &endptr);
			if (*endptr || quality < 0 || quality > 1)
				moix_send_error(HTTP_BAD_REQUEST);
		}
		else
			language_range = language_token;

		preference language_key;
		language_key.language = language_range;

		preference *found = bsearch(&language_key, prefs, nmemb,
			sizeof(preference), compar_strcmp);
		if (found) {
			moix_mem *match = found->match;
			if (!match || strlen(moix_string(match))
				< strlen(language_range)) {
				moix_free(match);
				found->match = moix_strdup(language_range);
				found->quality = quality;
			}
		}
		else if (!strcmp(language_range, "*")) {
			size_t i;
			for (i = 0; i < nmemb; i++) {
				preference *p = prefs + i;
				if (p->match)
					continue;
				p->match = moix_strdup("");
				p->quality = quality;
			}
		}

		moix_free(language_token_mem);
		moix_free(language_range_mem);
		if (!comma)
			break;
	}

	qsort(prefs, nmemb, sizeof(preference), compare_quality);
	moix_mem *acceptable = prefs->quality ? moix_strdup(prefs->language)
		: NULL;
	moix_free(prefs_mem);
	return acceptable;
}

static int compare_quality(const void *p1, const void *p2) {
	const preference *pref1 = p1;
	const preference *pref2 = p2;
	float q1 = pref1->quality;
	float q2 = pref2->quality;
	return q1 > q2 ? -1 : q1 < q2 ? 1 : 0;
}

const char *moix_language(const char * const *supported, size_t nmemb) {
	const moix_mem *last_preferred = moix_get_request_attr(&LAST);
	if (last_preferred)
		return moix_string(last_preferred);

	moix_mem *preferred = NULL;
	moix_mem *header = moix_get_header("Accept-Language");
	if (header) {
		preferred = acceptable_language(moix_string(header), supported,
			nmemb);
		moix_free(header);
	}

	if (!preferred)
		preferred = moix_strdup("");

	moix_set_request_attr(&LAST, preferred);
	return moix_string(preferred);
}
