// Exim 4 plug-in that handles S/MIME messages

#define _GNU_SOURCE

#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <exim4/local_scan.h>
#include <exim4-signat/common.h>
#include <exim4-signat/communication.h>
#include <sys/wait.h>

static bool is_space_delimiter(char c);
static char *process_output(int fd, char *result);
static bool scan_headers(char **micalg, char **boundary);
static inline uschar *us(const char *s);

static void add_spam_flag(void) {
	header_line *header;
	for (header = header_list; header; header = header->next) {
		static const char HEADER_NAME[] = "X-Spam-Flag";
		if (header_testname(header, us(HEADER_NAME),
			strlen(HEADER_NAME), TRUE))
			return;
	}
	header_add(' ', "X-Spam-Flag: YES\n");
}

static bool create_smtp_reply(FILE *in, FILE *out, char result) {
	int code;
	switch (result) {
	case RESULT_ACCEPT:
	case RESULT_FLAG:
		code = 250;
		break;
	case RESULT_TEMPREJECT:
		code = 451;
		break;
	case RESULT_FAKE_REJECT:
	case RESULT_REJECT:
		code = 550;
		break;
	default:
		return true;
	}

	int last = '\n';
	for (;;) {
		int c = getc(in);
		if (c == EOF) {
			if (ferror(in))
				return true;
			break;
		}
		if (c == '\0')
			break;
		if (last == '\n') {
			if (fprintf(out, "%d-", code) < 0)
				return true;
			last = '-';
		}
		if (c == '\n') {
			if (last != '\r' && putc('\r', out) == EOF)
				return true;
		}
		else {
			if (last == '\r' && fprintf(out, "\n%d-", code) < 0)
				return true;
		}
		if (putc(c, out) == EOF)
			return true;
		last = c;
	}

	switch (last) {
	default:
		if (putc('\r', out) == EOF)
			return true;
	case '\r':
		if (putc('\n', out) == EOF)
			return true;
	case '\n':
		return false;
	}
}

static bool is_boundary_char(char c) {
	return ('\'' <= c && c <= ')') || ('+' <= c && c <= ':') || c == '='
		|| c == '?' || ('A' <= c && c <= 'Z') || c == '_'
		|| ('a' <= c && c <= 'z');
}

static bool is_micalg_char(char c) {
	return c == '-' || ('0' <= c && c <= '9') || ('A' <= c && c <= 'Z')
		|| ('a' <= c && c <= 'z');
}

static bool is_parameter_end(char c, bool quoted) {
	return quoted? c == '"': !c || c == ';' || is_space_delimiter(c);
}

static bool is_space_delimiter(char c) {
	switch (c) {
	case '\t':
	case '\n':
	case '\r':
	case ' ':
		return true;
	}
	return false;
}

int local_scan(int fd, __attribute__((unused)) uschar **return_text) {
	if (!smtp_input)
		return LOCAL_SCAN_ACCEPT;

	int count = recipients_count;

	int local_recipients = 0;
	for (int i = 0; i < count; i++) {
		recipient_item *recipient = recipients_list + i;
		uschar *address = recipient->address;

		uid_t uid = lss_local_uid(address);
		if (uid == (uid_t) -1)
			continue;
		local_recipients++;
	}

	bool scanned_headers = false;
	char *micalg = NULL;
	char *boundary = NULL;

	bool error = false;
	bool fake_reject = false;
	bool mark_as_spam = false;
	bool reject = false;
	bool tempreject = false;
	char *proxy_msg = NULL;

	for (int i = 0; i < count; i++) {
		recipient_item *recipient = recipients_list + i;
		uschar *address = recipient->address;

		uid_t uid = lss_local_uid(address);
		if (uid == (uid_t) -1)
			continue;
		char uid_string[9];
		if (sprintf(uid_string, "%x", uid) < 0) {
			error = true;
			break;
		}

		if (!scanned_headers) {
			if (scan_headers(&micalg, &boundary)) {
				error = true;
				break;
			}
			scanned_headers = true;
		}

		int pipefd[2];
		if (pipe(pipefd)) {
			error = true;
			break;
		}

		pid_t pid = fork();
		if (pid == -1) {
			close(pipefd[0]);
			close(pipefd[1]);
			error = true;
			break;
		}
		if (!pid) {
			if (close(pipefd[0])
				|| dup2(pipefd[1], STDOUT_FILENO) == -1
				|| close(pipefd[1])
				|| dup2(fd, CONTENT_FD) == -1)
				exit(EXIT_FAILURE);
			static const char PATH[]
				= "/usr/lib/exim4-signat/proxy";
			execl(PATH, PATH, uid_string, address, sender_address,
				micalg, boundary, NULL);
			exit(EXIT_FAILURE);
		}

		if (close(pipefd[1]))
			error = true;

		char result;
		char *msg = process_output(pipefd[0], &result);
		if (!msg)
			error = true;
		else if (local_recipients == 1)
			proxy_msg = msg;
		else
			free(msg);

		siginfo_t info;
		if (waitid(P_PID, pid, &info, WEXITED)) {
			if (errno != ECHILD)
				error = true;
		}
		else if (info.si_code != CLD_EXITED)
			error = true;

		if (error)
			break;

		switch (result) {
		case RESULT_FLAG:
			mark_as_spam = true;
		case RESULT_ACCEPT:
			continue;
		case RESULT_FAKE_REJECT:
			fake_reject = true;
			continue;
		case RESULT_REJECT:
			reject = true;
			break;
		case RESULT_TEMPREJECT:
			tempreject = true;
			break;
		default:
			error = true;
		}
		break;
	}
	free(micalg);
	free(boundary);
	if (error) {
		free(proxy_msg);
		return LOCAL_SCAN_TEMPREJECT;
	}

	if (proxy_msg) {
		if (!smtp_batched_input && proxy_msg[0])
			smtp_printf("%s", proxy_msg);
		free(proxy_msg);
	}

	if (tempreject)
		return LOCAL_SCAN_TEMPREJECT;

	static const char REJECT_MESSAGE[] = "\
550-The message does not meet the trust level of one recipient at least\r\n\
550-See http://www.jasp.net/smtp/trust.xhtml\r\n";
	if (reject) {
		if (!smtp_batched_input && local_recipients != 1)
			smtp_printf(REJECT_MESSAGE);
		return LOCAL_SCAN_REJECT;
	}
	if (fake_reject) {
		if (!smtp_batched_input && local_recipients != 1)
			smtp_printf(REJECT_MESSAGE);
		add_spam_flag();
		return LOCAL_SCAN_FAKE_REJECT;
	}
	if (mark_as_spam)
		add_spam_flag();
	return LOCAL_SCAN_ACCEPT;
}

int local_scan_version_major(void) {
	return LOCAL_SCAN_ABI_VERSION_MAJOR;
}

int local_scan_version_minor(void) {
	return LOCAL_SCAN_ABI_VERSION_MINOR;
}

static char *process_output(int fd, char *result) {
	FILE *in = fdopen(fd, "r");
	if (!in) {
		close(fd);
		return NULL;
	}

	int r = getc(in);
	if (r == EOF) {
		fclose(in);
		return NULL;
	}

	char *ptr;
	size_t sizeloc;
	FILE *out = open_memstream(&ptr, &sizeloc);
	if (!out) {
		fclose(in);
		return NULL;
	}

	if (create_smtp_reply(in, out, r)) {
		fclose(out);
		free(ptr);
		fclose(in);
		return NULL;
	}

	if (fclose(out)) {
		free(ptr);
		fclose(in);
		return NULL;
	}

	if (fclose(in)) {
		free(ptr);
		return NULL;
	}

	*result = r;
	return ptr;
}

static bool scan_headers(char **micalg, char **boundary) {
	header_line *header;
	for (header = header_list; header; header = header->next) {
		static const char HEADER_NAME[] = "Content-Type";
		if (header_testname(header, (uschar *) HEADER_NAME,
			strlen(HEADER_NAME), TRUE))
			break;
	}
	if (!header)
		return false;

	char *c = strchr((char *) header->text, ':');
	c++;
	while (is_space_delimiter(*c))
		c++;
	static const char MIME_TYPE[] = "multipart/signed";
	static const size_t MIME_TYPE_LENGTH = sizeof MIME_TYPE - 1;
	if (strncasecmp(c, MIME_TYPE, MIME_TYPE_LENGTH))
		return false;
	c += MIME_TYPE_LENGTH;

	bool error = false;
	char *temp_micalg = NULL;
	char *temp_boundary = NULL;
	bool has_protocol = false;
	while (!(temp_boundary && temp_micalg && has_protocol)) {
		while (is_space_delimiter(*c))
			c++;
		if (*c != ';')
			goto failure;
		c++;
		while (is_space_delimiter(*c))
			c++;
		switch (to_c_lower(*c)) {
		case 'b':
		{
			if (temp_boundary)
				goto failure;

			static const char PARAM[] = "boundary=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			bool quoted = *c == '"';
			if (quoted)
				c++;
			char *start = c;
			while (is_boundary_char(*c))
				c++;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			size_t length = c - start;
			temp_boundary = malloc(length + 1);
			if (!temp_boundary) {
				error = true;
				goto failure;
			}
			char *end = mempcpy(temp_boundary, start, length);
			*end = '\0';
			if (quoted)
				c++;
			break;
		}
		case 'm':
		{
			if (temp_micalg)
				goto failure;

			static const char PARAM[] = "micalg=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			bool quoted = *c == '"';
			if (quoted)
				c++;
			char *start = c;
			while (is_micalg_char(*c))
				c++;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			size_t length = c - start;
			temp_micalg = malloc(length + 1);
			if (!temp_micalg) {
				error = true;
				goto failure;
			}
			char *end = mempcpy(temp_micalg, start, length);
			*end = '\0';
			if (quoted)
				c++;
			break;
		}
		case 'p':
		{
			if (has_protocol)
				goto failure;

			static const char PARAM[] = "protocol=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			bool quoted = *c == '"';
			if (quoted)
				c++;
			static const char PROTOCOL[]
				= "application/pkcs7-signature";
			static const size_t PROTOCOL_LENGTH = sizeof PROTOCOL
				- 1;
			if (strncasecmp(c, PROTOCOL, PROTOCOL_LENGTH))
				goto failure;
			c += PROTOCOL_LENGTH;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			if (quoted)
				c++;
			has_protocol = true;
			break;
		}
		default:
			goto failure;
		}
	}
	*micalg = temp_micalg;
	*boundary = temp_boundary;
	return error;
failure:
	free(temp_micalg);
	free(temp_boundary);
	return error;
}

static inline uschar *us(const char *s) {
	return (uschar *) s;
}
