// Check against user preferences

#define _GNU_SOURCE

#include <pwd.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <exim4-signat/checks.h>
#include <exim4-signat/communication.h>

static gid_t get_gid(uid_t uid);

int main(int argc, char **argv) {
	if (argc < 4)
		return EXIT_FAILURE;

	char *endptr;
	uid_t uid = strtoul(argv[1], &endptr, 16);
	if (*endptr)
		return EXIT_FAILURE;
	switch (uid) {
	case -1:
	case 0:
		return EXIT_FAILURE;
	}

	gid_t gid = get_gid(uid);
	if (gid == (gid_t) -1)
		return EXIT_FAILURE;

	if (setresgid(-1, gid, gid))
		return EXIT_FAILURE;
	if (setresuid(-1, uid, uid))
		return EXIT_FAILURE;

	char *dir = signat_conf_dir();
	if (!dir)
		return EXIT_FAILURE;

	if (euidaccess(dir, F_OK)) {
		putchar(RESULT_ACCEPT);
		free(dir);
		return EXIT_SUCCESS;
	}

	char *proxy;
	if (asprintf(&proxy, "%s/proxy", dir) == -1)
		return EXIT_FAILURE;
	free(dir);

	if (!euidaccess(proxy, F_OK)) {
		if (setresgid(gid, gid, gid))
			return EXIT_FAILURE;
		if (setresuid(uid, uid, uid))
			return EXIT_FAILURE;

		if (argc < 6)
			execl(proxy, proxy, argv[2], argv[3], NULL);
		else
			execl(proxy, proxy, argv[2], argv[3], argv[4], argv[5],
				NULL);
		return EXIT_FAILURE;
	}
	free(proxy);

	const char *sender = argv[3];
	if (!signat_whitelist_check(argv[2], sender)) {
		putchar(RESULT_ACCEPT);
		return EXIT_SUCCESS;
	}

	if (argc >= 6
		&& !signat_smime_check(CONTENT_FD, sender, argv[4], argv[5])) {
		putchar(RESULT_ACCEPT);
		return EXIT_SUCCESS;
	}

	char code = signat_reject_code();
	putchar(code);
	switch (code) {
	case RESULT_FAKE_REJECT:
	case RESULT_REJECT: {
		static const char MESSAGE[] = "\
The message does not meet the trust level of this recipient\n\
See http://www.jasp.net/smtp/trust.xhtml";
		fputs(MESSAGE, stdout);
	}
	}

	return EXIT_SUCCESS;
}

static gid_t get_gid(uid_t uid) {
	size_t buflen = sysconf(_SC_GETPW_R_SIZE_MAX);
	if (buflen == (size_t) -1)
		return -1;

	char *buf = malloc(buflen);
	if (!buf)
		return -1;

	struct passwd pwbuf;
	struct passwd *pwbufp;
	getpwuid_r(uid, &pwbuf, buf, buflen, &pwbufp);
	if (!pwbufp)  {
		free(buf);
		return -1;
	}

	gid_t gid = pwbuf.pw_gid;
	free(buf);
	return gid;
}
