// Check an e-mail using S/MIME

#include "smime_check.h"

#include <bestiola/base64.h>
#include <nspr/nspr.h>
#include <nss/cert.h>
#include <nss/cms.h>
#include <nss/nss.h>
#include <nss/secerr.h>

#include "common.h"

static int skip_header(FILE *stream);
static _Bool update_context(FILE *stream, const char *mm_boundary,
	NSSCMSDigestContext *cmsdigcx);
static _Bool update_decoder(FILE *stream, NSSCMSDecoderContext *p7dcx);

static _Bool check_main_value(FILE *stream, const char *value) {
	int c = getc(stream);
	while (c == '\t' || c == ' ')
		c = getc(stream);
	while (c != EOF) {
		if (*value) {
			char lower_c = to_c_lower(c);
			if (lower_c != *value++)
				return 0;
		}
		else {
			switch (c) {
			case '\n':
				if (ungetc(c, stream) == EOF)
					exit(EXIT_FAILURE);
			case '\t':
			case '\r':
			case ' ':
			case ';':
				return 1;
			}
			return 0;
		}
		c = getc(stream);
	}
	return 0;
}

static void check_memory_error(void) {
	if (PORT_GetError() == SEC_ERROR_NO_MEMORY)
		exit(EXIT_FAILURE);
}

static _Bool decode_signature(FILE *stream, const char *sender,
	SECOidTag digestalgtag, SECItem *digest) {
	NSSCMSDecoderContext *p7dcx = NSS_CMSDecoder_Start(NULL, NULL, NULL,
		NULL, NULL, NULL, NULL);
	if (!p7dcx)
		exit(EXIT_FAILURE);
	if (update_decoder(stream, p7dcx)) {
		NSS_CMSDecoder_Cancel(p7dcx);
		return 1;
	}
	NSSCMSMessage *cmsg = NSS_CMSDecoder_Finish(p7dcx);
	if (!cmsg) {
		check_memory_error();
		return 1;
	}

	NSSCMSContentInfo *cinfo;
	int count = NSS_CMSMessage_ContentLevelCount(cmsg);
	int n;
	for (n = 0; n < count; n++) {
		cinfo = NSS_CMSMessage_ContentLevel(cmsg, n);
		if (NSS_CMSContentInfo_GetContentTypeTag(cinfo)
			== SEC_OID_PKCS7_SIGNED_DATA)
			break;
	}
	if (!(n < count)) {
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	NSSCMSSignedData *sigd = NSS_CMSContentInfo_GetContent(cinfo);
	if (!sigd || NSS_CMSSignedData_HasDigests(sigd)) {
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	CERTCertDBHandle *certdb = CERT_GetDefaultCertDB();
	if (NSS_CMSSignedData_SetDigestValue(sigd, digestalgtag, digest)
		!= SECSuccess
		|| NSS_CMSSignedData_ImportCerts(sigd, certdb,
		certUsageEmailSigner, PR_FALSE) != SECSuccess) {
		check_memory_error();
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	if (NSS_CMSSignedData_SignerInfoCount(sigd) < 1) {
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	NSSCMSSignerInfo *sinfo = NSS_CMSSignedData_GetSignerInfo(sigd, 0);
	CERTCertificate *cert = NSS_CMSSignerInfo_GetSigningCertificate(sinfo,
		NULL);
	if (!cert) {
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}
	char *emailAddr = cert->emailAddr;
	if (!emailAddr || strcmp(sender, emailAddr)) {
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	if (NSS_CMSSignedData_VerifySignerInfo(sigd, 0, certdb,
		certUsageEmailSigner) != SECSuccess) {
		check_memory_error();
		NSS_CMSMessage_Destroy(cmsg);
		return 1;
	}

	NSS_CMSMessage_Destroy(cmsg);
	return 0;
}

static _Bool digest_text(FILE *stream, const char *mm_boundary, SECOidTag tag,
	PLArenaPool *poolp, SECItem *digest) {
	PLArenaPool *arena = PORT_NewArena(1024);
	if (!arena)
		exit(EXIT_FAILURE);
	SECAlgorithmID digestalg;
	if (SECOID_SetAlgorithmID(arena, &digestalg, tag, NULL)
		!= SECSuccess)
		exit(EXIT_FAILURE);
	NSSCMSDigestContext *cmsdigcx
		= NSS_CMSDigestContext_StartSingle(&digestalg);
	if (!cmsdigcx)
		exit(EXIT_FAILURE);

	if (update_context(stream, mm_boundary, cmsdigcx)) {
		NSS_CMSDigestContext_Cancel(cmsdigcx);
		PORT_FreeArena(arena, PR_FALSE);
		return 1;
	}

	if (NSS_CMSDigestContext_FinishSingle(cmsdigcx, poolp, digest)
		!= SECSuccess)
		exit(EXIT_FAILURE);
	PORT_FreeArena(arena, PR_FALSE);
	return 0;
}

static SECOidTag get_micalg_tag(const char *micalg) {
	static struct {
		const char *micalg;
		SECOidTag tag;
	} table[] = {
		{ "sha-256", SEC_OID_SHA256 },
		{ "sha1", SEC_OID_SHA1 }
	};

	unsigned i;
	for (i = 0; i < sizeof table / sizeof table[0]; i++) {
		if (!strcmp(micalg, table[i].micalg))
			return table[i].tag;
	}
	return SEC_OID_UNKNOWN;
}

static _Bool scan_headers(FILE *stream) {
	_Bool good_encoding = 0;
	_Bool good_type = 0;
	int c = getc(stream);
	for (;;) {
		switch (c) {
		case '\r':
			c = getc(stream);
			if (c != '\n')
				return 1;
		case '\n':
			return !(good_encoding && good_type);
		case EOF:
			return 1;
		}
		int i = 0;
		_Bool maybe_encoding = 1;
		_Bool maybe_type = 1;
		for (;;) {
			char lower_c = to_c_lower(c);
			if (maybe_encoding) {
				static const char ENCODING[]
					= "content-transfer-encoding";
				if (ENCODING[i]) {
					if (lower_c != ENCODING[i])
						maybe_encoding = 0;
				}
				else if (c == ':') {
					if (good_encoding
						|| !check_main_value(stream,
						"base64"))
						return 1;
					good_encoding = 1;
					break;
				}
				else
					maybe_encoding = 0;
			}
			if (maybe_type) {
				static const char TYPE[] = "content-type";
				if (TYPE[i]) {
					if (lower_c != TYPE[i])
						maybe_type = 0;
				}
				else if (c == ':') {
					if (good_type
						|| !check_main_value(stream,
						"application/pkcs7-signature"))
						return 1;
					good_type = 1;
					break;
				}
				else
					maybe_type = 0;
			}
			if (!(maybe_encoding || maybe_type))
				break;
			i++;
			c = getc(stream);
			if (c == EOF)
				return 1;
		}
		c = skip_header(stream);
	}
	return 1;
}

static int skip_header(FILE *stream) {
	int c = getc(stream);
	for (;;) {
		switch (c) {
		case '\n':
			c = getc(stream);
			if (c == '\t' || c == ' ')
				break;
		case EOF:
			return c;
		}
		c = getc(stream);
	}
}

_Bool smime_check(const char *sender, FILE *stream, const char *micalg,
	const char *mm_boundary, const char *configdir) {
	SECOidTag micalg_tag = get_micalg_tag(micalg);
	if (micalg_tag == SEC_OID_UNKNOWN)
		return 0;

	PR_Init(0, 0, 0);
	if (NSS_Init(configdir) != SECSuccess)
		exit(EXIT_FAILURE);
	PLArenaPool *poolp = PORT_NewArena(1024);
	if (!poolp)
		exit(EXIT_FAILURE);

	SECItem digest;
	_Bool error = digest_text(stream, mm_boundary, micalg_tag, poolp,
		&digest) || scan_headers(stream)
		|| decode_signature(stream, sender, micalg_tag, &digest);

	PORT_FreeArena(poolp, PR_FALSE);
	if (NSS_Shutdown() != SECSuccess)
		exit(EXIT_FAILURE);
	if (PR_Cleanup() != PR_SUCCESS)
		exit(EXIT_FAILURE);
	return !error;
}

static _Bool update_context(FILE *stream, const char *mm_boundary,
	NSSCMSDigestContext *cmsdigcx) {
	if (__builtin_choose_expr(BUFSIZ >= 80, 0, (void) 0)) {}
	unsigned char *data = malloc(BUFSIZ);
	if (!data)
		exit(EXIT_FAILURE);
	int len = 0;

	int state = 0;
	int i = 0;
	int c = getc(stream);
	while (c != EOF) {
		switch (state) {
		case 0:
			switch (c) {
			case '\n':
				state = 2;
				break;
			case '\r':
				state++;
				break;
			default:
				if (len + 1 > BUFSIZ) {
					NSS_CMSDigestContext_Update(cmsdigcx,
						data, len);
					len = 0;
				}
				data[len++] = c;
			}
			break;
		case 1:
			if (c == '\n')
				state++;
			else {
				state = 0;
				if (len + 2 > BUFSIZ) {
					NSS_CMSDigestContext_Update(cmsdigcx,
						data, len);
					len = 0;
				}
				data[len++] = '\r';
				data[len++] = '\n';
				continue;
			}
			break;
		case 2:
			if (c == mm_boundary[i]) {
				if (!mm_boundary[++i])
					state++;
			}
			else {
				state = 0;
				if (len + 2 + i > BUFSIZ) {
					NSS_CMSDigestContext_Update(cmsdigcx,
						data, len);
					len = 0;
				}
				data[len++] = '\r';
				data[len++] = '\n';
				if (i) {
					memcpy(data + len, mm_boundary, i);
					len += i;
					i = 0;
				}
				continue;
			}
			break;
		case 3:
			if (c == '\r') {
				state++;
				break;
			}
		case 4:
		{
			_Bool error = c != '\n';
			if (!error)
				NSS_CMSDigestContext_Update(cmsdigcx, data,
					len);
			free(data);
			return error;
		}
		}
		c = getc(stream);
	}
	free(data);
	return 1;
}

static _Bool update_decoder(FILE *stream, NSSCMSDecoderContext *p7dcx) {
	char *buf = malloc(BUFSIZ);
	if (!buf)
		exit(EXIT_FAILURE);
	unsigned long len = 0;

	_Bool last = 0;
	int i;
	while (!last) {
		char b[4];
		i = 0;
		for (;;) {
			int c = getc(stream);
			if (c == EOF) {
				free(buf);
				return 1;
			}
			if (is_base64(c)) {
				b[i] = c;
				if (i++ == 3)
					break;
			}
			else if (c == '-' || c == '=') {
				if (i == 1) {
					free(buf);
					return 1;
				}
				last = 1;
				int j;
				for (j = i; j < 4; j++)
					b[j] = 'A';
				break;
			}
		}

		if (len + 3 > BUFSIZ) {
			if (NSS_CMSDecoder_Update(p7dcx, buf, len)
				!= SECSuccess) {
				check_memory_error();
				free(buf);
				return 1;
			}
			len = 0;
		}

		unsigned char v[4];
		int j;
		for (j = 0; j < 4; j++)
			v[j] = base64_value(b[j]);
		buf[len++] = v[0] << 2 | v[1] >> 4;
		buf[len++] = v[1] << 4 | v[2] >> 2;
		buf[len++] = v[2] << 6 | v[3];
	}

	switch (i) {
	case 0:
		break;
	case 2:
		len -= 2;
		break;
	case 3:
		len--;
	}
	if (NSS_CMSDecoder_Update(p7dcx, buf, len) != SECSuccess) {
		check_memory_error();
		free(buf);
		return 1;
	}
	free(buf);
	return 0;
}
