// Require signed messages

#define _GNU_SOURCE

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <exim4/local_scan.h>
#include <sys/wait.h>

#include "common.h"
#include "communication.h"

static _Bool is_space_delimiter(char c);
static _Bool scan_headers(char **micalg, char **boundary);
static inline uschar *us(const char *s);

static void add_spam_flag(void) {
	header_line *header;
	for (header = header_list; header; header = header->next) {
		static const char HEADER_NAME[] = "X-Spam-Flag";
		if (header_testname(header, us(HEADER_NAME),
			strlen(HEADER_NAME), TRUE))
			return;
	}
	header_add(' ', "X-Spam-Flag: YES\n");
}

static _Bool is_boundary_char(char c) {
	return ('\'' <= c && c <= ')') || ('+' <= c && c <= ':') || c == '='
		|| c == '?' || ('A' <= c && c <= 'Z') || c == '_'
		|| ('a' <= c && c <= 'z');
}

static _Bool is_micalg_char(char c) {
	return c == '-' || ('0' <= c && c <= '9') || ('A' <= c && c <= 'Z')
		|| ('a' <= c && c <= 'z');
}

static _Bool is_parameter_end(char c, _Bool quoted) {
	return quoted ? c == '"' : !c || c == ';' || is_space_delimiter(c);
}

static _Bool is_space_delimiter(char c) {
	switch (c) {
	case '\n':
	case '\r':
	case ' ':
		return 1;
	}
	return 0;
}

int local_scan(int fd, __attribute__((unused)) uschar **return_text) {
	if (!smtp_input)
		return LOCAL_SCAN_ACCEPT;
//	if ((debug_selector & D_local_scan) != 0)
//		debug_printf("xxx", ...);

	_Bool scanned_headers = 0;
	char *micalg = NULL;
	char *boundary = NULL;

	_Bool mark_as_spam = 0;
	_Bool reject = 0;
	_Bool error = 0;
	int count = recipients_count;
	int i;
	for (i = 0; i < count; i++) {
		recipient_item *recipient = recipients_list + i;
		uschar *address = recipient->address;

		uid_t uid = lss_local_uid(address);
		if (uid == (uid_t) -1)
			continue;
		char uid_string[9];
		if (sprintf(uid_string, "%x", uid) < 0) {
			error = 1;
			break;
		}

		if (!scanned_headers) {
			if (scan_headers(&micalg, &boundary)) {
				error = 1;
				break;
			}
			scanned_headers = 1;
		}

		int pipefd[2];
		if (pipe(pipefd)) {
			error = 1;
			break;
		}

		pid_t pid = fork();
		if (pid == -1) {
			close(pipefd[0]);
			close(pipefd[1]);
			error = 1;
			break;
		}
		if (!pid) {
			if (close(pipefd[0])
				|| dup2(pipefd[1], STDOUT_FILENO) == -1
				|| close(pipefd[1])
				|| dup2(fd, CONTENT_FD) == -1)
				exit(EXIT_FAILURE);
			static const char PATH[]
				= "/usr/lib/exim4-signat/proxy";
			execl(PATH, PATH, uid_string, address, sender_address,
				micalg, boundary, NULL);
			exit(EXIT_FAILURE);
		}

		char result;
		if (close(pipefd[1]))
			error = 1;
		else if (read(pipefd[0], &result, 1) != 1)
			error = 1;
		if (close(pipefd[0]))
			error = 1;

		siginfo_t info;
		if (waitid(P_PID, pid, &info, WEXITED)) {
			if (errno != ECHILD)
				error = 1;
		}
		else if (info.si_code != CLD_EXITED)
			error = 1;

		if (error)
			break;

		switch (result) {
		case RESULT_FLAG:
			mark_as_spam = 1;
		case RESULT_ACCEPT:
			continue;
		default:
			reject = 1;
		}
		break;
	}
	free(micalg);
	free(boundary);
	if (error)
		return LOCAL_SCAN_TEMPREJECT;
	if (reject) {
		if (!smtp_batched_input)
			smtp_printf("\
550-The message does not meet the trust level of one recipient at least\r\n\
550-See http://www.jasp.net/smtp/trust.xhtml\r\n");
		return LOCAL_SCAN_REJECT;
	}
	if (mark_as_spam)
		add_spam_flag();
	return LOCAL_SCAN_ACCEPT;
}

int local_scan_version_major(void) {
	return LOCAL_SCAN_ABI_VERSION_MAJOR;
}

int local_scan_version_minor(void) {
	return LOCAL_SCAN_ABI_VERSION_MINOR;
}

static _Bool scan_headers(char **micalg, char **boundary) {
	header_line *header;
	for (header = header_list; header; header = header->next) {
		static const char HEADER_NAME[] = "Content-Type";
		if (header_testname(header, (uschar *) HEADER_NAME,
			strlen(HEADER_NAME), TRUE))
			break;
	}
	if (!header)
		return 0;

	char *c = strchr((char *) header->text, ':');
	c++;
	while (is_space_delimiter(*c))
		c++;
	static const char MIME_TYPE[] = "multipart/signed";
	static const size_t MIME_TYPE_LENGTH = sizeof MIME_TYPE - 1;
	if (strncasecmp(c, MIME_TYPE, MIME_TYPE_LENGTH))
		return 0;
	c += MIME_TYPE_LENGTH;

	_Bool error = 0;
	char *temp_micalg = NULL;
	char *temp_boundary = NULL;
	_Bool has_protocol = 0;
	while (!(temp_boundary && temp_micalg && has_protocol)) {
		while (is_space_delimiter(*c))
			c++;
		if (*c != ';')
			goto failure;
		c++;
		while (is_space_delimiter(*c))
			c++;
		switch (to_c_lower(*c)) {
		case 'b':
		{
			if (temp_boundary)
				goto failure;

			static const char PARAM[] = "boundary=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			_Bool quoted = *c == '"';
			if (quoted)
				c++;
			char *start = c;
			while (is_boundary_char(*c))
				c++;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			size_t length = c - start;
			temp_boundary = malloc(length + 1);
			if (!temp_boundary) {
				error = 1;
				goto failure;
			}
			char *end = mempcpy(temp_boundary, start, length);
			*end = '\0';
			if (quoted)
				c++;
			break;
		}
		case 'm':
		{
			if (temp_micalg)
				goto failure;

			static const char PARAM[] = "micalg=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			_Bool quoted = *c == '"';
			if (quoted)
				c++;
			char *start = c;
			while (is_micalg_char(*c))
				c++;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			size_t length = c - start;
			temp_micalg = malloc(length + 1);
			if (!temp_micalg) {
				error = 1;
				goto failure;
			}
			char *end = mempcpy(temp_micalg, start, length);
			*end = '\0';
			if (quoted)
				c++;
			break;
		}
		case 'p':
		{
			if (has_protocol)
				goto failure;

			static const char PARAM[] = "protocol=";
			static const size_t PARAM_LENGTH = sizeof PARAM - 1;
			if (strncasecmp(c, PARAM, PARAM_LENGTH))
				goto failure;
			c += PARAM_LENGTH;
			_Bool quoted = *c == '"';
			if (quoted)
				c++;
			static const char PROTOCOL[]
				= "application/pkcs7-signature";
			static const size_t PROTOCOL_LENGTH = sizeof PROTOCOL
				- 1;
			if (strncasecmp(c, PROTOCOL, PROTOCOL_LENGTH))
				goto failure;
			c += PROTOCOL_LENGTH;
			if (!is_parameter_end(*c, quoted))
				goto failure;
			if (quoted)
				c++;
			has_protocol = 1;
			break;
		}
		default:
			goto failure;
		}
	}
	*micalg = temp_micalg;
	*boundary = temp_boundary;
	return error;
failure:
	free(temp_micalg);
	free(temp_boundary);
	return error;
}

static inline uschar *us(const char *s) {
	return (uschar *) s;
}
