// Check against user preferences

#define _GNU_SOURCE

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "common.h"
#include "communication.h"
#include "smime_check.h"

static _Bool find_address(const char *address, const char *dir,
	const char *file);
static char reject_code(const char *dir);
static int skip_line(int c, FILE *stream);
static _Bool skip_prologue(FILE *stream, const char *mm_boundary);

int main(int argc, char **argv) {
	if (argc < 4)
		return EXIT_FAILURE;

	char *endptr;
	uid_t uid = strtoul(argv[1], &endptr, 16);
	if (*endptr)
		return EXIT_FAILURE;
	switch (uid) {
	case -1:
	case 0:
		return EXIT_FAILURE;
	}
	if (setresuid(-1, uid, uid))
		return EXIT_FAILURE;

	size_t buflen = sysconf(_SC_GETPW_R_SIZE_MAX);
	if (buflen == (size_t) -1)
		return EXIT_FAILURE;
	char *buf = malloc(buflen);
	if (!buf)
		return EXIT_FAILURE;
	struct passwd pwbuf;
	struct passwd *pwbufp;
	getpwuid_r(uid, &pwbuf, buf, buflen, &pwbufp);
	if (!pwbufp)
		return EXIT_FAILURE;

	char *conf_dir;
	if (asprintf(&conf_dir, "%s/." PACKAGE, pwbuf.pw_dir) == -1)
		return EXIT_FAILURE;
	free(buf);
	const char *sender = argv[3];
	if (euidaccess(conf_dir, F_OK)
		|| find_address(argv[2], conf_dir, "whitelist")
		|| find_address(sender, conf_dir, "whitelist-from")) {
		putchar(RESULT_ACCEPT);
		free(conf_dir);
		return EXIT_SUCCESS;
	}

	if (argc < 6) {
		putchar(reject_code(conf_dir));
		free(conf_dir);
		return EXIT_SUCCESS;
	}

	const char *micalg = argv[4];
	const char *boundary = argv[5];
	static const size_t MAX_BOUNDARY_SIZE = 71;
	size_t boundary_length = strnlen(boundary, MAX_BOUNDARY_SIZE);
	if (!boundary_length || boundary_length == MAX_BOUNDARY_SIZE) {
		putchar(reject_code(conf_dir));
		free(conf_dir);
		return EXIT_SUCCESS;
	}

	boundary_length++;
	char *mm_boundary = malloc(boundary_length + 2);
	if (!mm_boundary)
		return EXIT_FAILURE;
	char *boundary_p = mm_boundary;
	*boundary_p++ = '-';
	*boundary_p++ = '-';
	memcpy(boundary_p, boundary, boundary_length);

	FILE *stream = fdopen(CONTENT_FD, "r");
	if (!stream)
		return EXIT_FAILURE;

	if (skip_prologue(stream, mm_boundary)) {
		putchar(reject_code(conf_dir));
		free(conf_dir);
		free(mm_boundary);
		return EXIT_SUCCESS;
	}

	_Bool good = smime_check(sender, stream, micalg, mm_boundary, conf_dir);

	if (fclose(stream))
		return EXIT_FAILURE;

	putchar(good ? RESULT_ACCEPT : reject_code(conf_dir));
	free(conf_dir);
	free(mm_boundary);
	return EXIT_SUCCESS;
}

static _Bool find_address(const char *address, const char *dir,
	const char *file) {
	const char *at = strchr(address, '@');
	if (!at++)
		return 0;

	char *path;
	if (asprintf(&path, "%s/%s", dir, file) == -1)
		exit(EXIT_FAILURE);
	FILE *stream = fopen(path, "r");
	free(path);
	if (!stream)
		return 0;

	_Bool match = 0;
	int c = getc(stream);
	while (c == '\n')
		c = getc(stream);
	for (; c != EOF; c = skip_line(c, stream)) {
		if (c == '#')
			continue;
		const char *ac;
		if (c != '@') {
			ac = address;
			while (*ac != '@' && *ac == c) {
				ac++;
				c = getc(stream);
			}
			if (*ac != c)
				continue;
		}
		ac = at;
		c = getc(stream);
		while (*ac && c != EOF && to_c_lower(*ac) == to_c_lower(c)) {
			ac++;
			c = getc(stream);
		}
		if (!*ac && (c == '\n' || c == EOF)) {
			match = 1;
			break;
		}
	}

	if (fclose(stream))
		exit(EXIT_FAILURE);
	return match;
}

static char reject_code(const char *dir) {
	char *path;
	if (asprintf(&path, "%s/reject", dir) == -1)
		exit(EXIT_FAILURE);
	_Bool reject = !euidaccess(path, F_OK);
	free(path);
	return reject ? RESULT_REJECT : RESULT_FLAG;
}

static int skip_line(int c, FILE *stream) {
	while (c != '\n' && c != EOF)
		c = getc(stream);
	while (c == '\n')
		c = getc(stream);
	return c;
}

static _Bool skip_prologue(FILE *stream, const char *mm_boundary) {
	int state = 0;
	int i = 0;
	int c = getc(stream);
	while (c != EOF) {
		switch (state) {
		case 0:
			if (c == '\n')
				state++;
			break;
		case 1:
			if (c == mm_boundary[i]) {
				if (!mm_boundary[++i])
					state++;
			}
			else {
				state = 0;
				i = 0;
				continue;
			}
			break;
		case 2:
			if (c == '\r') {
				state++;
				break;
			}
		case 3:
			return c != '\n';
		}
		c = getc(stream);
	}
	return 1;
}
