// Advanced proxy for Exim 4 signat

#define _GNU_SOURCE

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <gcrypt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <exim4-signat/checks.h>
#include <exim4-signat/communication.h>
#include <libxml/parser.h>
#include <sys/stat.h>

static void free_advanced_conf(char **secretary);
static char *hex_encode_buffer(const char *s, size_t size);
static char hexchar(char value);
static bool load_advanced_conf(char **secretary);
static bool remove_old_files(const char *dir);

int main(int argc, char **argv) {
	if (argc < 3)
		return EXIT_FAILURE;

	const char *recipient = argv[1];
	const char *sender = argv[2];
	if (!signat_whitelist_check(recipient, sender)) {
		putchar(RESULT_ACCEPT);
		return EXIT_SUCCESS;
	}

	if (argc >= 5
		&& !signat_smime_check(CONTENT_FD, sender, argv[3], argv[4])) {
		putchar(RESULT_ACCEPT);
		return EXIT_SUCCESS;
	}

	char *secretary;
	if (load_advanced_conf(&secretary))
		return EXIT_FAILURE;

	if (secretary && !strcmp(recipient, secretary)) {
		putchar(RESULT_ACCEPT);
		free_advanced_conf(&secretary);
		return EXIT_SUCCESS;
	}

	char *conf_dir = signat_conf_dir();
	if (!conf_dir)
		return EXIT_FAILURE;

	char *tmpdir;
	if (asprintf(&tmpdir, "%s/tmp", conf_dir) == -1)
		return EXIT_FAILURE;
	free(conf_dir);

	if (mkdir(tmpdir, S_IRWXU) && errno != EEXIST)
		return EXIT_FAILURE;

	if (remove_old_files(tmpdir))
		return EXIT_FAILURE;

	char *template;
	if (asprintf(&template, "%s/tmpXXXXXX", tmpdir) == -1)
		return EXIT_FAILURE;

	int tmpmsg = mkstemp(template);
	if (tmpmsg == -1)
		return EXIT_FAILURE;

	FILE *in_stream = fdopen(CONTENT_FD, "r");
	if (!in_stream)
		return EXIT_FAILURE;
	FILE *out_stream = fdopen(tmpmsg, "w");
	if (!out_stream)
		return EXIT_FAILURE;

	if (fprintf(out_stream, "From: %s\r\nTo: %s\r\n\r\n", sender, recipient)
		< 0)
		return EXIT_FAILURE;

	for (;;) {
		int c = getc(in_stream);
		if (c == EOF) {
			if (ferror(in_stream))
				return EXIT_FAILURE;
			break;
		}
		if (putc(c, out_stream) == EOF)
			return EXIT_FAILURE;
	}

	if (fclose(out_stream))
		return EXIT_FAILURE;

	FILE *msg_stream = fopen(template, "r");
	if (!msg_stream)
		return EXIT_FAILURE;

	gcry_md_hd_t hd;
	if (gcry_md_open(&hd, GCRY_MD_SHA256, 0))
		return EXIT_FAILURE;

	for (;;) {
		int c = getc(msg_stream);
		if (c == EOF) {
			if (ferror(msg_stream))
				return EXIT_FAILURE;
			break;
		}
		gcry_md_putc(hd, c);
	}

	if (fclose(msg_stream))
		return EXIT_FAILURE;

	unsigned char *digest = gcry_md_read(hd, 0);
	if (!digest)
		return EXIT_FAILURE;

	char *hex_value = hex_encode_buffer((char *) digest, 256 / 8);
	if (!hex_value)
		return EXIT_FAILURE;

	gcry_md_close(hd);

	char *hex_path;
	if (asprintf(&hex_path, "%s/%s", tmpdir, hex_value) == -1)
		return EXIT_FAILURE;
	free(tmpdir);
	free(hex_value);

	bool accept = false;

	if (link(template, hex_path)) {
		if (errno != EEXIST)
			return EXIT_FAILURE;
		struct timespec tp;
		if (clock_gettime(CLOCK_REALTIME, &tp))
			return EXIT_FAILURE;
		struct stat buf;
		if (stat(hex_path, &buf))
			return EXIT_FAILURE;
		accept = tp.tv_sec - buf.st_mtim.tv_sec >= 900;
	}
	free(hex_path);

	if (unlink(template))
		return EXIT_FAILURE;
	free(template);

	if (accept) {
		if (secretary) {
			putchar(RESULT_REJECT);
			printf("Send your message to\n%s", secretary);
		}
		else
			putchar(RESULT_ACCEPT);
	}
	else
		putchar(RESULT_TEMPREJECT);

	free_advanced_conf(&secretary);
	return EXIT_SUCCESS;
}

static int filter_dot_dirs(const struct dirent *d) {
	const char *s = d->d_name;
	return !(s[0] == '.' && (s[1] == '\0'
		|| (s[1] == '.' && s[2] == '\0')));
}

static void free_advanced_conf(char **secretary) {
	free(*secretary);
}

static char *hex_encode_buffer(const char *s, size_t size) {
	char *value = malloc(2 * size + 1);
	if (!value)
		return NULL;

	for (size_t i = 0; i < size; i++) {
		value[2 * i] = hexchar(s[i] >> 4);
		value[2 * i + 1] = hexchar(s[i]);
	}
	value[2 * size] = '\0';

	return value;
}

// TODO: move to bestiola
// also in moixtomeu
static char hexchar(char value) {
	value &= 0x0f;
	if (value < 10)
		return '0' + value;
	return 'a' + value - 10;
}

static bool load_advanced_conf(char **secretary) {
	char *dir = signat_conf_dir();
	if (!dir)
		return true;

	char *conf;
	int ret = asprintf(&conf, "%s/advanced.xml", dir);
	free(dir);
	if (ret == -1)
		return true;

	*secretary = NULL;

	if (access(conf, F_OK)) {
		free(conf);
		return false;
	}

	xmlDoc *document = xmlParseFile(conf);
	free(conf);
	if (!document)
		return true;

	xmlNode *root = xmlDocGetRootElement(document);
	for (xmlNode *node = root->children; node; node = node->next) {
		if (xmlStrEqual(node->name, xmlCast("secretary"))) {
			if (!*secretary)
				*secretary = xmlUncast(xmlGetProp(node,
					xmlCast("value")));
		}
	}

	xmlFreeDoc(document);
	return false;
}

static bool remove_old_files(const char *dir) {
	struct timespec tp;
	if (clock_gettime(CLOCK_REALTIME, &tp))
		return true;

	int dirfd = open(dir, O_RDWR | O_DIRECTORY | O_PATH);
	if (dirfd == -1)
		return true;

	struct dirent **namelist;
	int n = scandir(dir, &namelist, filter_dot_dirs, alphasort);
	if (n == -1) {
		close(dirfd);
		return true;
	}

	bool error = false;

	for (int i = 0; i < n; i++) {
		struct stat buf;
		if (fstatat(dirfd, namelist[i]->d_name, &buf, 0)) {
			error = true;
			free(namelist[i]);
			continue;
		}
		if (tp.tv_sec - buf.st_mtim.tv_sec < 86400) {
			free(namelist[i]);
			continue;
		}
		if (unlinkat(dirfd, namelist[i]->d_name, 0))
			error = true;
		free(namelist[i]);
	}
	free(namelist);
	if (close(dirfd))
		error = true;
	return error;
}
