#define _GNU_SOURCE

#include <dlfcn.h>
#include <err.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <emu-stub/handler.h>
#include <emu-stub/handler-api.h>

#include "libpthread_const.h"
#include "libpthread_private.h"

#include "thread_data.h"

typedef struct {
	emu_cpu_env *env;
	unsigned routine_args[2];
	struct _pthread_cleanup_buffer *buffer;
} cleanup_routine_data;

struct {
	pthread_mutex_t mutex;
	emu_cpu_env *env;
	void (*init_routine)(void);
} init_routine_data = { .mutex = PTHREAD_MUTEX_INITIALIZER };

static int (*handle__libc_getspecific)(unsigned args[]);
static int (*handle__libc_key_create)(unsigned args[], emu_cpu_env *env);
static int (*handle__libc_setspecific)(unsigned args[]);
static int (*handle_lseek64)(unsigned args[]);
static int (*handle_open64)(unsigned args[]);

__attribute__((constructor)) static void init_extern_symbols(void) {
	void *handle = dlopen("$ORIGIN/libc_preload.so.6", RTLD_LAZY);
	handle__libc_getspecific = dlsym(handle, "handle__libc_getspecific");
	handle__libc_key_create = dlsym(handle, "handle__libc_key_create");
	handle__libc_setspecific = dlsym(handle, "handle__libc_setspecific");
	handle_lseek64 = dlsym(handle, "handle_lseek64");
	handle_open64 = dlsym(handle, "handle_open64");
}

static void cleanup_routine(void *arg) {
	cleanup_routine_data *data = arg;
	emu_cpu_env *env = data->env;
	emu_set_return_value(env, (unsigned) data->routine_args);
	emu_cpu_run(env);
	emu_cpu_free(env);
	free(data);
}

static void init_routine(void) {
	emu_cpu_env *env = emu_cpu_clone(init_routine_data.env);
	emu_set_return_value(env,
		(unsigned) tswap32(init_routine_data.init_routine));
	emu_cpu_run(env);
	emu_cpu_free(env);
}

struct thread_routine_data {
	thread_data *data;
	unsigned start_routine;
	unsigned arg;
};

static void *thread_routine(void *arg) {
	struct thread_routine_data *rdata = arg;
	thread_data *data = rdata->data;
	emu_set_thread_data(data);
	unsigned routine_args[2] = { rdata->start_routine, rdata->arg };
	free(rdata);
	emu_set_return_value(data->env, (unsigned) routine_args);
	emu_cpu_run(data->env);
	data->retval = (void *) routine_args[0];
	if (data->detachstate == PTHREAD_CREATE_DETACHED) {
		emu_cpu_free(data->env);
		free(data);
	}
	return data;
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case LSEEK64:
		return handle_lseek64(args);
	case OPEN64:
		return handle_open64(args);
	case PTHREAD_ATTR_DESTROY:
	{
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_attr_destroy(attr));
		return 0;
	}
	case PTHREAD_ATTR_GETSCHEDPARAM: {
		const pthread_attr_t *attr = (void *) tswap32(args[2]);
		struct sched_param *target_param = (void *) tswap32(args[3]);
		struct sched_param param;
		assign_result32(args[1], pthread_attr_getschedparam(attr,
			&param));
		target_param->sched_priority = tswap32(param.sched_priority);
		return 0;
	}
	case PTHREAD_ATTR_GETSTACK:
	{
		const pthread_attr_t *attr = (void *) tswap32(args[2]);
		void **target_stackaddr = (void *) tswap32(args[3]);
		void *stackaddr;
		size_t *target_stacksize = (void *) tswap32(args[4]);
		size_t stacksize;
		int ret = pthread_attr_getstack(attr, &stackaddr, &stacksize);
		assign_result32(args[1], ret);
		if (!ret) {
			*target_stackaddr = tswap32(emu_stackaddr(env));
			*target_stacksize = tswap32(emu_stacksize());
		}
		return 0;
	}
	case PTHREAD_ATTR_INIT:
	{
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_attr_init(attr));
		return 0;
	}
	case PTHREAD_ATTR_SETDETACHSTATE:
	{
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		int detachstate = tswap32(args[3]);
		assign_result32(args[1], pthread_attr_setdetachstate(attr,
			detachstate));
		return 0;
	}
	case PTHREAD_ATTR_SETSCHEDPARAM: {
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		const struct sched_param *target_param
			= (void *) tswap32(args[3]);
		struct sched_param param;
		param.sched_priority = tswap32(target_param->sched_priority);
		assign_result32(args[1], pthread_attr_setschedparam(attr,
			&param));
		return 0;
	}
	case PTHREAD_ATTR_SETSCOPE:
	{
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		int scope = tswap32(args[3]);
		assign_result32(args[1], pthread_attr_setscope(attr, scope));
		return 0;
	}
	case PTHREAD_ATTR_SETSTACKSIZE:
	{
		pthread_attr_t *attr = (void *) tswap32(args[2]);
		size_t stacksize = tswap32(args[3]);
		assign_result32(args[1], pthread_attr_setstacksize(attr,
			stacksize));
		size_t current_stacksize = emu_stacksize();
		if (stacksize > current_stacksize)
			warnx("pthread_attr_setstacksize %d %d", stacksize,
				current_stacksize);
		return 0;
	}
	case PTHREAD_CANCEL:
	{
warnx("cancelling thread");
		pthread_t thread = args[2];
		assign_result32(args[1], pthread_cancel(thread));
		return 0;
	}
	case _PTHREAD_CLEANUP_POP_RESTORE:
	{
		struct _pthread_cleanup_buffer *target_buffer
			= (void *) tswap32(args[1]);
		if (target_buffer->__routine != tswap32(&cleanup_routine))
			warnx("_pthread_cleanup_pop_restore:"
				" unexpected routine.");
		struct _pthread_cleanup_buffer *buffer = target_buffer->__prev;
		int execute = args[2];
		_pthread_cleanup_pop_restore(buffer, execute);
		free(buffer);
		return 0;
	}
	case _PTHREAD_CLEANUP_PUSH_DEFER:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
// TODO: check sizeof struct _pthread_cleanup_buffer
// TODO: Take care of cleanup on pop(upgrade), exit or cancel
		struct _pthread_cleanup_buffer *target_buffer
			= (void *) tswap32(args[1]);
		struct _pthread_cleanup_buffer *buffer
			= malloc(sizeof(struct _pthread_cleanup_buffer));
		target_buffer->__routine = tswap32(&cleanup_routine);
		target_buffer->__prev = buffer;
		cleanup_routine_data *data
			= malloc(sizeof(cleanup_routine_data));
		data->env = new_env;
		data->routine_args[0] = args[2];
		data->routine_args[1] = args[3];
		data->buffer = buffer;
		_pthread_cleanup_push_defer(buffer, cleanup_routine, data);
		return 0;
	}
	case PTHREAD_CONDATTR_INIT:
	{
		pthread_condattr_t *attr = (void *) tswap32(args[1]);
		pthread_condattr_init(attr);
		return 0;
	}
	case PTHREAD_CREATE:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		pthread_t *thread = (void *) tswap32(args[2]);
		const pthread_attr_t *attr = (void *) tswap32(args[3]);
		struct thread_routine_data *rdata
			= malloc(sizeof(struct thread_routine_data));
		thread_data *new_data = malloc(sizeof(thread_data));
		rdata->data = new_data;
		new_data->env = new_env;
		if (attr)
			pthread_attr_getdetachstate(attr,
				&new_data->detachstate);
		else
			new_data->detachstate = PTHREAD_CREATE_JOINABLE;
		thread_data *data = emu_get_thread_data();
		new_data->target_ss = data->target_ss;
		rdata->start_routine = args[4];
		rdata->arg = args[5];
		assign_result32(args[1], pthread_create(thread, attr,
			thread_routine, rdata));
		return 0;
	}
	case PTHREAD_EXIT:
	{
		void *retval = (void *) args[1];
		if (retval == PTHREAD_CANCELED)
			warnx("exiting canceled thread.");
		thread_data *data = emu_get_thread_data();
		data->retval = retval;
		if (data->detachstate == PTHREAD_CREATE_DETACHED) {
			emu_cpu_free(data->env);
			free(data);
		}
		pthread_exit(data);
		return -1;
	}
	case PTHREAD_GETATTR_NP:
	{
		pthread_t th = args[2];
		pthread_attr_t *attr = (void *) tswap32(args[3]);
		assign_result32(args[1], pthread_getattr_np(th, attr));
		return 0;
	}
	case PTHREAD_GETSCHEDPARAM:
	{
		pthread_t target_thread = args[2];
		int *target_policy = (void *) tswap32(args[3]);
		int policy;
		struct sched_param *target_param = (void *) tswap32(args[4]);
		struct sched_param param;
		assign_result32(args[1], pthread_getschedparam(target_thread,
			&policy, &param));
		*target_policy = tswap32(policy);
		target_param->sched_priority = tswap32(param.sched_priority);
		return 0;
	}
	case PTHREAD_GETSPECIFIC:
		return handle__libc_getspecific(args);
	case PTHREAD_JOIN:
	{
		pthread_t th = args[2];
		void **target_thread_return = (void *) tswap32(args[3]);
		void *thread_return;
		assign_result32(args[1], pthread_join(th, &thread_return));
		if (thread_return == PTHREAD_CANCELED)
			warnx("joined canceled thread.");
		else {
			thread_data *data = thread_return;
			thread_return = (void *) data->retval;
			emu_cpu_free(data->env);
			free(data);
		}
		if (target_thread_return)
			*target_thread_return = thread_return;
		return 0;
	}
	case PTHREAD_KEY_CREATE:
		return handle__libc_key_create(args, env);
	case PTHREAD_KEY_DELETE: {
		pthread_key_t key = args[2];
		assign_result32(args[1], pthread_key_delete(key));
		return 0;
	}
	case PTHREAD_MUTEX_TRYLOCK:
	{
		pthread_mutex_t *mutex = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_mutex_trylock(mutex));
		return 0;
	}
	case PTHREAD_MUTEXATTR_DESTROY:
	{
		pthread_mutexattr_t *attr = (void *) tswap32(args[1]);
		pthread_mutexattr_destroy(attr);
		return 0;
	}
	case PTHREAD_MUTEXATTR_INIT:
	{
		pthread_mutexattr_t *attr = (void *) tswap32(args[1]);
		pthread_mutexattr_init(attr);
		return 0;
	}
	case PTHREAD_MUTEXATTR_SETTYPE:
	{
		pthread_mutexattr_t *attr = (void *) tswap32(args[2]);
		int kind = tswap32(args[3]);
		assign_result32(args[1], pthread_mutexattr_settype(attr, kind));
		return 0;
	}
	case PTHREAD_ONCE:
	{
// TODO: check thread upgrade, add __libc_pthread_single_init to glibc
		pthread_once_t *once_control = (void *) tswap32(args[1]);
		if (*once_control & 2)
			return 0;
// TODO: use per once_control once_data, dynamic table
		pthread_mutex_lock(&init_routine_data.mutex);
		if (!(*once_control & 2)) {
			init_routine_data.env = env;
			init_routine_data.init_routine = (void *) args[2];
			pthread_once(once_control, init_routine);
		}
		pthread_mutex_unlock(&init_routine_data.mutex);
		return 0;
	}
	case __PTHREAD_RWLOCK_RDLOCK:
	{
		pthread_rwlock_t *rwlock = (void *) tswap32(args[2]);
		assign_result32(args[1], __pthread_rwlock_rdlock(rwlock));
		return 0;
	}
	case __PTHREAD_RWLOCK_UNLOCK:
	{
		pthread_rwlock_t *rwlock = (void *) tswap32(args[2]);
		assign_result32(args[1], __pthread_rwlock_unlock(rwlock));
		return 0;
	}
	case __PTHREAD_RWLOCK_WRLOCK:
	{
		pthread_rwlock_t *rwlock = (void *) tswap32(args[2]);
// TODO: check sizeof pthread_rwlock_t, thread upgrade
		assign_result32(args[1], __pthread_rwlock_wrlock(rwlock));
		return 0;
	}
	case PTHREAD_SETSCHEDPARAM:
	{
		pthread_t target_thread = args[2];
		int policy = tswap32(args[3]);
		const struct sched_param *target_param
			= (void *) tswap32(args[4]);
		struct sched_param param;
		param.sched_priority = tswap32(target_param->sched_priority);
		assign_result32(args[1], pthread_setschedparam(target_thread,
			policy, &param));
		return 0;
	}
	case PTHREAD_SETSPECIFIC:
		return handle__libc_setspecific(args);
	case PTHREAD_SIGMASK:
	{
		int how = tswap32(args[2]);
		const sigset_t *target_newmask = (void *) tswap32(args[3]);
		sigset_t newmask;
		memswap32(&newmask, target_newmask, size32of(sigset_t));
		sigset_t *target_oldmask = (void *) tswap32(args[4]);
		sigset_t *oldmask = target_oldmask ? alloca(sizeof(sigset_t))
			: NULL;
		assign_result32(args[1], pthread_sigmask(how, &newmask,
			oldmask));
		if (target_oldmask)
			memswap32(target_oldmask, oldmask, size32of(sigset_t));
		return 0;
	}
	case SEM_DESTROY:
	{
		sem_t *sem = (void *) tswap32(args[2]);
		assign_result32(args[1], sem_destroy(sem));
		return 0;
	}
	case SEM_INIT:
	{
		sem_t *sem = (void *) tswap32(args[2]);
		int pshared = tswap32(args[3]);
		unsigned int value = tswap32(args[4]);
		assign_result32(args[1], sem_init(sem, pshared, value));
		return 0;
	}
	case SEM_WAIT:
	{
		sem_t *sem = (void *) tswap32(args[2]);
		assign_result32(args[1], sem_wait(sem));
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
