#include <stdio.h>
#include <cairo/cairo.h>
#include <cairo/cairo-ft.h>

#include "libcairo_const.h"
#include "libfreetype.h"

static FT_FaceProxy *face_proxy;
static int lock_count;

static inline void swap_matrix(cairo_matrix_t *dest,
	const cairo_matrix_t *src) {
	dest->xx = swapd(src->xx);
	dest->yx = swapd(src->yx);
	dest->xy = swapd(src->xy);
	dest->yy = swapd(src->yy);
	dest->x0 = swapd(src->x0);
	dest->y0 = swapd(src->y0);
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case CAIRO_FONT_FACE_DESTROY:
	{
		cairo_font_face_t *font_face = (void *) args[1];
		cairo_font_face_destroy(font_face);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_COPY:
	{
		const cairo_font_options_t *original = (void *) args[2];
		cairo_font_options_t **result = (void *) tswap32(args[1]);
		*result = cairo_font_options_copy(original);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_CREATE:
	{
		cairo_font_options_t **result = (void *) tswap32(args[1]);
		*result = cairo_font_options_create();
		return 0;
	}
	case CAIRO_FONT_OPTIONS_DESTROY:
	{
		cairo_font_options_t *options = (void *) args[1];
		cairo_font_options_destroy(options);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_GET_HINT_METRICS:
	{
		const cairo_font_options_t *options = (void *) args[2];
		assign_result32(args[1],
			cairo_font_options_get_hint_metrics(options));
		return 0;
	}
	case CAIRO_FONT_OPTIONS_HASH:
	{
		const cairo_font_options_t *options = (void *) args[2];
//TODO: remove hash swap
		assign_result32(args[1], cairo_font_options_hash(options));
		return 0;
	}
	case CAIRO_FONT_OPTIONS_MERGE:
	{
		cairo_font_options_t *options = (void *) args[1];
		const cairo_font_options_t *other = (void *) args[2];
		cairo_font_options_merge(options, other);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_SET_ANTIALIAS:
	{
		cairo_font_options_t *options = (void *) args[1];
		cairo_antialias_t antialias = tswap32(args[2]);
		cairo_font_options_set_antialias(options, antialias);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_SET_HINT_STYLE:
	{
		cairo_font_options_t *options = (void *) args[1];
		cairo_hint_style_t hint_style = tswap32(args[2]);
		cairo_font_options_set_hint_style(options, hint_style);
		return 0;
	}
	case CAIRO_FONT_OPTIONS_SET_SUBPIXEL_ORDER:
	{
		cairo_font_options_t *options = (void *) args[1];
		cairo_subpixel_order_t subpixel_order = tswap32(args[2]);
		cairo_font_options_set_subpixel_order(options, subpixel_order);
		return 0;
	}
	case CAIRO_FT_FONT_FACE_CREATE_FOR_PATTERN:
	{
		FcPattern *pattern = (void *) args[2];
		cairo_font_face_t **result = (void *) tswap32(args[1]);
		*result = cairo_ft_font_face_create_for_pattern(pattern);
		return 0;
	}
	case CAIRO_FT_FONT_OPTIONS_SUBSTITUTE:
	{
		const cairo_font_options_t *options = (void *) args[1];
		FcPattern *pattern = (void *) args[2];
		cairo_ft_font_options_substitute(options, pattern);
		return 0;
	}
	case CAIRO_FT_SCALED_FONT_LOCK_FACE:
	{
		cairo_scaled_font_t *scaled_font = (void *) args[2];
		FT_Face face = cairo_ft_scaled_font_lock_face(scaled_font);
		if (!lock_count++) {
			face_proxy = malloc(sizeof(FT_FaceProxy));
			memswap32(face_proxy, face, 8);
			if (face->num_fixed_sizes)
				printf("face num_fixed_sizes %d\n",
					face->num_fixed_sizes);
			face_proxy->available_sizes = NULL;
			face_proxy->num_charmaps = tswap32(face->num_charmaps);
			FT_CharMap *charmaps = malloc(face->num_charmaps
				* sizeof(FT_CharMap));
			face_proxy->charmaps = tswap32(charmaps);
			int i;
			for (i = 0; i < face->num_charmaps; i++) {
				FT_CharMap charmap
					= malloc(sizeof(FT_CharMapRec));
				charmap->encoding
					= tswap32(face->charmaps[i]->encoding);
				charmaps[i] = tswap32(charmap);
			}
			face_proxy->native = face;
		}
		else if (face_proxy->native != face)
			printf("private face distinct\n");
		assign_result32(args[1], face_proxy);
		return 0;
	}
	case CAIRO_FT_SCALED_FONT_UNLOCK_FACE:
	{
		cairo_scaled_font_t *scaled_font = (void *) args[1];
		if (!--lock_count) {
printf("unlocking face\n");
			free(face_proxy);
		}
		cairo_ft_scaled_font_unlock_face(scaled_font);
		return 0;
	}
	case CAIRO_SCALED_FONT_CREATE:
	{
		cairo_font_face_t *font_face = (void *) args[2];
		const cairo_matrix_t *target_font_matrix
			= (void *) tswap32(args[3]);
		cairo_matrix_t font_matrix;
		swap_matrix(&font_matrix, target_font_matrix);
		const cairo_matrix_t *target_ctm = (void *) tswap32(args[4]);
		cairo_matrix_t ctm;
		swap_matrix(&ctm, target_ctm);
		const cairo_font_options_t *options = (void *) args[5];
		cairo_scaled_font_t **result = (void *) tswap32(args[1]);
		*result = cairo_scaled_font_create(font_face, &font_matrix,
			&ctm, options);
		return 0;
	}
	case CAIRO_SCALED_FONT_STATUS:
	{
		cairo_scaled_font_t *scaled_font = (void *) args[2];
		assign_result32(args[1], cairo_scaled_font_status(scaled_font));
		return 0;
	}
	case CAIRO_SURFACE_DESTROY:
	{
		cairo_surface_t *surface = (void *) args[1];
		cairo_surface_destroy(surface);
		return 0;
	}
	default:
		fprintf(stderr, "Unimplemented function #%d.\n", function);
		return -1;
	}
}
