#define _GNU_SOURCE

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <langinfo.h>
#include <libintl.h>
#include <limits.h>
#include <locale.h>
#include <pthread.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <emu-stub/handler.h>
#include <emu-stub/handler-api.h>
#include <sys/epoll.h>
#include <sys/inotify.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/wait.h>

#include "libc_preload_const.h"

#include "thread_data.h"

void *__libc_getspecific(pthread_key_t key);
int __libc_key_create(pthread_key_t *key, void (*destr)(void *));
int __libc_setspecific(pthread_key_t key, const void *value);

typedef struct specific_data {
	void *pointer;
	pthread_key_t key;
	struct specific_data *prev;
	struct specific_data *next;
} specific_data;

#if defined(TARGET_I386)
struct target_epoll_event {
	uint32_t events;
	epoll_data_t data;
} __attribute__((packed));
#elif defined(TARGET_POWERPC)
struct target_epoll_event {
	uint32_t events;
	int pad;
	epoll_data_t data;
};
#endif

static emu_cpu_env *destr_function_env;

struct {
	void *function;
	specific_data *head;
	pthread_mutex_t mutex;
} key_destructor[PTHREAD_KEYS_MAX];

static struct {
	emu_cpu_env *env;
	struct {
		void *handler;
		int onstack;
	} sig[NSIG];
} sigaction_handlers;

static struct {
	emu_cpu_env *env;
	void *handler[NSIG];
} signal_handlers;

static void setup_sigaction_handler(int signum, _Bool sa_siginfo,
	void *target_handler, _Bool onstack, emu_cpu_env *env);
static void sigaction_handler(int signum, siginfo_t *info, void *uc);
static void signal_handler(int signum);
static int target_to_host_signum(int signum);

static int add_to_buffer(char **buffer, int written, char *suffix, int suflen) {
	if (suflen < 0) {
		free(*buffer);
		*buffer = NULL;
		return suflen;
	}
	if (!written) {
		*buffer = suffix;
		return suflen;
	}
	char *new_buffer = realloc(*buffer, written + suflen + 1);
	if (!new_buffer) {
		free(suffix);
		free(*buffer);
		*buffer = NULL;
		return -1;
	}
	*buffer = new_buffer;
	strcat(*buffer, suffix);
	free(suffix);
	return suflen;
}

static int add_end(char **buffer, int written, const char *token) {
	char *suffix;
	int suflen = asprintf(&suffix, token);
	return add_to_buffer(buffer, written, suffix, suflen);
}

static int add_uarg(char **buffer, int written, const char *token,
	unsigned uarg) {
	char *suffix;
	int suflen = asprintf(&suffix, token, uarg);
	return add_to_buffer(buffer, written, suffix, suflen);
}

static void destr_function(void *pointer) {
	specific_data *data = pointer;
	pthread_key_t key = data->key;
	if (key_destructor[key].function) {
		pthread_mutex_lock(&key_destructor[key].mutex);
		specific_data *prev = data->prev;
		specific_data *next = data->next;
		if (prev)
			prev->next = next;
		else
			key_destructor[key].head = next;
		if (next)
			next->prev = prev;
		pthread_mutex_unlock(&key_destructor[key].mutex);

		emu_cpu_env *env = emu_cpu_clone(destr_function_env);
		unsigned callback_args[2] = {
			(unsigned) key_destructor[key].function,
			(unsigned) data->pointer };
		emu_set_return_value(env, (unsigned) callback_args);
		emu_cpu_run(env);
		emu_cpu_free(env);
	}
	free(data);
}

int handle__libc_getspecific(unsigned args[]) {
	pthread_key_t key = args[2];
	void **result = (void *) tswap32(args[1]);
	specific_data *data = __libc_getspecific(key);
	*result = data ? data->pointer : NULL;
	return 0;
}

int handle__libc_key_create(unsigned args[], emu_cpu_env *env) {
	if (!destr_function_env)
		destr_function_env = emu_cpu_clone(env);
	pthread_key_t *key = (void *) tswap32(args[2]);
	void *target_destr_function = (void *) args[3];
	int ret = __libc_key_create(key, destr_function);
	assign_result32(args[1], ret);
	if (!ret) {
		key_destructor[*key].function = target_destr_function;
		specific_data *data = key_destructor[*key].head;
		while (data) {
			specific_data *next = data->next;
			free(data);
			data = next;
		}
		key_destructor[*key].head = NULL;
	}
	return 0;
}

int handle__libc_setspecific(unsigned args[]) {
	pthread_key_t key = args[2];
	void *pointer = (void *) args[3];
	int ret = 0;
	specific_data *data = __libc_getspecific(key);
	if (!data && pointer) {
		data = malloc(sizeof(specific_data));
		ret = __libc_setspecific(key, data);
		if (ret)
			free(data);
		else {
			data->key = key;
			if (key_destructor[key].function) {
				data->prev = NULL;
				pthread_mutex_lock(&key_destructor[key].mutex);
				data->next = key_destructor[key].head;
				if (data->next)
					data->next->prev = data;
				key_destructor[key].head = data;
				pthread_mutex_unlock(
					&key_destructor[key].mutex);
			}
		}
	}
	assign_result32(args[1], ret);
	if (!ret && data)
		data->pointer = pointer;
	return 0;
}

int handle_lseek64(unsigned args[]) {
	int fd = tswap32(args[2]);
#ifdef BSWAP_NEEDED
	static const int I0 = 1;
	static const int I1 = 0;
#else
	static const int I0 = 0;
	static const int I1 = 1;
#endif
	int_s64 is = { .i = { tswap32(args[I0 + 3]), tswap32(args[I1 + 3]) } };
	off64_t offset = is.i64;
	int whence = tswap32(args[5]);
	off64_t ret = lseek64(fd, offset, whence);
#ifdef BSWAP_NEEDED
#define tswap64(v) ((typeof(v)) bswap_64((uint64_t) v))
#else
#define tswap64(v) v
#endif
	off64_t *result = (void *) tswap32(args[1]);
	*result = tswap64(ret);
	if (ret == -1)
		errno = tswap32(errno);
	return 0;
}

int handle_open64(unsigned args[]) {
	const char *pathname = (void *) tswap32(args[2]);
	int flags = tswap32(args[3]);
	mode_t mode = tswap32(args[4]);
	int ret = open64(pathname, flags, mode);
	assign_result32(args[1], ret);
	if (ret == -1)
		errno = tswap32(errno);
	return 0;
}

static int handle_sigaction(unsigned args[], emu_cpu_env *env) {
	int signum = target_to_host_signum(tswap32(args[2]));
	void *custom_handler = NULL;
	_Bool sa_siginfo = 0;
	_Bool onstack = 0;
	const struct sigaction *target_act = (void *) tswap32(args[3]);
	struct sigaction *act;
	if (target_act) {
		act = alloca(sizeof(struct sigaction));
		memswap32(act, target_act, size32of(struct sigaction));
		if (act->sa_flags & SA_SIGINFO) {
			if (act->sa_sigaction != (void *) SIG_DFL
				&& act->sa_sigaction != (void *) SIG_IGN) {
				custom_handler = target_act->sa_sigaction;
				act->sa_sigaction = sigaction_handler;
				sa_siginfo = 1;
				onstack = act->sa_flags & SA_ONSTACK;
				act->sa_flags &= ~SA_ONSTACK;
			}
		}
		else {
			if (act->sa_handler != (void *) SIG_DFL
				&& act->sa_handler != (void *) SIG_IGN) {
				custom_handler = target_act->sa_handler;
				act->sa_handler = signal_handler;
				onstack = act->sa_flags & SA_ONSTACK;
				act->sa_flags &= ~SA_ONSTACK;
				warnx("signal env %p", signal_handlers.env);
				if (onstack)
					warnx("sa_handler with onstack");
			}
		}
	}
	else
		act = NULL;
	struct sigaction *target_oldact = (void *) tswap32(args[4]);
	struct sigaction *oldact = target_oldact ?
		alloca(sizeof(struct sigaction)) : NULL;
	int ret = sigaction(signum, act, oldact);
	int *result = (void *) tswap32(args[1]);
	*result = ret;
	if (ret)
		warnx("sigaction: errno must be swapped");
	else {
		if (target_oldact) {
			memswap32(target_oldact, oldact,
				size32of(struct sigaction));
			if (oldact->sa_flags & SA_SIGINFO) {
				switch ((unsigned) oldact->sa_sigaction) {
				case (unsigned) SIG_DFL:
				case (unsigned) SIG_IGN:
					break;
				default:
					if (oldact->sa_sigaction
						== sigaction_handler)
// TODO: restore onstack
						warnx("sigaction: oldact with"
							" proxy handler");
					else
						warnx("sigaction: returning"
							" function handler");
				}
			}
			else
				warnx("no old siginfo");
		}
// TODO: race condition, block signum
		if (custom_handler)
			setup_sigaction_handler(signum, sa_siginfo,
				custom_handler, onstack, env);
	}
	return 0;
}

static int host_to_target_signum(int signum) {
//TODO: use translation table for other arch combinations
	return signum;
}

static void setup_sigaction_handler(int signum, _Bool sa_siginfo,
	void *target_handler, _Bool onstack, emu_cpu_env *env) {
	if ((unsigned) signum >= NSIG) {
		warnx("unsupported signum %d", signum);
		return;
	}
	if (!sigaction_handlers.env) {
		emu_cpu_env *new_env = emu_cpu_clone(env);
		sigaction_handlers.env = new_env;
	}
	if (sa_siginfo) {
		sigaction_handlers.sig[signum].handler = target_handler;
		sigaction_handlers.sig[signum].onstack = onstack;
	}
	else
		warnx("no siginfo");
}

static void setup_signal_handler(int signum, sighandler_t target_handler,
	emu_cpu_env *env) {
	if ((unsigned) signum >= NSIG) {
		warnx("unsupported signum %d", signum);
		return;
	}
	if (!signal_handlers.env) {
		emu_cpu_env *new_env = emu_cpu_clone(env);
		signal_handlers.env = new_env;
	}
	signal_handlers.handler[signum] = target_handler;
}

static void sigaction_handler(int signum, siginfo_t *info,
	__attribute__((unused)) void *uc) {
if (signum == SIGSEGV) {
 warnx("segv addr %p", info->si_addr);
 emu_print_debug();
}
	emu_cpu_env *env = emu_cpu_clone(sigaction_handlers.env);
	unsigned sa_handler_args[4];
	emu_set_return_value(env, (unsigned) sa_handler_args);
	thread_data *data = emu_get_thread_data();
	emu_copy_internal(env, data->env);
	stack_t oss;
	int change_stack = sigaction_handlers.sig[signum].onstack;
	if (change_stack)
		emu_set_stack(env, &data->target_ss, &oss);
	sa_handler_args[0] = (unsigned) sigaction_handlers.sig[signum].handler;
	sa_handler_args[1] = tswap32(host_to_target_signum(signum));
// TODO: Check offsets on powerpc
	siginfo_t target_info;
	memswap32(&target_info, info, size32of(siginfo_t));
	sa_handler_args[2] = (unsigned) tswap32(&target_info);
// TODO: Use thread current cpu_env ?
	target_ucontext_t *target_uc = emu_create_ucontext(env);
	sa_handler_args[3] = (unsigned) tswap32(target_uc);
// TODO: emu_cpu_run may not return. Free parent envs ?
	emu_cpu_run(env);
	if (change_stack)
		emu_set_stack(env, &oss, NULL);
	emu_cpu_free(env);
	free(target_uc);
}

static void signal_handler(int signum) {
	emu_cpu_env *env = emu_cpu_clone(signal_handlers.env);
	unsigned sa_handler_args[2];
	emu_set_return_value(env, (unsigned) sa_handler_args);
	sa_handler_args[0] = (unsigned) signal_handlers.handler[signum];
	sa_handler_args[1] = tswap32(host_to_target_signum(signum));
// TODO: emu_cpu_run may not return. Free parent envs ?
	emu_cpu_run(env);
	emu_cpu_free(env);
}

static void swap_sockaddr_body(struct sockaddr *sa) {
	switch (sa->sa_family) {
	case AF_LOCAL:
	case AF_INET:
		break;
	default:
		warnx("unhandled address family %d", sa->sa_family);
	}
}

static int target_to_host_map_flags(int target_flags) {
	int flags = target_flags & 0xffff9f3f;
#if defined(TARGET_I386)
	static const int TARGET_MAP_LOCKED = 0x02000;
	static const int TARGET_MAP_NORESERVE = 0x04000;
#elif defined(TARGET_POWERPC)
	static const int TARGET_MAP_LOCKED = 0x00080;
	static const int TARGET_MAP_NORESERVE = 0x00040;
#endif
	if (target_flags & TARGET_MAP_LOCKED)
		flags |= MAP_LOCKED;
	if (target_flags & TARGET_MAP_NORESERVE)
		flags |= MAP_NORESERVE;
	return flags;
}

static int target_to_host_signum(int signum) {
//TODO: use translation table for other arch combinations
	return signum;
}

static void target_to_host_sockaddr(struct sockaddr *host,
	const struct sockaddr *target, socklen_t addrlen) {
	memcpy(host, target, addrlen);
	host->sa_family = tswap16(host->sa_family);
	swap_sockaddr_body(host);
}

static int vsprintf_wrapper(char **buffer, const char *format, unsigned *ap) {
	*buffer = NULL;
	const char *start = format;
	const char *percent = strchr(format, '%');
	int written = 0;
	while (percent) {
		percent++;
		switch (*percent) {
		case '\0':
			percent = NULL;
			break;
		case '%':
			percent = strchr(percent + 1, '%');
			break;
		case 'X':
		case 'c':
		case 'd':
		case 'i':
		case 'o':
		case 'p':
		case 's':
		case 'u':
		case 'x':
		{
			unsigned uarg = tswap32(*ap++);
			percent = strchr(percent + 1, '%');
			if (percent) {
				char *token = strndup(start, percent - start);
				int n = add_uarg(buffer, written, token, uarg);
				free(token);
				if (n < 0)
					return n;
				written += n;
				start = percent;
			}
			else {
				int n = add_uarg(buffer, written, start, uarg);
				if (n < 0)
					return n;
				return written + n;
			}
			break;
		}
		case 'e':
		case 'E':

		case 'f':
		case 'F':

		case 'g':
		case 'G':

		case 'a':
		case 'A':

		case 'n':

		case 'm':

//		case 'C':
//		case 'S':
			warnx("unhandled %c\n", *percent);
		}
	}
	int n = add_end(buffer, written, start);
	if (n < 0)
		return n;
	return written + n;
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case ACCEPT: {
		int sockfd = tswap32(args[2]);
		struct sockaddr *addr = (void *) tswap32(args[3]);
		socklen_t *target_addrlen = (void *) tswap32(args[4]);
		socklen_t addrlen = tswap32(*target_addrlen);
		int ret = accept(sockfd, addr, &addrlen);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("accept: errno must be swapped");
		swap_sockaddr_body(addr);
		addr->sa_family = tswap16(addr->sa_family);
		*target_addrlen = tswap32(addrlen);
		return 0;
	}
	case BINDTEXTDOMAIN:
	{
		const char *domainname = (void *) tswap32(args[2]);
		const char *dirname = (void *) tswap32(args[3]);
		char *ret = bindtextdomain(domainname, dirname);
		assign_result32(args[1], ret);
		if (!ret)
			warnx("bindtextdomain: errno must be swapped");
		return 0;
	}
	case CALLOC:
	{
		size_t nmemb = tswap32(args[2]);
		size_t size = tswap32(args[3]);
		assign_result32(args[1], calloc(nmemb, size));
		return 0;
	}
	case CLOSE: {
		int fd = tswap32(args[2]);
		int ret = close(fd);
		assign_result32(args[1], ret);
		if (ret)
			errno = tswap32(errno);
		return 0;
	}
	case CONNECT: {
		int sockfd = tswap32(args[2]);
		const struct sockaddr *target_serv_addr
			= (void *) tswap32(args[3]);
		struct sockaddr serv_addr;
		socklen_t addrlen = tswap32(args[4]);
		target_to_host_sockaddr(&serv_addr, target_serv_addr, addrlen);
		int ret = connect(sockfd, &serv_addr, addrlen);
		int *result = (void *) tswap32(args[1]);
		*result = ret;
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case DCGETTEXT:
	{
		const char *domainname = (void *) tswap32(args[2]);
		const char *msgid = (void *) tswap32(args[3]);
		int category = tswap32(args[4]);
		assign_result32(args[1], dcgettext(domainname, msgid,
			category));
		return 0;
	}
	case __DCGETTEXT: {
		const char *domainname = (void *) tswap32(args[2]);
		const char *msgid = (void *) tswap32(args[3]);
		int category = tswap32(args[4]);
		assign_result32(args[1], __dcgettext(domainname, msgid,
			category));
		return 0;
	}
	case EPOLL_CREATE:
	{
		int size = tswap32(args[2]);
		int ret = epoll_create(size);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("epoll_create: errno must be swapped");
		return 0;
	}
	case EPOLL_CTL:
	{
		int epfd = tswap32(args[2]);
		int op = tswap32(args[3]);
		int fd = tswap32(args[4]);
		struct target_epoll_event *target_event
			= (void *) tswap32(args[5]);
		struct epoll_event event;
		event.events = tswap32(target_event->events);
		event.data = target_event->data;
		int ret = epoll_ctl(epfd, op, fd, &event);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case EPOLL_WAIT:
	{
		int epfd = tswap32(args[2]);
		struct target_epoll_event *target_events
			= (void *) tswap32(args[3]);
		int maxevents = tswap32(args[4]);
		int timeout = tswap32(args[5]);
		struct epoll_event *events = malloc(maxevents
			* sizeof(struct epoll_event));
		int ret = epoll_wait(epfd, events, maxevents, timeout);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("epoll_wait: errno must be swapped");
		int i;
		for (i = 0; i < ret; i++) {
			target_events[i].events = tswap32(events[i].events);
			target_events[i].data = events[i].data;
		}
		free(events);
		return 0;
	}
	case __ERRNO_LOCATION: {
		assign_result32(args[1], __errno_location());
		return 0;
	}
/*	case EXIT:
	{
warnx("exit called\n");
		int status = tswap32(args[1]);
		exit(status);
		return 0;
	}*/
/*	case FCLOSE:
	{
		void *fp = (void *) tswap32(args[2]);
		assign_result32(args[1], fclose(fp));
		return 0;
	}*/
	case FCNTL: {
		int fd = tswap32(args[2]);
		int cmd = tswap32(args[3]);
		unsigned arg = tswap32(args[4]);
		int ret = fcntl(fd, cmd, arg);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("fcntl: errno must be swapped");
		return 0;
	}
	case FFLUSH:
	{
		void *stream = (void *) tswap32(args[2]);
		assign_result32(args[1], fflush(stream));
		return 0;
	}
	case FOPEN64:
	{
		void *path = (void *) tswap32(args[2]);
		void *mode = (void *) tswap32(args[3]);
warnx("fopen64 call\n");
		assign_result32(args[1], fopen64(path, mode));
		return 0;
	}
	case FORK:
	{
		pid_t ret = fork();
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("fork: errno must be swapped");
		return 0;
	}
	case FPUTC:
	{
		int c = tswap32(args[2]);
		void *stream = (void *) tswap32(args[3]);
		assign_result32(args[1], fputc(c, stream));
		return 0;
	}
	case FREE:
	{
		void *ptr = (void *) tswap32(args[1]);
		free(ptr);
		return 0;
	}
	case FSYNC: {
		int fd = tswap32(args[2]);
		int *result = (void *) tswap32(args[1]);
		*result = fsync(fd);
		if (*result)
			errno = tswap32(errno);
		return 0;
	}
	case FTRUNCATE64:
	{
#ifdef BSWAP_NEEDED
		static const int I0 = 1;
		static const int I1 = 0;
#else
		static const int I0 = 0;
		static const int I1 = 1;
#endif
		int fd = tswap32(args[2]);
		int_s64 is = { .i = { tswap32(args[I0 + 3]),
			tswap32(args[I1 + 3]) } };
		off64_t length = is.i64;
		int ret = ftruncate64(fd, length);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("ftruncate64: errno must be swapped\n");
		return 0;
	}
	case FWRITE:
	{
		void *ptr = (void *) tswap32(args[2]);
		size_t size = tswap32(args[3]);
		size_t nmemb = tswap32(args[4]);
		void *stream = (void *) tswap32(args[5]);
		assign_result32(args[1], fwrite(ptr, size, nmemb, stream));
		return 0;
	}
	case GETCWD:
	{
		char *buf = (void *) tswap32(args[2]);
		size_t size = tswap32(args[3]);
		assign_result32(args[1], getcwd(buf, size));
		return 0;
	}
	case GETLINE:
	{
		void *lineptr = (void *) tswap32(args[2]);
		if (lineptr != NULL)
			warnx("lineptr not null\n");
		void *n = (void *) tswap32(args[3]);
		void *stream = (void *) tswap32(args[4]);
		assign_result32(args[1], getline(lineptr, n, stream));
		return 0;
	}
	case GETTEXT:
	{
		const char *msgid = (void *) tswap32(args[2]);
		assign_result32(args[1], gettext(msgid));
		return 0;
	}
	case INOTIFY_INIT1: {
		int flags = tswap32(args[2]);
		int ret = inotify_init1(flags);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case __LIBC_GETSPECIFIC:
		return handle__libc_getspecific(args);
	case __LIBC_KEY_CREATE:
		return handle__libc_key_create(args, env);
	case __LIBC_SETSPECIFIC:
		return handle__libc_setspecific(args);
	case LSEEK: {
		int fd = tswap32(args[2]);
		off_t offset = tswap32(args[3]);
		int whence = tswap32(args[4]);
		off_t ret = lseek(fd, offset, whence);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case LSEEK64:
		return handle_lseek64(args);
	case MALLOC:
	{
		size_t size = tswap32(args[2]);
		assign_result32(args[1], malloc(size));
		return 0;
	}
	case MKDIR:
	{
		const char *pathname = (void *) tswap32(args[2]);
		mode_t mode = tswap32(args[3]);
		int *result = (void *) tswap32(args[1]);
		*result = mkdir(pathname, mode);
		if (*result) {
			int *target_errno = (void *) tswap32(args[4]);
//translate errnos
			*target_errno = tswap32(errno);
		}
		return 0;
	}
	case MMAP:
	{
		void *addr = (void *) tswap32(args[2]);
		size_t length = tswap32(args[3]);
		int prot = tswap32(args[4]);
		int target_flags = tswap32(args[5]);
		int flags = target_to_host_map_flags(target_flags);
		int fd = tswap32(args[6]);
		off_t offset = tswap32(args[7]);
		void *map = mmap(addr, length, prot, flags, fd, offset);
		assign_result32(args[1], map);
		if (map == MAP_FAILED)
			errno = tswap32(errno);
		return 0;
	}
	case MMAP64:
	{
		void *addr = (void *) tswap32(args[2]);
		size_t length = tswap32(args[3]);
		int prot = tswap32(args[4]);
		int target_flags = tswap32(args[5]);
		int flags = target_to_host_map_flags(target_flags);
		int fd = tswap32(args[6]);
#ifdef BSWAP_NEEDED
		static const int I0 = 1;
		static const int I1 = 0;
#else
		static const int I0 = 0;
		static const int I1 = 1;
#endif
		int_s64 is = { .i = { tswap32(args[I0 + 7]),
			tswap32(args[I1 + 7]) } };
		off64_t offset = is.i64;
		static void *stack_guard;
		if (!stack_guard)
			stack_guard = (void *) ((unsigned) &offset
				& 0xff000000);
		if (!(flags & MAP_FIXED) && addr + length > stack_guard)
			addr = stack_guard - length;
		void *map = mmap64(addr, length, prot, flags, fd, offset);
		assign_result32(args[1], map);
		if (map == MAP_FAILED)
			errno = tswap32(errno);
		return 0;
	}
	case MSYNC: {
		void *addr = (void *) tswap32(args[2]);
		size_t length = tswap32(args[3]);
		int flags = tswap32(args[4]);
		int ret = msync(addr, length, flags);
		int *result = (void *) tswap32(args[1]);
		*result = ret;
		if (ret == -1)
			warnx("msync: errno must be swapped");
		return 0;
	}
	case MUNMAP:
	{
		void *addr = (void *) tswap32(args[2]);
		size_t length = tswap32(args[3]);
		int *result = (void *) tswap32(args[1]);
		*result = munmap(addr, length);
		if (*result)
			errno = tswap32(errno);
		return 0;
	}
	case NANOSLEEP: {
		const struct timespec *target_req = (void *) tswap32(args[2]);
		struct timespec req;
		memswap32(&req, target_req, size32of(struct timespec));
		struct timespec *target_rem = (void *) tswap32(args[3]);
		struct timespec *rem = target_rem ?
			alloca(sizeof(struct timespec)) : NULL;
		int *result = (void *) tswap32(args[1]);
		*result = nanosleep(&req, rem);
		if (*result) {
			if (errno == EINTR && target_rem)
				memswap32(target_rem, rem,
					size32of(struct timespec));
			errno = tswap32(errno);
		}
		return 0;
	}
	case NL_LANGINFO: {
		nl_item item = tswap32(args[2]);
		assign_result32(args[1], nl_langinfo(item));
		return 0;
	}
	case OPEN: {
		const char *pathname = (void *) tswap32(args[2]);
		int flags = tswap32(args[3]);
		mode_t mode = tswap32(args[4]);
		int ret = open(pathname, flags, mode);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case OPEN64:
		return handle_open64(args);
	case PIPE:
	{
		int *target_pipefd = (void *) tswap32(args[2]);
		int pipefd[2];
		int *result = (void *) tswap32(args[1]);
		*result = pipe(pipefd);
		if (*result)
			errno = tswap32(errno);
		memswap32(target_pipefd, pipefd, 2);
		return 0;
	}
	case PIPE2: {
		int *target_pipefd = (void *) tswap32(args[2]);
		int pipefd[2];
		int flags = tswap32(args[3]);
		int *result = (void *) tswap32(args[1]);
		*result = pipe2(pipefd, flags);
		if (*result)
			errno = tswap32(errno);
		memswap32(target_pipefd, pipefd, 2);
		return 0;
	}
	case POSIX_MEMALIGN:
	{
		void **target_memptr = (void *) tswap32(args[2]);
		void *memptr;
		size_t alignment = tswap32(args[3]);
		size_t size = tswap32(args[4]);
		assign_result32(args[1], posix_memalign(&memptr, alignment,
			size));
		*target_memptr = tswap32(memptr);
		return 0;
	}
	case PRINTF:
	{
/*		const char *format = (void *) tswap32(args[2]);
		unsigned *ap = (void *) tswap32(args[3]);
		assign_result32(args[1], vfprintf_wrapper(stdout, format, ap));
*/		return 0;
	}
	case PTHREAD_COND_BROADCAST: {
		pthread_cond_t *cond = (void *) tswap32(args[1]);
		pthread_cond_broadcast(cond);
		return 0;
	}
	case PTHREAD_COND_DESTROY: {
		pthread_cond_t *cond = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_cond_destroy(cond));
		return 0;
	}
	case PTHREAD_COND_INIT: {
		pthread_cond_t *cond = (void *) tswap32(args[1]);
		const pthread_condattr_t *cond_attr = (void *) tswap32(args[2]);
		pthread_cond_init(cond, cond_attr);
		return 0;
	}
	case PTHREAD_COND_SIGNAL: {
		pthread_cond_t *cond = (void *) tswap32(args[1]);
		pthread_cond_signal(cond);
		return 0;
	}
	case PTHREAD_COND_TIMEDWAIT: {
		pthread_cond_t *cond = (void *) tswap32(args[2]);
		pthread_mutex_t *mutex = (void *) tswap32(args[3]);
		const struct timespec *target_abstime
			= (void *) tswap32(args[4]);
		struct timespec abstime;
		memswap32(&abstime, target_abstime, size32of(struct timespec));
		assign_result32(args[1], pthread_cond_timedwait(cond, mutex,
			&abstime));
		return 0;
	}
	case PTHREAD_COND_WAIT: {
		pthread_cond_t *cond = (void *) tswap32(args[1]);
		pthread_mutex_t *mutex = (void *) tswap32(args[2]);
		pthread_cond_wait(cond, mutex);
		return 0;
	}
	case PTHREAD_MUTEX_DESTROY:
	{
		pthread_mutex_t *mutex = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_mutex_destroy(mutex));
		return 0;
	}
	case PTHREAD_MUTEX_INIT:
	{
		pthread_mutex_t *mutex = (void *) tswap32(args[1]);
		const pthread_mutexattr_t *mutexattr
			= (void *) tswap32(args[2]);
		pthread_mutex_init(mutex, mutexattr);
		return 0;
	}
	case PTHREAD_MUTEX_LOCK:
	{
		pthread_mutex_t *mutex = (void *) tswap32(args[2]);
// check EDEADLK
		assign_result32(args[1], pthread_mutex_lock(mutex));
		return 0;
	}
	case PTHREAD_MUTEX_UNLOCK:
	{
		pthread_mutex_t *mutex = (void *) tswap32(args[2]);
		assign_result32(args[1], pthread_mutex_unlock(mutex));
		return 0;
	}
	case PTHREAD_SELF: {
		pthread_t *result = (void *) tswap32(args[1]);
		*result = pthread_self();
		return 0;
	}
	case PUTCHAR:
	{
		int c = tswap32(args[2]);
		assign_result32(args[1], putchar(c));
		return 0;
	}
	case PUTS:
	{
warnx("puts call\n");
		void *s = (void *) tswap32(args[2]);
		assign_result32(args[1], puts(s));
		return 0;
	}
	case READ: {
		int fd = tswap32(args[2]);
		void *buf = (void *) tswap32(args[3]);
		size_t count = tswap32(args[4]);
		int ret = read(fd, buf, count);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case REALLOC:
	{
		void *ptr = (void *) tswap32(args[2]);
		size_t size = tswap32(args[3]);
		assign_result32(args[1], realloc(ptr, size));
		return 0;
	}
	case RECV: {
		int s = tswap32(args[2]);
		void *buf = (void *) tswap32(args[3]);
		size_t len = tswap32(args[4]);
		int flags = tswap32(args[5]);
		ssize_t ret = recv(s, buf, len, flags);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case RECVFROM: {
		int s = tswap32(args[2]);
		void *buf = (void *) tswap32(args[3]);
		size_t len = tswap32(args[4]);
		int flags = tswap32(args[5]);
		struct sockaddr *from = (void *) tswap32(args[6]);
		socklen_t *target_fromlen = (void *) tswap32(args[7]);
		socklen_t fromlen = tswap32(*target_fromlen);
		ssize_t ret = recvfrom(s, buf, len, flags, from, &fromlen);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("recvfrom: errno must be swapped");
		swap_sockaddr_body(from);
		from->sa_family = tswap16(from->sa_family);
		*target_fromlen = tswap32(fromlen);
		return 0;
	}
	case RECVMSG: {
		int s = tswap32(args[2]);
		struct msghdr *target_msg = (void *) tswap32(args[3]);
		struct msghdr msg;
		if (target_msg->msg_name) {
			socklen_t namelen = tswap32(target_msg->msg_namelen);
			msg.msg_namelen = namelen;
			msg.msg_name = malloc(namelen);
			target_to_host_sockaddr(msg.msg_name,
				tswap32(target_msg->msg_name), namelen);
		}
		else {
			msg.msg_name = NULL;
			msg.msg_namelen = 0;
		}
		size_t iovlen = tswap32(target_msg->msg_iovlen);
		msg.msg_iovlen = iovlen;
		msg.msg_iov = malloc(iovlen * sizeof(struct iovec));
		memswap32(msg.msg_iov, tswap32(target_msg->msg_iov),
			iovlen * size32of(struct iovec));
		socklen_t controllen = tswap32(target_msg->msg_controllen);
		msg.msg_controllen = controllen;
		msg.msg_control = tswap32(target_msg->msg_control);
		int flags = tswap32(args[4]);
		ssize_t ret = recvmsg(s, &msg, flags);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("recvmsg: errno must be swapped");
		struct cmsghdr *curhdr = NULL;
		struct cmsghdr *nxthdr = CMSG_FIRSTHDR(&msg);
		while (nxthdr) {
			curhdr = nxthdr;
			nxthdr = CMSG_NXTHDR(&msg, curhdr);
			if (curhdr) {
				if (curhdr->cmsg_level != SOL_SOCKET
					|| curhdr->cmsg_type != SCM_RIGHTS)
					warnx("unknown ancillary data");
				unsigned char *data = CMSG_DATA(curhdr);
				size_t datalen = curhdr->cmsg_len - (data
					- (unsigned char *) curhdr);
				if (datalen != sizeof(int))
					warnx("unexpected datalen %d", datalen);
				int *fd = (int *) data;
				*fd = tswap32(*fd);
				memswap32(curhdr, curhdr, 3);
			}
		}
		target_msg->msg_controllen = tswap32(msg.msg_controllen);
		target_msg->msg_flags = tswap32(msg.msg_flags);
		free(msg.msg_name);
		free(msg.msg_iov);
		return 0;
	}
	case SEND: {
		int s = tswap32(args[2]);
		const void *buf = (void *) tswap32(args[3]);
		size_t len = tswap32(args[4]);
		int flags = tswap32(args[5]);
		ssize_t ret = send(s, buf, len, flags);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("send: errno must be swapped");
		return 0;
	}
	case SENDMSG: {
		int s = tswap32(args[2]);
		const struct msghdr *target_msg = (void *) tswap32(args[3]);
		struct msghdr msg;
		if (target_msg->msg_name) {
			socklen_t namelen = tswap32(target_msg->msg_namelen);
			msg.msg_namelen = namelen;
			msg.msg_name = malloc(namelen);
			target_to_host_sockaddr(msg.msg_name,
				tswap32(target_msg->msg_name), namelen);
		}
		else {
			msg.msg_name = NULL;
			msg.msg_namelen = 0;
		}
		size_t iovlen = tswap32(target_msg->msg_iovlen);
		msg.msg_iovlen = iovlen;
		msg.msg_iov = malloc(iovlen * sizeof(struct iovec));
		memswap32(msg.msg_iov, tswap32(target_msg->msg_iov),
			iovlen * size32of(struct iovec));
		socklen_t controllen = tswap32(target_msg->msg_controllen);
		msg.msg_controllen = controllen;
		msg.msg_control = malloc(controllen);
		void *target_msg_control = tswap32(target_msg->msg_control);
		memcpy(msg.msg_control, target_msg_control, controllen);
		socklen_t i = 0;
		struct cmsghdr *target_cmsg = target_msg_control;
		struct cmsghdr *host_cmsg = msg.msg_control;
// TODO: clean up this
		while (i < controllen) {
			memswap32(host_cmsg, target_cmsg, 3);
			if (host_cmsg->cmsg_level != SOL_SOCKET
				|| host_cmsg->cmsg_type != SCM_RIGHTS)
				warnx("unknown ancillary data");
			unsigned char *target_data = CMSG_DATA(target_cmsg);
			unsigned char *data = CMSG_DATA(host_cmsg);
			size_t datalen = host_cmsg->cmsg_len - (data
				- (unsigned char *) host_cmsg);
			if (datalen != sizeof(int))
				warnx("unexpected datalen %d", datalen);
			int *target_fd = (int *) target_data;
			int *fd = (int *) data;
			*fd = tswap32(*target_fd);
			size_t hdrlen = CMSG_ALIGN(host_cmsg->cmsg_len);
			i += hdrlen;
			target_cmsg = (struct cmsghdr *)
				(((unsigned char *) target_cmsg) + hdrlen);
			host_cmsg = (struct cmsghdr *)
				(((unsigned char *) host_cmsg) + hdrlen);
		}
		int flags = tswap32(args[4]);
		ssize_t ret = sendmsg(s, &msg, flags);
		assign_result32(args[1], ret);
		if (ret == -1)
			warnx("sendmsg: errno must be swapped");
		free(msg.msg_name);
		free(msg.msg_iov);
		free(msg.msg_control);
		return 0;
	}
	case SETENV:
	{
		const char *name = (void *) tswap32(args[1]);
		const char *value = (void *) tswap32(args[2]);
		int overwrite = tswap32(args[3]);
		setenv(name, value, overwrite);
		return 0;
	}
	case SETLOCALE:
	{
// TODO: check category values
		int category = tswap32(args[2]);
		const char *locale = (void *) tswap32(args[3]);
		assign_result32(args[1], setlocale(category, locale));
		return 0;
	}
	case SIGACTION:
		return handle_sigaction(args, env);
	case SIGALTSTACK:
	{
		const stack_t *target_ss = (void *) tswap32(args[2]);
		stack_t *target_oss = (void *) tswap32(args[3]);
		thread_data *data = emu_get_thread_data();
		if (target_oss)
			memswap32(target_oss, &data->target_ss,
				size32of(stack_t));
		if (target_ss)
			memswap32(&data->target_ss, target_ss,
				size32of(stack_t));
		int ret = 0;
		assign_result32(args[1], ret);
//		if (ret == -1)
//			errno = tswap32(errno);
		return 0;
	}
	case SIGNAL: {
		int signum = target_to_host_signum(tswap32(args[2]));
		sighandler_t target_handler = (void *) args[3];
		sighandler_t handler = tswap32(target_handler);
		_Bool custom_handler = 0;
		sighandler_t ret;
		switch ((unsigned) handler) {
		case (unsigned) SIG_DFL:
		case (unsigned) SIG_IGN:
			ret = signal(signum, handler);
			break;
		default:
			custom_handler = 1;
			ret = signal(signum, signal_handler);
		}
		if (ret == SIG_ERR)
			warnx("signal: errno must be swapped");
		else {
			switch ((unsigned) ret) {
			case (unsigned) SIG_DFL:
			case (unsigned) SIG_IGN:
				break;
			default:
				if (ret == signal_handler)
					warnx("signal: "
						"returning proxy handler");
				else
					warnx("signal: "
						"returning function handler");
			}
// TODO: race condition, block signum
			if (custom_handler)
				setup_signal_handler(signum, target_handler,
					env);
		}
		assign_result32(args[1], ret);
		return 0;
	}
	case STRFTIME:
	{
// TODO: check sizeof struct tm, last elements
		char *s = (void *) tswap32(args[2]);
		size_t max = tswap32(args[3]);
		const char *format = (void *) tswap32(args[4]);
		const struct tm *target_tm = (void *) tswap32(args[5]);
		struct tm tm;
		memswap32(&tm, target_tm, size32of(struct tm));
		assign_result32(args[1], strftime(s, max, format, &tm));
		return 0;
	}
	case STUB_INIT:
	{
		thread_data *data = malloc(sizeof(thread_data));
		data->env = env;
		data->detachstate = PTHREAD_CREATE_JOINABLE;
		data->target_ss.ss_sp = NULL;
		data->target_ss.ss_flags = SS_DISABLE;
		data->target_ss.ss_size = 0;
		emu_set_thread_data(data);
		return 0;
	}
	case TEXTDOMAIN:
	{
		const char *domainname = (void *) tswap32(args[2]);
		char *ret = textdomain(domainname);
		assign_result32(args[1], ret);
		if (!ret)
			warnx("textdomain: errno must be swapped");
		return 0;
	}
	case UNSETENV:
	{
		const char *name = (void *) tswap32(args[1]);
		unsetenv(name);
		return 0;
	}
	case VFPRINTF_FREE:
	{
		char *buffer = (void *) tswap32(args[1]);
		free(buffer);
		return 0;
	}
	case VFPRINTF_PRINT:
	{
		char **target_buffer = (void *) tswap32(args[2]);
		char *buffer;
		const char *format = (void *) tswap32(args[3]);
		unsigned *ap = (void *) tswap32(args[4]);
		assign_result32(args[1], vsprintf_wrapper(&buffer, format, ap));
		*target_buffer = tswap32(buffer);
		return 0;
	}
	case WAITPID: {
		pid_t pid = tswap32(args[2]);
		int *target_status = (void *) tswap32(args[3]);
		int status;
		int options = tswap32(args[4]);
		pid_t ret = waitpid(pid, &status, options);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		if (target_status)
			*target_status = tswap32(status);
		return 0;
	}
	case WRITE: {
		int fd = tswap32(args[2]);
		void *buf = (void *) tswap32(args[3]);
		size_t count = tswap32(args[4]);
		int ret = write(fd, buf, count);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		return 0;
	}
	case WRITEV:
	{
		int fd = tswap32(args[2]);
		const struct iovec *target_iov = (void *) tswap32(args[3]);
		int iovcnt = tswap32(args[4]);
		struct iovec *iov = malloc(iovcnt * sizeof(struct iovec));
		memswap32(iov, target_iov, iovcnt * size32of(struct iovec));
		int ret = writev(fd, iov, iovcnt);
		assign_result32(args[1], ret);
		if (ret == -1)
			errno = tswap32(errno);
		free(iov);
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.\n", function);
		return -1;
	}
}
