#include <err.h>
#include <alsa/asoundlib.h>
#include <emu-stub/handler.h>
#include <emu-stub/handler-api.h>

#include "libasound_const.h"

typedef struct elem_callback_data {
	snd_mixer_elem_callback_t callback;
	void *private;
	struct elem_callback_data *next;
} elem_callback_data;

static struct {
	emu_cpu_env *env;
	unsigned callback_args[3];
} elem_callback_env;

static elem_callback_data *first_elem_callback_data;

static int elem_callback(snd_mixer_elem_t *elem, unsigned int mask) {
	static int in_progress;
	if (in_progress) {
		warnx("elem_callback in progress!");
		exit(EXIT_FAILURE);
	}
	in_progress = 1;
	elem_callback_data *data = snd_mixer_elem_get_callback_private(elem);
	elem_callback_env.callback_args[0] = (unsigned) data->callback;
	elem_callback_env.callback_args[1] = (unsigned) elem;
	elem_callback_env.callback_args[2] = tswap32(mask);
	emu_cpu_run(elem_callback_env.env);
	int result = tswap32(elem_callback_env.callback_args[0]);
	in_progress = 0;
	return result;
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case SND_CARD_GET_INDEX:
	{
		const char *name = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_card_get_index(name));
		return 0;
	}
	case SND_CARD_NEXT:
	{
		int *card = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_card_next(card));
		return 0;
	}
	case SND_CONFIG_GET_INTEGER:
	{
		const snd_config_t *config = (void *) args[2];
		long *target_value = (void *) tswap32(args[3]);
		long value;
		assign_result32(args[1], snd_config_get_integer(config,
			&value));
		*target_value = tswap32(value);
		return 0;
	}
	case SND_CONFIG_GET_STRING:
	{
		const snd_config_t *config = (void *) args[2];
		const char **target_value = (void *) tswap32(args[3]);
		const char *value;
		assign_result32(args[1], snd_config_get_string(config, &value));
		*target_value = tswap32(value);
		return 0;
	}
	case SND_CONFIG_GET_TYPE:
	{
		const snd_config_t *config = (void *) args[2];
		assign_result32(args[1], snd_config_get_type(config));
		return 0;
	}
	case SND_CONFIG_ITERATOR_END:
	{
		const snd_config_t *node = (void *) args[2];
		snd_config_iterator_t *result = (void *) tswap32(args[1]);
		*result = snd_config_iterator_end(node);
		return 0;
	}
	case SND_CONFIG_ITERATOR_ENTRY:
	{
		const snd_config_iterator_t iterator = (void *) args[2];
		snd_config_t **result = (void *) tswap32(args[1]);
		*result = snd_config_iterator_entry(iterator);
		return 0;
	}
	case SND_CONFIG_ITERATOR_FIRST:
	{
		const snd_config_t *node = (void *) args[2];
		snd_config_iterator_t *result = (void *) tswap32(args[1]);
		*result = snd_config_iterator_first(node);
		return 0;
	}
	case SND_CONFIG_ITERATOR_NEXT:
	{
		const snd_config_iterator_t iterator = (void *) args[2];
		snd_config_iterator_t *result = (void *) tswap32(args[1]);
		*result = snd_config_iterator_next(iterator);
		return 0;
	}
	case SND_CONFIG_SEARCH:
	{
		snd_config_t *config = (void *) args[2];
		const char *key = (void *) tswap32(args[3]);
		snd_config_t **_result = (void *) tswap32(args[4]);
		assign_result32(args[1], snd_config_search(config, key,
			_result));
		return 0;
	}
	case SND_CONFIG_UPDATE:
	{
		assign_result32(args[1], snd_config_update());
		snd_config_t **target_snd_config = (void *) tswap32(args[2]);
		*target_snd_config = snd_config;
		return 0;
	}
	case SND_CONFIG_UPDATE_FREE_GLOBAL:
	{
		assign_result32(args[1], snd_config_update_free_global());
		return 0;
	}
	case SND_CTL_CARD_INFO:
	{
		void *ctl = (void *) args[2];
		void *info = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_ctl_card_info(ctl, info));
		return 0;
	}
	case SND_CTL_CARD_INFO_GET_ID:
	{
		void *obj = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_ctl_card_info_get_id(obj));
		return 0;
	}
	case SND_CTL_CARD_INFO_GET_NAME:
	{
		const snd_ctl_card_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_ctl_card_info_get_name(obj));
		return 0;
	}
	case SND_CTL_CARD_INFO_SIZEOF:
	{
		assign_result32(args[1], snd_ctl_card_info_sizeof());
		return 0;
	}
	case SND_CTL_CLOSE:
	{
		snd_ctl_t *ctl = (void *) args[2];
		assign_result32(args[1], snd_ctl_close(ctl));
		return 0;
	}
	case SND_CTL_ELEM_ID_CLEAR:
	{
		snd_ctl_elem_id_t *obj = (void *) args[1];
		snd_ctl_elem_id_clear(obj);
		return 0;
	}
	case SND_CTL_ELEM_ID_FREE:
	{
		snd_ctl_elem_id_t *obj = (void *) args[1];
		snd_ctl_elem_id_free(obj);
		return 0;
	}
	case SND_CTL_ELEM_ID_MALLOC:
	{
		// Assuming snd_ctl_elem_id is always created by host.
		snd_ctl_elem_id_t **ptr = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_ctl_elem_id_malloc(ptr));
		return 0;
	}
	case SND_CTL_ELEM_ID_SET_INTERFACE:
	{
		snd_ctl_elem_id_t *obj = (void *) args[1];
		snd_ctl_elem_iface_t val = tswap32(args[2]);
		snd_ctl_elem_id_set_interface(obj, val);
		return 0;
	}
	case SND_CTL_ELEM_ID_SET_NAME:
	{
		snd_ctl_elem_id_t *obj = (void *) args[1];
		const char *val = (void *) tswap32(args[2]);
		snd_ctl_elem_id_set_name(obj, val);
		return 0;
	}
	case SND_CTL_ELEM_INFO_CLEAR:
	{
		snd_ctl_elem_info_t *obj = (void *) args[1];
		snd_ctl_elem_info_clear(obj);
		return 0;
	}
	case SND_CTL_ELEM_INFO_FREE:
	{
		snd_ctl_elem_info_t *obj = (void *) args[1];
		snd_ctl_elem_info_free(obj);
		return 0;
	}
	case SND_CTL_ELEM_INFO_GET_COUNT:
	{
		const snd_ctl_elem_info_t *obj = (void *) args[2];
		assign_result32(args[1], snd_ctl_elem_info_get_count(obj));
		return 0;
	}
	case SND_CTL_ELEM_INFO_MALLOC:
	{
		// Assuming snd_ctl_elem_info is always created by host.
		snd_ctl_elem_info_t **ptr = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_ctl_elem_info_malloc(ptr));
		return 0;
	}
	case SND_CTL_ELEM_VALUE_CLEAR:
	{
		snd_ctl_elem_value_t *obj = (void *) args[1];
		snd_ctl_elem_value_clear(obj);
		return 0;
	}
	case SND_CTL_ELEM_VALUE_FREE:
	{
		snd_ctl_elem_value_t *obj = (void *) args[1];
		snd_ctl_elem_value_free(obj);
		return 0;
	}
	case SND_CTL_ELEM_VALUE_MALLOC:
	{
		// Assuming snd_ctl_elem_value is always created by host.
		snd_ctl_elem_value_t **ptr = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_ctl_elem_value_malloc(ptr));
		return 0;
	}
	case SND_CTL_NAME:
	{
		snd_ctl_t *ctl = (void *) args[2];
		assign_result32(args[1], snd_ctl_name(ctl));
		return 0;
	}
	case SND_CTL_OPEN:
	{
		snd_ctl_t **ctl = (void *) tswap32(args[2]);
		const char *name = (void *) tswap32(args[3]);
		int mode = tswap32(args[4]);
		assign_result32(args[1], snd_ctl_open(ctl, name, mode));
		return 0;
	}
	case SND_CTL_PCM_INFO:
	{
		snd_ctl_t *ctl = (void *) args[2];
		snd_pcm_info_t *info = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_ctl_pcm_info(ctl, info));
		return 0;
	}
	case SND_CTL_PCM_NEXT_DEVICE:
	{
		void *ctl = (void *) args[2];
#ifdef BSWAP_NEEDED
		int *target_device = (void *) tswap32(args[3]);
		int device = tswap32(*target_device);
		assign_result32(args[1], snd_ctl_pcm_next_device(ctl, &device));
		*target_device = tswap32(device);
#else
		void *device = (void *) args[3];
		assign_result32(args[1], snd_ctl_pcm_next_device(ctl, device));
#endif
		return 0;
	}
	case SND_HCTL_CLOSE:
	{
		snd_hctl_t *hctl = (void *) args[2];
		assign_result32(args[1], snd_hctl_close(hctl));
		return 0;
	}
	case SND_HCTL_ELEM_INFO:
	{
		snd_hctl_elem_t *elem = (void *) args[2];
		snd_ctl_elem_info_t *info = (void *) args[3];
		assign_result32(args[1], snd_hctl_elem_info(elem, info));
		return 0;
	}
	case SND_HCTL_FIND_ELEM:
	{
		snd_hctl_t *hctl = (void *) args[2];
		const snd_ctl_elem_id_t *id = (void *) args[3];
		snd_hctl_elem_t **result = (void *) tswap32(args[1]);
		*result = snd_hctl_find_elem(hctl, id);
		return 0;
	}
	case SND_HCTL_FREE:
	{
		snd_hctl_t *hctl = (void *) args[2];
		assign_result32(args[1], snd_hctl_free(hctl));
		return 0;
	}
	case SND_HCTL_LOAD:
	{
		snd_hctl_t *hctl = (void *) args[2];
		assign_result32(args[1], snd_hctl_load(hctl));
		return 0;
	}
	case SND_HCTL_OPEN:
	{
		snd_hctl_t **hctl = (void *) tswap32(args[2]);
		const char *name = (void *) tswap32(args[3]);
		int mode = tswap32(args[4]);
		assign_result32(args[1], snd_hctl_open(hctl, name, mode));
		return 0;
	}
	case SND_MIXER_ATTACH:
	{
		snd_mixer_t *mixer = (void *) args[2];
		const char *name = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_mixer_attach(mixer, name));
		return 0;
	}
	case SND_MIXER_CLOSE:
	{
		snd_mixer_t *mixer = (void *) args[2];
		assign_result32(args[1], snd_mixer_close(mixer));
		elem_callback_data *data = first_elem_callback_data;
		while (data) {
			elem_callback_data *next = data->next;
			free(data);
			data = next;
		}
		return 0;
	}
	case SND_MIXER_ELEM_GET_CALLBACK_PRIVATE:
	{
		const snd_mixer_elem_t *obj = (void *) args[2];
		elem_callback_data *data
			= snd_mixer_elem_get_callback_private(obj);
		void **result = (void *) tswap32(args[1]);
		*result = data->private;
		return 0;
	}
	case SND_MIXER_ELEM_NEXT:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		snd_mixer_elem_t **result = (void *) tswap32(args[1]);
		*result = snd_mixer_elem_next(elem);
		return 0;
	}
	case SND_MIXER_ELEM_SET_CALLBACK:
	{
		if (!elem_callback_env.env) {
			emu_cpu_env *new_env = emu_cpu_clone(env);
			elem_callback_env.env = new_env;
			emu_set_return_value(new_env,
				(unsigned) elem_callback_env.callback_args);
		}
		snd_mixer_elem_t *obj = (void *) args[1];
		snd_mixer_elem_callback_t val = (void *) args[2];
		elem_callback_data *data
			= snd_mixer_elem_get_callback_private(obj);
		if (data)
			data->callback = val;
		else {
			data = malloc(sizeof(elem_callback_data));
			data->callback = val;
			data->private = NULL;
			data->next = first_elem_callback_data;
			first_elem_callback_data = data;
			snd_mixer_elem_set_callback(obj, elem_callback);
			snd_mixer_elem_set_callback_private(obj, data);
		}
		return 0;
	}
	case SND_MIXER_ELEM_SET_CALLBACK_PRIVATE:
	{
		snd_mixer_elem_t *obj = (void *) args[1];
		void *val = (void *) args[2];
		elem_callback_data *data
			= snd_mixer_elem_get_callback_private(obj);
		data->private = val;
		return 0;
	}
	case SND_MIXER_FIRST_ELEM:
	{
		snd_mixer_t *mixer = (void *) args[2];
		snd_mixer_elem_t **result = (void *) tswap32(args[1]);
		*result = snd_mixer_first_elem(mixer);
		return 0;
	}
	case SND_MIXER_LOAD:
	{
		snd_mixer_t *mixer = (void *) args[2];
		assign_result32(args[1], snd_mixer_load(mixer));
		return 0;
	}
	case SND_MIXER_OPEN:
	{
		snd_mixer_t **mixer = (void *) tswap32(args[2]);
		int mode = tswap32(args[3]);
		if (first_elem_callback_data)
			warnx("Another mixer is already opened.");
		assign_result32(args[1], snd_mixer_open(mixer, mode));
		return 0;
	}
	case SND_MIXER_SELEM_GET_CAPTURE_VOLUME_RANGE:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		long *target_min = (void *) tswap32(args[3]);
		long min;
		long *target_max = (void *) tswap32(args[4]);
		long max;
		assign_result32(args[1],
			snd_mixer_selem_get_capture_volume_range(elem, &min,
			&max));
		*target_min = tswap32(min);
		*target_max = tswap32(max);
		return 0;
	}
	case SND_MIXER_SELEM_GET_NAME:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1], snd_mixer_selem_get_name(elem));
		return 0;
	}
	case SND_MIXER_SELEM_GET_PLAYBACK_VOLUME_RANGE:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		long *target_min = (void *) tswap32(args[3]);
		long min;
		long *target_max = (void *) tswap32(args[4]);
		long max;
		assign_result32(args[1],
			snd_mixer_selem_get_playback_volume_range(elem, &min,
			&max));
		*target_min = tswap32(min);
		*target_max = tswap32(max);
		return 0;
	}
	case SND_MIXER_SELEM_HAS_CAPTURE_CHANNEL:
	{
		snd_mixer_elem_t *obj = (void *) args[2];
		snd_mixer_selem_channel_id_t channel = tswap32(args[3]);
		assign_result32(args[1],
			snd_mixer_selem_has_capture_channel(obj, channel));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_CAPTURE_SWITCH:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1],
			snd_mixer_selem_has_capture_switch(elem));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_CAPTURE_SWITCH_EXCLUSIVE:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1],
			snd_mixer_selem_has_capture_switch_exclusive(elem));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_CAPTURE_VOLUME:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1],
			snd_mixer_selem_has_capture_volume(elem));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_PLAYBACK_CHANNEL:
	{
		snd_mixer_elem_t *obj = (void *) args[2];
		snd_mixer_selem_channel_id_t channel = tswap32(args[3]);
		assign_result32(args[1],
			snd_mixer_selem_has_playback_channel(obj, channel));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_PLAYBACK_SWITCH:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1],
			snd_mixer_selem_has_playback_switch(elem));
		return 0;
	}
	case SND_MIXER_SELEM_HAS_PLAYBACK_VOLUME:
	{
		snd_mixer_elem_t *elem = (void *) args[2];
		assign_result32(args[1],
			snd_mixer_selem_has_playback_volume(elem));
		return 0;
	}
	case SND_MIXER_SELEM_REGISTER:
	{
		snd_mixer_t *mixer = (void *) args[2];
		struct snd_mixer_selem_regopt *options = (void *) args[3];
		if (options)
			warnx("snd_mixer_selem_register options is not null.");
		snd_mixer_class_t **classp = (void *) args[4];
		if (classp)
			warnx("snd_mixer_selem_register classp is not null.");
		assign_result32(args[1], snd_mixer_selem_register(mixer,
			options, classp));
		return 0;
	}
	case SND_OUTPUT_CLOSE:
	{
		void *output = (void *) args[2];
		assign_result32(args[1], snd_output_close(output));
		return 0;
	}
	case SND_OUTPUT_STDIO_ATTACH:
	{
		void *outputp = (void *) tswap32(args[2]);
		void *fp = (void *) tswap32(args[3]);
		int _close = tswap32(args[4]);
		assign_result32(args[1], snd_output_stdio_attach(outputp, fp,
			_close));
		return 0;
	}
	case SND_PCM_ACCESS_MASK_SIZEOF:
	{
		assign_result32(args[1], snd_pcm_access_mask_sizeof());
		return 0;
	}
	case SND_PCM_ACCESS_MASK_TEST:
	{
		const snd_pcm_access_mask_t *mask = (void *) tswap32(args[2]);
		snd_pcm_access_t val = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_access_mask_test(mask, val));
		return 0;
	}
	case SND_PCM_AVAIL_UPDATE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_avail_update(pcm));
		return 0;
	}
	case SND_PCM_BYTES_TO_FRAMES:
	{
		snd_pcm_t *pcm = (void *) args[2];
		ssize_t bytes = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_bytes_to_frames(pcm, bytes));
		return 0;
	}
	case SND_PCM_CLOSE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_close(pcm));
		return 0;
	}
	case SND_PCM_DELAY:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sframes_t *target_delayp = (void *) tswap32(args[3]);
		snd_pcm_sframes_t delayp;
		assign_result32(args[1], snd_pcm_delay(pcm, &delayp));
		*target_delayp = tswap32(delayp);
		return 0;
	}
	case SND_PCM_DRAIN:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_drain(pcm));
		return 0;
	}
	case SND_PCM_DROP:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_drop(pcm));
		return 0;
	}
	case SND_PCM_FORMAT_DESCRIPTION:
	{
		const snd_pcm_format_t format = tswap32(args[2]);
		assign_result32(args[1], snd_pcm_format_description(format));
		return 0;
	}
	case SND_PCM_FORMAT_MASK_SIZEOF:
	{
		assign_result32(args[1], snd_pcm_format_mask_sizeof());
		return 0;
	}
	case SND_PCM_FORMAT_MASK_TEST:
	{
		const snd_pcm_format_mask_t *mask = (void *) tswap32(args[2]);
		snd_pcm_format_t val = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_format_mask_test(mask, val));
		return 0;
	}
	case SND_PCM_FORMAT_NAME:
	{
		const snd_pcm_format_t format = tswap32(args[2]);
		assign_result32(args[1], snd_pcm_format_name(format));
		return 0;
	}
	case SND_PCM_FORMAT_PHYSICAL_WIDTH:
	{
		snd_pcm_format_t format = tswap32(args[2]);
		assign_result32(args[1],
			snd_pcm_format_physical_width(format));
		return 0;
	}
	case SND_PCM_FORMAT_SET_SILENCE:
	{
		snd_pcm_format_t format = tswap32(args[2]);
		void *buf = (void *) tswap32(args[3]);
		unsigned int samples = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_format_set_silence(format, buf,
			samples));
		return 0;
	}
	case SND_PCM_FRAMES_TO_BYTES:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sframes_t frames = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_frames_to_bytes(pcm, frames));
		return 0;
	}
	case SND_PCM_HW_PARAMS:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_pcm_hw_params(pcm, params));
		return 0;
	}
	case SND_PCM_HW_PARAMS_ANY:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_pcm_hw_params_any(pcm, params));
		return 0;
	}
	case SND_PCM_HW_PARAMS_FREE:
	{
		snd_pcm_hw_params_t *obj = (void *) tswap32(args[1]);
		snd_pcm_hw_params_free(obj);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_ACCESS_MASK:
	{
		snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		snd_pcm_access_mask_t *mask = (void *) tswap32(args[3]);
		assign_result32(args[1],
			snd_pcm_hw_params_get_access_mask(params, mask));
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_BUFFER_SIZE:
	{
		void *params = (void *) tswap32(args[2]);
#ifdef BSWAP_NEEDED
		snd_pcm_uframes_t *target_val = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val;
		assign_result32(args[1],
			snd_pcm_hw_params_get_buffer_size(params, &val));
		*target_val = tswap32(val);
#else
		void *val = (void *) args[3];
		assign_result32(args[1],
			snd_pcm_hw_params_get_buffer_size(params, val));
#endif
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_BUFFER_TIME_MAX:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		int *target_dir = (void *) tswap32(args[4]);
		int dir;
		assign_result32(args[1],
			snd_pcm_hw_params_get_buffer_time_max(params, &val,
			&dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_CHANNELS_MAX:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		assign_result32(args[1],
			snd_pcm_hw_params_get_channels_max(params, &val));
		*target_val = tswap32(val);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_CHANNELS_MIN:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		assign_result32(args[1],
			snd_pcm_hw_params_get_channels_min(params, &val));
		*target_val = tswap32(val);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_FORMAT_MASK:
	{
		snd_pcm_hw_params_t *params = (void *) tswap32(args[1]);
		snd_pcm_format_mask_t *mask = (void *) tswap32(args[2]);
		snd_pcm_hw_params_get_format_mask(params, mask);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_PERIOD_SIZE:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		snd_pcm_uframes_t *target_frames = (void *) tswap32(args[3]);
		snd_pcm_uframes_t frames;
		int *target_dir = (void *) tswap32(args[4]);
		int dir;
		assign_result32(args[1],
			snd_pcm_hw_params_get_period_size(params, &frames,
			&dir));
		*target_frames = tswap32(frames);
		if (target_dir)
			*target_dir = tswap32(dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_PERIOD_TIME:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		int *target_dir = (void *) tswap32(args[4]);
		int dir;
		assign_result32(args[1],
			snd_pcm_hw_params_get_period_time(params, &val, &dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_RATE_MAX:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		int *target_dir = (void *) tswap32(args[4]);
		int dir;
		assign_result32(args[1], snd_pcm_hw_params_get_rate_max(params,
			&val, &dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_GET_RATE_MIN:
	{
		const snd_pcm_hw_params_t *params = (void *) tswap32(args[2]);
		unsigned int *target_val = (void *) tswap32(args[3]);
		unsigned int val;
		int *target_dir = (void *) tswap32(args[4]);
		int dir;
		assign_result32(args[1], snd_pcm_hw_params_get_rate_min(params,
			&val, &dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_MALLOC:
	{
		snd_pcm_hw_params_t **target_ptr = (void *) tswap32(args[2]);
		snd_pcm_hw_params_t *ptr;
		assign_result32(args[1], snd_pcm_hw_params_malloc(&ptr));
		*target_ptr = tswap32(ptr);
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_ACCESS:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_access_t _access = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_hw_params_set_access(pcm,
			params, _access));
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_BUFFER_TIME_NEAR:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		unsigned int *target_val = (void *) tswap32(args[4]);
		unsigned int val = tswap32(*target_val);
		int *target_dir = (void *) tswap32(args[5]);
		int *dir;
		if (target_dir) {
			dir = alloca(sizeof(int));
			*dir = tswap32(*target_dir);
		}
		else
			dir = NULL;
		assign_result32(args[1],
			snd_pcm_hw_params_set_buffer_time_near(pcm, params,
			&val, dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(*dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_CHANNELS:
	{
		void *pcm = (void *) args[2];
		void *params = (void *) tswap32(args[3]);
		unsigned int val = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_hw_params_set_channels(pcm,
			params, val));
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_FORMAT:
	{
		void *pcm = (void *) args[2];
		void *params = (void *) tswap32(args[3]);
		snd_pcm_format_t val = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_hw_params_set_format(pcm,
			params, val));
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_PERIOD_TIME_NEAR:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		unsigned int *target_val = (void *) tswap32(args[4]);
		unsigned int val = tswap32(*target_val);
		int *target_dir = (void *) tswap32(args[5]);
		int *dir;
		if (target_dir) {
			dir = alloca(sizeof(int));
			*dir = tswap32(*target_dir);
		}
		else
			dir = NULL;
		assign_result32(args[1],
			snd_pcm_hw_params_set_period_time_near(pcm, params,
			&val, dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(*dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_SET_RATE_NEAR:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_hw_params_t *params = (void *) tswap32(args[3]);
		unsigned int *target_val = (void *) tswap32(args[4]);
		unsigned int val = tswap32(*target_val);
		int *target_dir = (void *) tswap32(args[5]);
		int *dir;
		if (target_dir) {
			dir = alloca(sizeof(int));
			*dir = tswap32(*target_dir);
		}
		else
			dir = NULL;
		assign_result32(args[1], snd_pcm_hw_params_set_rate_near(pcm,
			params, &val, dir));
		*target_val = tswap32(val);
		if (target_dir)
			*target_dir = tswap32(*dir);
		return 0;
	}
	case SND_PCM_HW_PARAMS_SIZEOF:
	{
		assign_result32(args[1], snd_pcm_hw_params_sizeof());
		return 0;
	}
	case SND_PCM_INFO:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_info_t *info = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_pcm_info(pcm, info));
		return 0;
	}
	case SND_PCM_INFO_GET_ID:
	{
		const snd_pcm_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_pcm_info_get_id(obj));
		return 0;
	}
	case SND_PCM_INFO_GET_NAME:
	{
		const snd_pcm_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_pcm_info_get_name(obj));
		return 0;
	}
	case SND_PCM_INFO_GET_SUBDEVICE_NAME:
	{
		const snd_pcm_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_pcm_info_get_subdevice_name(obj));
		return 0;
	}
	case SND_PCM_INFO_GET_SUBDEVICES_AVAIL:
	{
		const snd_pcm_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1],
			snd_pcm_info_get_subdevices_avail(obj));
		return 0;
	}
	case SND_PCM_INFO_GET_SUBDEVICES_COUNT:
	{
		const snd_pcm_info_t *obj = (void *) tswap32(args[2]);
		assign_result32(args[1],
			snd_pcm_info_get_subdevices_count(obj));
		return 0;
	}
	case SND_PCM_INFO_SET_DEVICE:
	{
		snd_pcm_info_t *obj = (void *) tswap32(args[1]);
		unsigned int val = tswap32(args[2]);
		snd_pcm_info_set_device(obj, val);
		return 0;
	}
	case SND_PCM_INFO_SET_STREAM:
	{
		snd_pcm_info_t *obj = (void *) tswap32(args[1]);
		snd_pcm_stream_t val = tswap32(args[2]);
		snd_pcm_info_set_stream(obj, val);
		return 0;
	}
	case SND_PCM_INFO_SET_SUBDEVICE:
	{
		snd_pcm_info_t *obj = (void *) tswap32(args[1]);
		unsigned int val = tswap32(args[2]);
		snd_pcm_info_set_subdevice(obj, val);
		return 0;
	}
	case SND_PCM_INFO_SIZEOF:
	{
		assign_result32(args[1], snd_pcm_info_sizeof());
		return 0;
	}
	case SND_PCM_MMAP_WRITEI:
	{
		snd_pcm_t *pcm = (void *) args[2];
		const void *buffer = (void *) tswap32(args[3]);
		snd_pcm_uframes_t size = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_mmap_writei(pcm, buffer,
			size));
		return 0;
	}
	case SND_PCM_NONBLOCK__FUNCTION:
	{
		snd_pcm_t *pcm = (void *) args[2];
		int nonblock = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_nonblock(pcm, nonblock));
		return 0;
	}
	case SND_PCM_OPEN:
	{
		snd_pcm_t **pcm = (void *) tswap32(args[2]);
		const char *name = (void *) tswap32(args[3]);
		snd_pcm_stream_t stream = tswap32(args[4]);
		int mode = tswap32(args[5]);
		assign_result32(args[1], snd_pcm_open(pcm, name, stream, mode));
		return 0;
	}
	case SND_PCM_PREPARE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_prepare(pcm));
		return 0;
	}
	case SND_PCM_READI:
	{
		snd_pcm_t *pcm = (void *) args[2];
		void *buffer = (void *) tswap32(args[3]);
		snd_pcm_uframes_t size = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_readi(pcm, buffer, size));
		return 0;
	}
	case SND_PCM_START:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_start(pcm));
		return 0;
	}
	case SND_PCM_STATE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		assign_result32(args[1], snd_pcm_state(pcm));
		return 0;
	}
	case SND_PCM_STREAM_NAME:
	{
		const snd_pcm_stream_t stream = tswap32(args[2]);
		assign_result32(args[1], snd_pcm_stream_name(stream));
		return 0;
	}
	case SND_PCM_SW_PARAMS:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_pcm_sw_params(pcm, params));
		return 0;
	}
	case SND_PCM_SW_PARAMS_CURRENT:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_pcm_sw_params_current(pcm,
			params));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_AVAIL_MIN:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_sw_params_set_avail_min(pcm,
			params, val));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_SILENCE_SIZE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_sw_params_set_silence_size(pcm,
			params, val));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_SILENCE_THRESHOLD:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val = tswap32(args[4]);
		assign_result32(args[1],
			snd_pcm_sw_params_set_silence_threshold(pcm, params,
			val));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_START_THRESHOLD:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val = tswap32(args[4]);
		assign_result32(args[1],
			snd_pcm_sw_params_set_start_threshold(pcm, params,
			val));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_STOP_THRESHOLD:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_uframes_t val = tswap32(args[4]);
		assign_result32(args[1],
			snd_pcm_sw_params_set_stop_threshold(pcm, params, val));
		return 0;
	}
	case SND_PCM_SW_PARAMS_SET_XRUN_MODE:
	{
		snd_pcm_t *pcm = (void *) args[2];
		snd_pcm_sw_params_t *params = (void *) tswap32(args[3]);
		snd_pcm_xrun_t val = tswap32(args[4]);
		snd_pcm_uframes_t threshold;
		int ret = 0;
		switch (val) {
		case SND_PCM_XRUN_NONE:
		{
			snd_pcm_sw_params_t *temp_params;
			snd_pcm_sw_params_alloca(&temp_params);
			ret = snd_pcm_sw_params_current(pcm, temp_params);
			if (ret)
				break;
			ret = snd_pcm_sw_params_get_boundary(temp_params,
				&threshold);
			break;
		}
		case SND_PCM_XRUN_STOP:
		{
			snd_pcm_hw_params_t *temp_params;
			snd_pcm_hw_params_alloca(&temp_params);
			ret = snd_pcm_hw_params_current(pcm, temp_params);
			if (ret)
				break;
			ret = snd_pcm_hw_params_get_buffer_size(temp_params,
				&threshold);
			break;
		}
		default:
			ret = -EINVAL;
		}
		if (!ret)
			ret = snd_pcm_sw_params_set_stop_threshold(pcm, params,
				threshold);
		assign_result32(args[1], ret);
		return 0;
	}
	case SND_PCM_SW_PARAMS_SIZEOF:
	{
		assign_result32(args[1], snd_pcm_sw_params_sizeof());
		return 0;
	}
	case SND_PCM_WAIT:
	{
		snd_pcm_t *pcm = (void *) args[2];
		int timeout = tswap32(args[3]);
		assign_result32(args[1], snd_pcm_wait(pcm, timeout));
		return 0;
	}
	case SND_PCM_WRITEI:
	{
		snd_pcm_t *pcm = (void *) args[2];
		const void *buffer = (void *) tswap32(args[3]);
		snd_pcm_uframes_t size = tswap32(args[4]);
		assign_result32(args[1], snd_pcm_writei(pcm, buffer, size));
		return 0;
	}
	case SND_SEQ_CLIENT_INFO_GET_CLIENT:
	{
		const snd_seq_client_info_t *info = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_seq_client_info_get_client(info));
		return 0;
	}
	case SND_SEQ_CLIENT_INFO_GET_NAME:
	{
		snd_seq_client_info_t *info = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_seq_client_info_get_name(info));
		return 0;
	}
	case SND_SEQ_CLIENT_INFO_SET_CLIENT:
	{
		snd_seq_client_info_t *info = (void *) tswap32(args[1]);
		int client = tswap32(args[2]);
		snd_seq_client_info_set_client(info, client);
		return 0;
	}
	case SND_SEQ_CLIENT_INFO_SIZEOF:
	{
		assign_result32(args[1], snd_seq_client_info_sizeof());
		return 0;
	}
	case SND_SEQ_CLOSE:
	{
		snd_seq_t *handle = (void *) args[2];
		assign_result32(args[1], snd_seq_close(handle));
		return 0;
	}
	case SND_SEQ_DELETE_SIMPLE_PORT:
	{
		snd_seq_t *seq = (void *) args[2];
		int port = tswap32(args[3]);
		assign_result32(args[1], snd_seq_delete_simple_port(seq, port));
		return 0;
	}
	case SND_SEQ_OPEN:
	{
		snd_seq_t **handle = (void *) tswap32(args[2]);
		const char *name = (void *) tswap32(args[3]);
		int streams = tswap32(args[4]);
		int mode = tswap32(args[5]);
		assign_result32(args[1], snd_seq_open(handle, name, streams,
			mode));
		return 0;
	}
	case SND_SEQ_PORT_INFO_GET_ADDR:
	{
		const snd_seq_port_info_t *info = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_seq_port_info_get_addr(info));
		return 0;
	}
	case SND_SEQ_PORT_INFO_GET_CAPABILITY:
	{
		const snd_seq_port_info_t *info = (void *) tswap32(args[2]);
		assign_result32(args[1],
			snd_seq_port_info_get_capability(info));
		return 0;
	}
	case SND_SEQ_PORT_INFO_GET_TYPE:
	{
		const snd_seq_port_info_t *info = (void *) tswap32(args[2]);
		assign_result32(args[1], snd_seq_port_info_get_type(info));
		return 0;
	}
	case SND_SEQ_PORT_INFO_SET_CLIENT:
	{
		snd_seq_port_info_t *info = (void *) tswap32(args[1]);
		int client = tswap32(args[2]);
		snd_seq_port_info_set_client(info, client);
		return 0;
	}
	case SND_SEQ_PORT_INFO_SET_PORT:
	{
		snd_seq_port_info_t *info = (void *) tswap32(args[1]);
		int port = tswap32(args[2]);
		snd_seq_port_info_set_port(info, port);
		return 0;
	}
	case SND_SEQ_PORT_INFO_SIZEOF:
	{
		assign_result32(args[1], snd_seq_port_info_sizeof());
		return 0;
	}
	case SND_SEQ_QUERY_NEXT_CLIENT:
	{
		snd_seq_t *handle = (void *) args[2];
		snd_seq_client_info_t *info = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_seq_query_next_client(handle,
			info));
		return 0;
	}
	case SND_SEQ_QUERY_NEXT_PORT:
	{
		snd_seq_t *handle = (void *) args[2];
		snd_seq_port_info_t *info = (void *) tswap32(args[3]);
		assign_result32(args[1], snd_seq_query_next_port(handle, info));
		return 0;
	}
	case SND_STRERROR:
	{
		int errnum = tswap32(args[2]);
		assign_result32(args[1], snd_strerror(errnum));
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
