#include <err.h>
#include <stdlib.h>
#include <X11/extensions/Xrender.h>

#include "libX11.h"
#include "libXrender_const.h"

typedef struct {
	struct formats_info_pair *pair;
	size_t size;
} formats_info;

typedef struct formats_info_pair {
	const XRenderPictFormat *host;
	XRenderPictFormat *target;
} formats_info_pair;

static inline void swap_format(XRenderPictFormat *dest,
	const XRenderPictFormat *src);

static int formats_info_compar(const void *key, const void *member) {
	const XRenderPictFormat *host = key;
	const formats_info_pair *pair = member;
	return host < pair->host ? -1
		: host > pair->host ? 1
		: 0;
}

static void free_formats(void *data) {
warnx("freeing formats");
	formats_info *info = data;
	size_t i;
	for (i = 0; i < info->size; i++)
		free(info->pair[i].target);
	free(info->pair);
	free(info);
}

static XRenderPictFormat *get_target_format(DisplayProxy *proxy,
	const XRenderPictFormat *format) {
	if (!format)
		return NULL;
	ProxyExtension *extension = proxy->target_ext;
	while (extension && extension->free_data != free_formats)
		extension = extension->next;
	if (!extension) {
		extension = calloc(1, sizeof(ProxyExtension));
		extension->next = proxy->target_ext;
		extension->free_data = free_formats;
		formats_info_pair *pair = malloc(sizeof(formats_info_pair));
		pair->host = format;
		XRenderPictFormat *target_format
			= malloc(sizeof(XRenderPictFormat));
		swap_format(target_format, format);
		pair->target = target_format;
		formats_info *info = malloc(sizeof(formats_info));
		info->pair = pair;
		info->size = 1;
		extension->data = info;
		proxy->target_ext = extension;
		return target_format;
	}
	formats_info *info = extension->data;
	formats_info_pair *pair = bsearch(format, info->pair, info->size,
		sizeof(formats_info_pair), formats_info_compar);
	if (pair)
		return pair->target;
	info->pair = realloc(info->pair, ++(info->size)
		* sizeof(formats_info_pair));
	int i;
	for (i = info->size - 1; i && info->pair[i - 1].host > format; i--)
		info->pair[i] = info->pair[i - 1];
	info->pair[i].host = format;
	XRenderPictFormat *target_format = malloc(sizeof(XRenderPictFormat));
	swap_format(target_format, format);
	info->pair[i].target = target_format;
	return info->pair[i].target;
}

static inline void swap_attributes(XRenderPictureAttributes *dest,
	const XRenderPictureAttributes *src) {
	dest->repeat = tswap32(src->repeat);
	dest->alpha_map = src->alpha_map;
	memswap32(&dest->alpha_x_origin, &src->alpha_x_origin, 11);
}

static inline void swap_format(XRenderPictFormat *dest,
	const XRenderPictFormat *src) {
	memswap32(dest, src, 3);
	memswap16(&dest->direct, &src->direct, size16of(XRenderDirectFormat));
	dest->colormap = tswap32(src->colormap);
}

int handle_syscall(unsigned args[],
	__attribute__((unused)) struct emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case XRENDERADDGLYPHS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		GlyphSet glyphset = args[2];
		const Glyph *target_gids = (void *) tswap32(args[3]);
		const XGlyphInfo *target_glyphs = (void *) tswap32(args[4]);
		int nglyphs = tswap32(args[5]);
		Glyph *gids = malloc(nglyphs * sizeof(Glyph));
		memswap32(gids, target_gids, nglyphs);
		XGlyphInfo *glyphs = malloc(nglyphs * sizeof(XGlyphInfo));
		memswap16(glyphs, target_glyphs, nglyphs
			* size16of(XGlyphInfo));
		const char *images = (void *) tswap32(args[6]);
		int nbyte_images = tswap32(args[7]);
		XRenderAddGlyphs(dpy, glyphset, gids, glyphs, nglyphs, images,
			nbyte_images);
		free(gids);
		free(glyphs);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCHANGEPICTURE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		Picture picture = args[2];
		unsigned long valuemask = tswap32(args[3]);
		const XRenderPictureAttributes *target_attributes
			= (void *) tswap32(args[4]);
		XRenderPictureAttributes attributes;
		swap_attributes(&attributes, target_attributes);
		XRenderChangePicture(dpy, picture, valuemask, &attributes);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCOMPOSITE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		int op = tswap32(args[2]);
		Picture src = args[3];
		Picture mask = args[4];
		Picture dst = args[5];
		int src_x = tswap32(args[6]);
		int src_y = tswap32(args[7]);
		int mask_x = tswap32(args[8]);
		int mask_y = tswap32(args[9]);
		int dst_x = tswap32(args[10]);
		int dst_y = tswap32(args[11]);
		unsigned int width = tswap32(args[12]);
		unsigned int height = tswap32(args[13]);
		XRenderComposite(dpy, op, src, mask, dst, src_x, src_y, mask_x,
			mask_y, dst_x, dst_y, width, height);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCOMPOSITETEXT16:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		int op = tswap32(args[2]);
		Picture src = args[3];
		Picture dst = args[4];
		const XRenderPictFormat *target_maskFormat
			= (void *) tswap32(args[5]);
		XRenderPictFormat maskFormat;
		swap_format(&maskFormat, target_maskFormat);
		int xSrc = tswap32(args[6]);
		int ySrc = tswap32(args[7]);
		int xDst = tswap32(args[8]);
		int yDst = tswap32(args[9]);
		const XGlyphElt16 *target_elts = (void *) tswap32(args[10]);
		int nelt = tswap32(args[11]);
		XGlyphElt16 *elts = malloc(nelt * sizeof(XGlyphElt16));
		int i;
		for (i = 0; i < nelt; i++) {
			elts[i].glyphset = target_elts[i].glyphset;
			const unsigned short *target_chars
				= tswap32(target_elts[i].chars);
			memswap32(&elts[i].nchars, &target_elts[i].nchars, 3);
			unsigned short *chars = malloc(elts[i].nchars
				* sizeof(unsigned short));
			memswap16(chars, target_chars, elts[i].nchars);
			elts[i].chars = chars;
		}
		XRenderCompositeText16(dpy, op, src, dst, &maskFormat, xSrc,
			ySrc, xDst, yDst, elts, nelt);
		for (i = 0; i < nelt; i++)
			free((void *) elts[i].chars);
		free(elts);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCOMPOSITETEXT32:
	{
		Display *dpy = get_display(args[1]);
		int op = tswap32(args[2]);
		Picture src = args[3];
		Picture dst = args[4];
		const XRenderPictFormat *target_maskFormat
			= (void *) tswap32(args[5]);
		XRenderPictFormat maskFormat;
		swap_format(&maskFormat, target_maskFormat);
		int xSrc = tswap32(args[6]);
		int ySrc = tswap32(args[7]);
		int xDst = tswap32(args[8]);
		int yDst = tswap32(args[9]);
		const XGlyphElt32 *target_elts = (void *) tswap32(args[10]);
		int nelt = tswap32(args[11]);
		XGlyphElt32 *elts = malloc(nelt * sizeof(XGlyphElt32));
		int i;
		for (i = 0; i < nelt; i++) {
			elts[i].glyphset = target_elts[i].glyphset;
			const unsigned int *target_chars
				= tswap32(target_elts[i].chars);
			memswap32(&elts[i].nchars, &target_elts[i].nchars, 3);
			unsigned int *chars = malloc(elts[i].nchars
				* sizeof(unsigned int));
			memswap32(chars, target_chars, elts[i].nchars);
			elts[i].chars = chars;
		}
warnx("before xrender-4 %ld", ((_XPrivDisplay) dpy)->request);
		XRenderCompositeText32(dpy, op, src, dst, &maskFormat, xSrc,
			ySrc, xDst, yDst, elts, nelt);
warnx("after xrender-4 %ld", ((_XPrivDisplay) dpy)->request);
		for (i = 0; i < nelt; i++)
			free((void *) elts[i].chars);
		free(elts);
		return 0;
	}
	case XRENDERCOMPOSITETEXT8:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		int op = tswap32(args[2]);
		Picture src = args[3];
		Picture dst = args[4];
		const XRenderPictFormat *target_maskFormat
			= (void *) tswap32(args[5]);
		XRenderPictFormat maskFormat;
		swap_format(&maskFormat, target_maskFormat);
		int xSrc = tswap32(args[6]);
		int ySrc = tswap32(args[7]);
		int xDst = tswap32(args[8]);
		int yDst = tswap32(args[9]);
		const XGlyphElt8 *target_elts = (void *) tswap32(args[10]);
		int nelt = tswap32(args[11]);
		XGlyphElt8 *elts = malloc(nelt * sizeof(XGlyphElt8));
		int i;
		for (i = 0; i < nelt; i++) {
			elts[i].glyphset = target_elts[i].glyphset;
			memswap32(&elts[i].chars, &target_elts[i].chars, 4);
		}
		XRenderCompositeText8(dpy, op, src, dst, &maskFormat, xSrc,
			ySrc, xDst, yDst, elts, nelt);
		free(elts);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCOMPOSITETRAPEZOIDS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		int op = tswap32(args[2]);
		Picture src = args[3];
		Picture dst = args[4];
		const XRenderPictFormat *target_maskFormat
			= (void *) tswap32(args[5]);
		XRenderPictFormat maskFormat;
		swap_format(&maskFormat, target_maskFormat);
		int xSrc = tswap32(args[6]);
		int ySrc = tswap32(args[7]);
		const XTrapezoid *target_traps = (void *) tswap32(args[8]);
		int ntrap = tswap32(args[9]);
		XTrapezoid *traps = malloc(ntrap * sizeof(XTrapezoid));
		memswap32(traps, target_traps, ntrap * size32of(XTrapezoid));
		XRenderCompositeTrapezoids(dpy, op, src, dst, &maskFormat, xSrc,
			ySrc, traps, ntrap);
		free(traps);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCREATEGLYPHSET:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		const XRenderPictFormat *target_format
			= (void *) tswap32(args[3]);
		XRenderPictFormat format;
		swap_format(&format, target_format);
		GlyphSet *result = (void *) tswap32(args[1]);
		*result = XRenderCreateGlyphSet(dpy, &format);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERCREATEPICTURE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Drawable drawable = tswap32(args[3]);
		const XRenderPictFormat *target_format
			= (void *) tswap32(args[4]);
		XRenderPictFormat format;
		swap_format(&format, target_format);
		unsigned long valuemask = tswap32(args[5]);
		const XRenderPictureAttributes *target_attributes
			= (void *) tswap32(args[6]);
		XRenderPictureAttributes *attributes;
		if (target_attributes) {
			attributes = alloca(sizeof(XRenderPictureAttributes));
			swap_attributes(attributes, target_attributes);
		}
		else
			attributes = NULL;
		Picture *result = (void *) tswap32(args[1]);
		*result = XRenderCreatePicture(dpy, drawable, &format,
			valuemask, attributes);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERFILLRECTANGLES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		int op = tswap32(args[2]);
		Picture dst = args[3];
		const XRenderColor *target_color = (void *) tswap32(args[4]);
		XRenderColor color;
		memswap16(&color, target_color, size16of(XRenderColor));
		const XRectangle *target_rectangles = (void *) tswap32(args[5]);
		int n_rects = tswap32(args[6]);
		XRectangle *rectangles = malloc(n_rects * sizeof(XRectangle));
		memswap16(rectangles, target_rectangles, n_rects
			* size16of(XRectangle));
		XRenderFillRectangles(dpy, op, dst, &color, rectangles,
			n_rects);
		free(rectangles);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERFINDFORMAT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned long mask = tswap32(args[3]);
		const XRenderPictFormat *target_templ
			= (void *) tswap32(args[4]);
		XRenderPictFormat templ;
		swap_format(&templ, target_templ);
		int count = tswap32(args[5]);
		XRenderPictFormat *format = XRenderFindFormat(dpy, mask, &templ,
			count);
		assign_result32(args[1], get_target_format(proxy, format));
		return 0;
	}
	case XRENDERFINDSTANDARDFORMAT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int format = tswap32(args[3]);
		XRenderPictFormat *pict_format = XRenderFindStandardFormat(dpy,
			format);
		assign_result32(args[1], get_target_format(proxy, pict_format));
		return 0;
	}
	case XRENDERFINDVISUALFORMAT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		const Visual *visual = get_visual(args[3]);
		XRenderPictFormat *format = XRenderFindVisualFormat(dpy,
			visual);
		assign_result32(args[1], get_target_format(proxy, format));
		return 0;
	}
	case XRENDERFREEGLYPHSET:
	{
		Display *dpy = get_display(args[1]);
		GlyphSet glyphset = args[2];
warnx("before xrender-13 %ld", ((_XPrivDisplay) dpy)->request);
		XRenderFreeGlyphSet(dpy, glyphset);
warnx("after xrender-13 %ld", ((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERFREEPICTURE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		Picture picture = args[2];
		XRenderFreePicture(dpy, picture);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRENDERQUERYEXTENSION:
	{
		Display *dpy = get_display(args[2]);
		int *target_event_basep = (void *) tswap32(args[3]);
		int event_basep;
		int *target_error_basep = (void *) tswap32(args[4]);
		int error_basep;
		assign_result32(args[1], XRenderQueryExtension(dpy,
			&event_basep, &error_basep));
		*target_event_basep = tswap32(event_basep);
		*target_error_basep = tswap32(error_basep);
		return 0;
	}
	case XRENDERQUERYVERSION:
	{
		Display *dpy = get_display(args[2]);
		int *target_major_versionp = (void *) tswap32(args[3]);
		int major_versionp;
		int *target_minor_versionp = (void *) tswap32(args[4]);
		int minor_versionp;
		assign_result32(args[1], XRenderQueryVersion(dpy,
			&major_versionp, &minor_versionp));
		*target_major_versionp = tswap32(major_versionp);
		*target_minor_versionp = tswap32(minor_versionp);
		return 0;
	}
	case XRENDERSETPICTURECLIPRECTANGLES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		Picture picture = args[2];
		int xOrigin = tswap32(args[3]);
		int yOrigin = tswap32(args[4]);
		const XRectangle *target_rects = (void *) tswap32(args[5]);
		int n = tswap32(args[6]);
		XRectangle *rects = malloc(n * sizeof(XRectangle));
		memswap16(rects, target_rects, n * size16of(XRectangle));
		XRenderSetPictureClipRectangles(dpy, picture, xOrigin, yOrigin,
			rects, n);
		free(rects);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
