#include <dlfcn.h>
#include <err.h>
#include <stdlib.h>
#include <emu-stub/handler.h>
#include <X11/extensions/Xrandr.h>

#include "libX11.h"
#include "libXrandr_const.h"

typedef struct {
	XRRCrtcInfo base;
	XRRCrtcInfo *native;
} XRRCrtcInfoProxy;

typedef struct {
	XRROutputInfo base;
	XRROutputInfo *native;
} XRROutputInfoProxy;

typedef struct {
	XRRScreenResources base;
	XRRScreenResources *native;
} XRRScreenResourcesProxy;

static void (*target_to_host_event)(XEvent *host, const XEvent *target);

__attribute__((constructor)) static void init_extern_symbols(void) {
	static void *handle;
	handle = dlopen("$ORIGIN/libX11.so.6", RTLD_LAZY);
	target_to_host_event = dlsym(handle, "target_to_host_event");

	__attribute__((destructor)) void destructor(void) {
		dlclose(handle);
	}
}

static XRRCrtcInfoProxy *create_crtc_proxy(XRRCrtcInfo *crtcInfo) {
	if (!crtcInfo)
		return NULL;
	int noutput = crtcInfo->noutput;
	int npossible = crtcInfo->npossible;
	XRRCrtcInfoProxy *proxy = malloc(sizeof(XRRCrtcInfoProxy)
		+ noutput * sizeof(RROutput) + npossible * sizeof(RROutput));
	memswap32(&proxy->base, crtcInfo, 6);
	proxy->base.rotation = tswap16(crtcInfo->rotation);
	proxy->base.noutput = tswap32(noutput);
	RROutput *outputs = (RROutput *) (proxy + 1);
	proxy->base.outputs = tswap32(outputs);
	memswap32(outputs, crtcInfo->outputs, noutput);
	proxy->base.rotations = tswap16(crtcInfo->rotations);
	proxy->base.npossible = tswap32(npossible);
	RROutput *possible = (RROutput *) (outputs + noutput);
	proxy->base.possible = tswap32(possible);
	memswap32(possible, crtcInfo->possible, npossible);
	proxy->native = crtcInfo;
	return proxy;
}

static XRROutputInfoProxy *create_output_proxy(XRROutputInfo *outputInfo) {
	if (!outputInfo)
		return NULL;
	int ncrtc = outputInfo->ncrtc;
	int nclone = outputInfo->nclone;
	int nmode = outputInfo->nmode;
	XRROutputInfoProxy *proxy = malloc(sizeof(XRROutputInfoProxy)
		+ ncrtc * sizeof(RRCrtc) + nclone * sizeof(RROutput)
		+ nmode * sizeof(RRMode));
	memswap32(&proxy->base, outputInfo, 6);
	memswap16(&proxy->base.connection, &outputInfo->connection, 2);
	proxy->base.ncrtc = tswap32(ncrtc);
	RRCrtc *crtcs = (RRCrtc *) (proxy + 1);
	proxy->base.crtcs = tswap32(crtcs);
	memswap32(crtcs, outputInfo->crtcs, ncrtc);
	proxy->base.nclone = tswap32(nclone);
	RROutput *clones = (RROutput *) (crtcs + ncrtc);
	proxy->base.clones = tswap32(clones);
	memswap32(clones, outputInfo->clones, nclone);
	proxy->base.nmode = tswap32(nmode);
	proxy->base.npreferred = tswap32(outputInfo->npreferred);
	RRMode *modes = (RRMode *) (clones + nclone);
	proxy->base.modes = tswap32(modes);
	memswap32(modes, outputInfo->modes, nmode);
	proxy->native = outputInfo;
	return proxy;
}

static XRRScreenResourcesProxy *create_resources_proxy(
	XRRScreenResources *resources) {
	if (!resources)
		return NULL;
	int ncrtc = resources->ncrtc;
	int noutput = resources->noutput;
	int nmode = resources->nmode;
	XRRScreenResourcesProxy *proxy = malloc(sizeof(XRRScreenResourcesProxy)
		+ ncrtc * sizeof(RRCrtc) + noutput * sizeof(RROutput)
		+ nmode * sizeof(XRRModeInfo));
	memswap32(&proxy->base, resources, 3);
	RRCrtc *crtcs = (RRCrtc *) (proxy + 1);
	proxy->base.crtcs = tswap32(crtcs);
	memswap32(crtcs, resources->crtcs, ncrtc);
	proxy->base.noutput = tswap32(noutput);
	RROutput *outputs = (RROutput *) (crtcs + ncrtc);
	proxy->base.outputs = tswap32(outputs);
	memswap32(outputs, resources->outputs, noutput);
	proxy->base.nmode = tswap32(nmode);
	XRRModeInfo *modes = (XRRModeInfo *) (outputs + noutput);
	proxy->base.modes = tswap32(modes);
	memswap32(modes, resources->modes, nmode * size32of(XRRModeInfo));
	proxy->native = resources;
	return proxy;
}

static void free_event_base(void *data) {
warnx("freeing xrandr event base");
	free(data);
}

static int swap_event(XEvent *dest, const XEvent *src, const XEvent *host,
	void *data) {
	int *event_base = data;
	int type = src == host ? src->type : tswap32(src->type);
	switch (type - *event_base) {
	case RRScreenChangeNotify: {
		XRRScreenChangeNotifyEvent *d
			= (XRRScreenChangeNotifyEvent *) dest;
		const XRRScreenChangeNotifyEvent *s
			= (const XRRScreenChangeNotifyEvent *) src;
		memswap32(d, s, 8);
		memswap16(&d->size_index, &s->size_index, 3);
		memswap32(&d->width, &s->width, 4);
		break;
	}
	case RRNotify:
		memswap32(dest, src, size32of(XRRNotifyEvent));
		break;
	default:
		return 0;
	}
	return 1;
}

int handle_syscall(unsigned args[],
	__attribute__((unused)) struct emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case XRRFREECRTCINFO: {
		XRRCrtcInfoProxy *proxy = (void *) tswap32(args[1]);
		if (proxy) {
			XRRFreeCrtcInfo(proxy->native);
			free(proxy);
		}
		return 0;
	}
	case XRRFREEOUTPUTINFO: {
		XRROutputInfoProxy *proxy = (void *) tswap32(args[1]);
		if (proxy) {
			XRRFreeOutputInfo(proxy->native);
			free(proxy);
		}
		return 0;
	}
	case XRRFREESCREENRESOURCES: {
		XRRScreenResourcesProxy *proxy = (void *) tswap32(args[1]);
		if (proxy) {
			XRRFreeScreenResources(proxy->native);
			free(proxy);
		}
		return 0;
	}
	case XRRGETCRTCINFO: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XRRScreenResourcesProxy *resources = (void *) tswap32(args[3]);
		RRCrtc crtc = tswap32(args[4]);
		XRRCrtcInfo *crtcInfo = XRRGetCrtcInfo(dpy, resources->native,
			crtc);
		assign_result32(args[1], create_crtc_proxy(crtcInfo));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRGETOUTPUTINFO: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XRRScreenResourcesProxy *resources = (void *) tswap32(args[3]);
		RROutput output = tswap32(args[4]);
		XRROutputInfo *outputInfo = XRRGetOutputInfo(dpy,
			resources->native, output);
		assign_result32(args[1], create_output_proxy(outputInfo));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRGETOUTPUTPRIMARY: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Window window = tswap32(args[3]);
		assign_result32(args[1], XRRGetOutputPrimary(dpy, window));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRGETSCREENRESOURCESCURRENT: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Window window = tswap32(args[3]);
		XRRScreenResources *resources
			= XRRGetScreenResourcesCurrent(dpy, window);
		assign_result32(args[1], create_resources_proxy(resources));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRQUERYEXTENSION: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_event_base_return = (void *) tswap32(args[3]);
		int event_base_return;
		int *target_error_base_return = (void *) tswap32(args[4]);
		int error_base_return;
		assign_result32(args[1], XRRQueryExtension(dpy,
			&event_base_return, &error_base_return));
		*target_event_base_return = tswap32(event_base_return);
		*target_error_base_return = tswap32(error_base_return);

		ProxyExtension *extension = proxy->target_ext;
		while (extension && extension->free_data != free_event_base)
			extension = extension->next;
		if (!extension) {
			extension = calloc(1, sizeof(ProxyExtension));
			extension->next = proxy->target_ext;
			extension->free_data = free_event_base;
			int *event_base = malloc(sizeof(int));
			*event_base = event_base_return;
			extension->data = event_base;
			extension->swap_event = swap_event;
			proxy->target_ext = extension;
		}

		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRQUERYVERSION: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_major_version_return = (void *) tswap32(args[3]);
		int major_version_return;
		int *target_minor_version_return = (void *) tswap32(args[4]);
		int minor_version_return;
		assign_result32(args[1], XRRQueryVersion(dpy,
			&major_version_return, &minor_version_return));
		*target_major_version_return = tswap32(major_version_return);
		*target_minor_version_return = tswap32(minor_version_return);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRSELECTINPUT: {
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		Window window = tswap32(args[2]);
		int mask = tswap32(args[3]);
		XRRSelectInput(dpy, window, mask);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XRRUPDATECONFIGURATION:
	{
		XEvent *target_event = (void *) tswap32(args[2]);
		XEvent event;
		target_to_host_event(&event, target_event);
		assign_result32(args[1], XRRUpdateConfiguration(&event));
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
