#include <err.h>
#include <stdlib.h>
#include <string.h>
#include <X11/extensions/shape.h>
#include <X11/extensions/sync.h>
#include <X11/extensions/XShm.h>

#include "libX11.h"
#include "libXext_const.h"

static XImage *get_image(XImageProxy *proxy) {
	XImage *image = proxy->native;
	if (proxy->initial_sync) {
		proxy->initial_sync = 0;
		image->data = tswap32(proxy->data);
		image->byte_order = tswap32(proxy->byte_order);
		image->bitmap_bit_order = tswap32(proxy->bitmap_bit_order);
		memswap32(image->obdata, tswap32(proxy->obdata),
			size32of(XShmSegmentInfo));
	}
	return image;
}

int handle_syscall(unsigned args[],
	__attribute__((unused)) struct emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case DESTROY_IMAGE:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[2]);
		XImage *ximage = image_proxy->native;
		free(ximage->obdata);
		assign_result32(args[1], XDestroyImage(ximage));
		free(image_proxy);
		return 0;
	}
	case PUT_PIXEL:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[2]);
		XImage *ximage = get_image(image_proxy);
		int x = tswap32(args[3]);
		int y = tswap32(args[4]);
		unsigned long pixel = tswap32(args[5]);
		assign_result32(args[1], XPutPixel(ximage, x, y, pixel));
		return 0;
	}
	case XSHAPECOMBINEMASK:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window dest = tswap32(args[2]);
		int dest_kind = tswap32(args[3]);
		int x_off = tswap32(args[4]);
		int y_off = tswap32(args[5]);
		Pixmap src = tswap32(args[6]);
		int op = tswap32(args[7]);
		XShapeCombineMask(display, dest, dest_kind, x_off, y_off, src,
			op);
		proxy->request = tswap32(((_XPrivDisplay) display)->request);
		return 0;
	}
	case XSHAPECOMBINERECTANGLES: {
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window dest = tswap32(args[2]);
		int dest_kind = tswap32(args[3]);
		int x_off = tswap32(args[4]);
		int y_off = tswap32(args[5]);
		XRectangle *target_rectangles = (void *) tswap32(args[6]);
		int n_rects = tswap32(args[7]);
		XRectangle *rectangles = malloc(n_rects * sizeof(XRectangle));
		memswap16(rectangles, target_rectangles,
			n_rects * size16of(XRectangle));
		int op = tswap32(args[8]);
		int ordering = tswap32(args[9]);
		XShapeCombineRectangles(display, dest, dest_kind, x_off, y_off,
			rectangles, n_rects, op, ordering);
		free(rectangles);
		proxy->request = tswap32(((_XPrivDisplay) display)->request);
		return 0;
	}
	case XSHAPEQUERYEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		int *target_event_base = (void *) tswap32(args[3]);
		int event_base;
		int *target_error_base = (void *) tswap32(args[4]);
		int error_base;
		assign_result32(args[1], XShapeQueryExtension(display,
			&event_base, &error_base));
		*target_event_base = tswap32(event_base);
		*target_error_base = tswap32(error_base);
		proxy->request = tswap32(((_XPrivDisplay) display)->request);
		return 0;
	}
	case XSHAPEQUERYVERSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		int *target_major_version = (void *) tswap32(args[3]);
		int major_version;
		int *target_minor_version = (void *) tswap32(args[4]);
		int minor_version;
		assign_result32(args[1], XShapeQueryVersion(display,
			&major_version, &minor_version));
		*target_major_version = tswap32(major_version);
		*target_minor_version = tswap32(minor_version);
		proxy->request = tswap32(((_XPrivDisplay) display)->request);
		return 0;
	}
	case XSHMATTACH:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XShmSegmentInfo *target_shminfo = (void *) tswap32(args[3]);
		XShmSegmentInfo shminfo;
		memswap32(&shminfo, target_shminfo, size32of(XShmSegmentInfo));
		assign_result32(args[1], XShmAttach(dpy, &shminfo));
		target_shminfo->shmseg = tswap32(shminfo.shmseg);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSHMCREATEIMAGE:
	{
		Display *dpy = get_display(args[2]);
		Visual *visual = get_visual(args[3]);
		unsigned int depth = tswap32(args[4]);
		int format = tswap32(args[5]);
		char *data = (void *) tswap32(args[6]);
		XShmSegmentInfo *raw_shminfo = (void *) args[7];
		XShmSegmentInfo *target_shminfo = tswap32(raw_shminfo);
		XShmSegmentInfo *shminfo = malloc(sizeof(XShmSegmentInfo));
		memswap32(shminfo, target_shminfo, size32of(XShmSegmentInfo));
		unsigned int width = tswap32(args[8]);
		unsigned int height = tswap32(args[9]);
		XImage *image = XShmCreateImage(dpy, visual, depth, format,
			data, shminfo, width, height);
		XImageProxy *proxy = malloc(sizeof(XImageProxy));
		memswap32(proxy, image, 12);
		memcpy(&proxy->red_mask, &image->red_mask,
			3 * sizeof(unsigned long));
		proxy->obdata = (XPointer) raw_shminfo;
		proxy->native = image;
		proxy->initial_sync = 1;
		assign_result32(args[1], proxy);
		return 0;
	}
	case XSHMCREATEPIXMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Drawable d = tswap32(args[3]);
		char *data = (void *) tswap32(args[4]);
		XShmSegmentInfo *target_shminfo = (void *) tswap32(args[5]);
		XShmSegmentInfo shminfo;
		memswap32(&shminfo, target_shminfo, size32of(XShmSegmentInfo));
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		unsigned int depth = tswap32(args[8]);
		assign_result32(args[1], XShmCreatePixmap(dpy, d, data,
			&shminfo, width, height, depth));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSHMDETACH:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XShmSegmentInfo *target_shminfo = (void *) tswap32(args[3]);
		XShmSegmentInfo shminfo;
		memswap32(&shminfo, target_shminfo, size32of(XShmSegmentInfo));
		assign_result32(args[1], XShmDetach(dpy, &shminfo));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSHMGETEVENTBASE:
	{
		Display *dpy = get_display(args[2]);
		assign_result32(args[1], XShmGetEventBase(dpy));
		return 0;
	}
	case XSHMPUTIMAGE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		XImageProxy *image_proxy = (void *) tswap32(args[5]);
		XImage *image = get_image(image_proxy);
		int src_x = tswap32(args[6]);
		int src_y = tswap32(args[7]);
		int dst_x = tswap32(args[8]);
		int dst_y = tswap32(args[9]);
		unsigned int src_width = tswap32(args[10]);
		unsigned int src_height = tswap32(args[11]);
		Bool send_event = tswap32(args[12]);
		assign_result32(args[1], XShmPutImage(dpy, d, gc, image, src_x,
			src_y, dst_x, dst_y, src_width, src_height,
			send_event));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSHMQUERYEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		assign_result32(args[1], XShmQueryExtension(dpy));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSHMQUERYVERSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_majorVersion = (void *) tswap32(args[3]);
		int majorVersion;
		int *target_minorVersion = (void *) tswap32(args[4]);
		int minorVersion;
		Bool *target_sharedPixmaps = (void *) tswap32(args[5]);
		Bool sharedPixmaps;
		assign_result32(args[1], XShmQueryVersion(dpy, &majorVersion,
			&minorVersion, &sharedPixmaps));
		*target_majorVersion = tswap32(majorVersion);
		*target_minorVersion = tswap32(minorVersion);
		*target_sharedPixmaps = tswap32(sharedPixmaps);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCCREATECOUNTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XSyncValue initial_value = { tswap32(args[3]),
			tswap32(args[4]) };
		XSyncCounter *result = (void *) tswap32(args[1]);
		*result = XSyncCreateCounter(dpy, initial_value);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCDESTROYCOUNTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XSyncCounter counter = args[3];
		assign_result32(args[1], XSyncDestroyCounter(dpy, counter));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCINITIALIZE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_major_version_return = (void *) tswap32(args[3]);
		int major_version_return;
		int *target_minor_version_return = (void *) tswap32(args[4]);
		int minor_version_return;
		assign_result32(args[1], XSyncInitialize(dpy,
			&major_version_return, &minor_version_return));
		*target_major_version_return = tswap32(major_version_return);
		*target_minor_version_return = tswap32(minor_version_return);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCINTTOVALUE: {
		XSyncValue *target_pv = (void *) tswap32(args[1]);
		XSyncValue pv;
		memswap32(&pv, target_pv, size32of(XSyncValue));
		int i = tswap32(args[2]);
		XSyncIntToValue(&pv, i);
		memswap32(target_pv, &pv, size32of(XSyncValue));
		return 0;
	}
	case XSYNCQUERYEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_event_base_return = (void *) tswap32(args[3]);
		int event_base_return;
		int *target_error_base_return = (void *) tswap32(args[4]);
		int error_base_return;
		assign_result32(args[1], XSyncQueryExtension(dpy,
			&event_base_return, &error_base_return));
		*target_event_base_return = tswap32(event_base_return);
		*target_error_base_return = tswap32(error_base_return);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCSETCOUNTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XSyncCounter counter = args[3];
		XSyncValue value = { tswap32(args[4]), tswap32(args[5]) };
		assign_result32(args[1], XSyncSetCounter(dpy, counter, value));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case XSYNCVALUEISZERO: {
		XSyncValue a = { tswap32(args[2]), tswap32(args[3]) };
		assign_result32(args[1], XSyncValueIsZero(a));
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
