#define _GNU_SOURCE

#include <dlfcn.h>
#include <err.h>
#include <stdlib.h>
#include <string.h>
#include <emu-stub/handler-api.h>
#include <X11/ImUtil.h>
#include <X11/XKBlib.h>
#include <X11/Xlibint.h>
#include <X11/Xresource.h>
#include <X11/Xutil.h>

#include "libX11.h"
#include "libX11_const.h"

typedef struct {
	emu_cpu_env *env;
	unsigned callback_args[4];
} predicate_data;

typedef struct sent_request {
	struct sent_request *next;
	unsigned long request;
	CARD8 reqType;
} sent_request;

typedef struct _XExtenProxy {
	struct _XExtenProxy *next;
	XExtCodes codes;
	struct {
		emu_cpu_env *env;
		unsigned callback_args[3];
	} close_display_data;
} _XExtensionProxy;

typedef struct {
	XIM native;
	DisplayProxy *display;
} XIMProxy;

typedef struct {
	DisplayProxy *dpy;
	unsigned short flags;
	unsigned short device_spec;
	KeyCode min_key_code;
	KeyCode max_key_code;
	XkbControlsPtr ctrls;
	XkbServerMapPtr server;
	XkbClientMapPtr map;
	XkbIndicatorPtr indicators;
	XkbNamesPtr names;
	XkbCompatMapPtr compat;
	XkbGeometryPtr geom;
	XkbDescPtr native;
} XkbDescRecProxy;

static struct {
	emu_cpu_env *env;
	unsigned callback_args[6];
} async_handler_data;

static struct {
	emu_cpu_env *env;
	unsigned callback_args[3];
} ErrorHandler_data;

static struct _XLockPtrs *stub_lock_fns;

DisplayProxy *find_display_proxy(Display *display);
static int free_display_proxy(XExtData *extension);
static int free_visual_proxy(XExtData *extension);
static void swap_event(XEvent *dest, const XEvent *src, const XEvent *host);
static void swap_XkbMods(XkbModsPtr dest, const XkbModsPtr src);
static inline void to_native_display(Display **display);

static Bool async_handler(Display *dpy, xReply *rep, char *buf, int len,
	XPointer data) {
	DisplayProxy *proxy = (DisplayProxy *) data;
	proxy->last_request_read = tswap32(dpy->last_request_read);
	async_handler_data.callback_args[1] = (unsigned) tswap32(proxy);
	async_handler_data.callback_args[2] = (unsigned) tswap32(rep);
	async_handler_data.callback_args[3] = (unsigned) buf;
	async_handler_data.callback_args[4] = len;
	Bool handled;
	_XAsyncHandler *target_async = tswap32(proxy->async_handlers);
	do {
		_XAsyncHandler *next = tswap32(target_async->next);
		async_handler_data.callback_args[0]
			= (unsigned) target_async->handler;
		async_handler_data.callback_args[5]
			= (unsigned) target_async->data;
		emu_cpu_run(async_handler_data.env);
		handled = tswap32(async_handler_data.callback_args[0]);
		target_async = next;
	} while (!handled && target_async);

	sent_request *pending = proxy->sent_requests;
	if (pending && pending->request == dpy->last_request_read) {
		proxy->sent_requests = pending->next;
		free(pending);
	}
	if (!proxy->async_handlers) {
		_XAsyncHandler *async = dpy->async_handlers;
		if (async->handler != async_handler)
			warnx("wrong async_handler removal!");
		dpy->async_handlers = async->next;
		free(async);
		proxy->async_handler_set = False;
	}

	return handled;
}

static int close_display(Display *display, XExtCodes *codes) {
warnx("close_display called.");
	DisplayProxy *proxy = find_display_proxy(display);
	if (!proxy) {
		warnx("close_display: native display without proxy.");
		exit(EXIT_FAILURE);
	}
	int target_extension = tswap32(codes->extension);
	_XExtensionProxy *ext_proxy = proxy->ext_proxies;
	while (ext_proxy->codes.extension != target_extension)
		ext_proxy = ext_proxy->next;
	emu_cpu_run(ext_proxy->close_display_data.env);
	emu_cpu_free(ext_proxy->close_display_data.env);
	return 0;
}

static XImageProxy *create_image_proxy(XImage *image) {
	XImageProxy *proxy = malloc(sizeof(XImageProxy));
	memswap32(proxy, image, 12);
	memcpy(&proxy->red_mask, &image->red_mask, 3 * sizeof(unsigned long));
	proxy->obdata = NULL;
	proxy->native = image;
	proxy->initial_sync = 1;
	return proxy;
}

static void create_visual_proxy(VisualProxy *proxy, Visual *visual,
	DisplayProxy *display_proxy) {
	proxy->native = visual;
	memswap32(&proxy->visualid, &visual->visualid, 7);
	XEDataObject object = { .visual = visual };
	XExtData **structure = XEHeadOfExtensionList(object);
	XExtData *ext_data = malloc(sizeof(XExtData));
	ext_data->number = display_proxy->extension;
	ext_data->free_private = free_visual_proxy;
	ext_data->private_data = (XPointer) proxy;
	XAddToExtensionList(structure, ext_data);
}

static int ErrorHandler(Display *display, XErrorEvent *error_event) {
	DisplayProxy *proxy = find_display_proxy(display);
	int temp_proxy = !proxy;
	if (temp_proxy) {
		warnx("error handler: native display without proxy.");
		proxy = malloc(sizeof(DisplayProxy));
		proxy->native = display;
	}
	ErrorHandler_data.callback_args[1] = (unsigned) tswap32(proxy);
	XErrorEvent target_error_event;
	target_error_event.type = tswap32(error_event->type);
	if (display != error_event->display)
		warnx("error display is different");
	target_error_event.display = (Display *) tswap32(proxy);
	target_error_event.resourceid = tswap32(error_event->resourceid);
	memswap32(&target_error_event.resourceid, &error_event->resourceid, 2);
	memcpy(&target_error_event.error_code, &error_event->error_code, 3);
	ErrorHandler_data.callback_args[2]
		= (unsigned) tswap32(&target_error_event);
	emu_cpu_run(ErrorHandler_data.env);
	if (temp_proxy)
		free(proxy);
	return 0;
}

DisplayProxy *find_display_proxy(Display *display) {
	XExtData *ext_data = display->ext_data;
	for (; ext_data; ext_data = ext_data->next)
		if (ext_data->free_private == free_display_proxy)
			return (DisplayProxy *) ext_data->private_data;
	return NULL;
}

static int free_display_proxy(XExtData *extension) {
	DisplayProxy *proxy = (DisplayProxy *) extension->private_data;
	free(tswap32(proxy->lock_fns));
	if (proxy->async_handlers)
		warnx("pending handlers.");
	_XExtensionProxy *ext_proxy = proxy->ext_proxies;
	while (ext_proxy) {
		_XExtensionProxy *next = ext_proxy->next;
		free(ext_proxy);
		ext_proxy = next;
	}
	ProxyExtension *target_ext = proxy->target_ext;
	while (target_ext) {
		ProxyExtension *next = target_ext->next;
		target_ext->free_data(target_ext->data);
		free(target_ext);
		target_ext = next;
	}
	free(proxy);
	return 0;
}

static int free_screen_proxy(XExtData *extension) {
	ScreenProxy *proxy = (ScreenProxy *) extension->private_data;
	int ndepths = tswap32(proxy->ndepths);
	Depth *target_depths = tswap32(proxy->depths);
	int i;
	for (i = 0; i < ndepths; i++)
		free(tswap32(target_depths[i].visuals));
	free(target_depths);
	free(proxy);
	return 0;
}

static int free_visual_proxy(__attribute__((unused)) XExtData *extension) {
	return 0;
}

static XImage *get_image(XImageProxy *proxy) {
	XImage *image = proxy->native;
	if (proxy->initial_sync) {
		proxy->initial_sync = 0;
		image->data = tswap32(proxy->data);
		image->byte_order = tswap32(proxy->byte_order);
		image->bitmap_unit = tswap32(proxy->bitmap_unit);
		image->bitmap_bit_order = tswap32(proxy->bitmap_bit_order);
	}
	return image;
}

static inline Screen *get_screen(unsigned arg) {
	ScreenProxy *proxy = (void *) tswap32(arg);
	return proxy->native;
}

static void host_to_target_event(XEvent *target, const XEvent *host) {
	swap_event(target, host, host);
	DisplayProxy *proxy = find_display_proxy(host->xany.display);
// initialize other proxies on detection, use extension to free
	if (!proxy)
		warnx("event display without proxy");
	target->xany.display = (Display *) tswap32(proxy);
}

static int IOErrorHandler(Display *display) {
warnx("IOErrorHandler called!");
if (display) {}
return 0;
}

static Bool predicate_handler(Display *display, XEvent *event, XPointer arg) {
	predicate_data *pdata = (predicate_data *) arg;
	DisplayProxy *proxy = find_display_proxy(display);
	if (!proxy)
		warnx("predicate_handler: native display without proxy.");
	pdata->callback_args[1] = (unsigned) tswap32(proxy);
	XEvent target_event;
	host_to_target_event(&target_event, event);
	pdata->callback_args[2] = (unsigned) tswap32(&target_event);
	emu_cpu_run(pdata->env);
	Bool match = tswap32(pdata->callback_args[1]);
	return match;
}

static void refresh_display_native(DisplayProxy *proxy) {
	Display *display = proxy->native;
	display->last_req = tswap32(proxy->last_req);
	char *reqptr = display->bufptr;
	display->bufptr = tswap32(proxy->bufptr);
	while (reqptr < display->bufptr) {
		display->request++;
		CARD8 reqType = *reqptr;
		reqptr += 2;
		memswap16(reqptr, reqptr, 1);
		reqptr += sizeof(short);
		switch (reqType) {
		case X_SetInputFocus:
			memswap32(reqptr, reqptr, 2);
			reqptr += 2 * sizeof(int);
			break;
		case X_GetInputFocus:
		{
			sent_request *sent = malloc(sizeof(sent_request));
			sent->next = NULL;
			sent->request = display->request;
			sent->reqType = reqType;
			sent_request *insert = proxy->sent_requests;
			if (insert) {
				while (insert->next)
					insert = insert->next;
				insert->next = sent;
			}
			else
				proxy->sent_requests = sent;
			break;
		}
		default:
			warnx("unhandled proto request %d", reqType);
			reqptr = display->bufptr;
		}
	}
}

static void set_screen_proxy(Screen **screen, DisplayProxy *display_proxy) {
	XEDataObject object = { .screen = *screen };
	XExtData **structure = XEHeadOfExtensionList(object);
	XExtData *ext_data = XFindOnExtensionList(structure,
		display_proxy->extension);
	*screen = (Screen *) ext_data->private_data;
}

void set_visual_proxy(Visual **visual, DisplayProxy *display_proxy) {
	XEDataObject object = { .visual = *visual };
	XExtData **structure = XEHeadOfExtensionList(object);
	XExtData *ext_data = XFindOnExtensionList(structure,
		display_proxy->extension);
	*visual = (Visual *) ext_data->private_data;
}

static inline void swap_color(XColor *dest, const XColor *src) {
	dest->pixel = tswap32(src->pixel);
	memswap16(&dest->red, &src->red, 3);
	dest->flags = src->flags;
}

static void swap_event(XEvent *dest, const XEvent *src, const XEvent *host) {
	int type = src == host ? src->type : tswap32(src->type);
	switch (type) {
	case KeyPress:
	case KeyRelease:
		memswap32(dest, src, size32of(XKeyEvent));
		break;
	case ButtonPress:
	case ButtonRelease:
		memswap32(dest, src, size32of(XButtonEvent));
		break;
	case MotionNotify:
		memswap32(dest, src, 13);
		dest->xmotion.is_hint = src->xmotion.is_hint;
		dest->xmotion.same_screen = tswap32(src->xmotion.same_screen);
		break;
	case EnterNotify:
	case LeaveNotify:
		memswap32(dest, src, size32of(XCrossingEvent));
		break;
	case FocusIn:
	case FocusOut:
		memswap32(dest, src, size32of(XFocusChangeEvent));
		break;
	case KeymapNotify:
		memswap32(dest, src, 5);
		memcpy(dest->xkeymap.key_vector, src->xkeymap.key_vector,
			sizeof(src->xkeymap.key_vector));
		break;
	case Expose:
		memswap32(dest, src, size32of(XExposeEvent));
		break;
	case GraphicsExpose:
		memswap32(dest, src, size32of(XGraphicsExposeEvent));
		break;
	case NoExpose:
		memswap32(dest, src, size32of(XNoExposeEvent));
		break;
	case VisibilityNotify:
		memswap32(dest, src, size32of(XVisibilityEvent));
		break;
	case CreateNotify:
		memswap32(dest, src, size32of(XCreateWindowEvent));
		break;
	case DestroyNotify:
		memswap32(dest, src, size32of(XDestroyWindowEvent));
		break;
	case UnmapNotify:
		memswap32(dest, src, size32of(XUnmapEvent));
		break;
	case MapNotify:
		memswap32(dest, src, size32of(XMapEvent));
		break;
	case ReparentNotify:
		memswap32(dest, src, size32of(XReparentEvent));
		break;
	case ConfigureNotify:
		memswap32(dest, src, size32of(XConfigureEvent));
		break;
	case PropertyNotify:
		memswap32(dest, src, size32of(XPropertyEvent));
		break;
	case SelectionClear:
		memswap32(dest, src, size32of(XSelectionClearEvent));
		break;
	case SelectionRequest:
		memswap32(dest, src, size32of(XSelectionRequestEvent));
		break;
	case SelectionNotify:
		memswap32(dest, src, size32of(XSelectionEvent));
		break;
	case ClientMessage:
		memswap32(dest, src, 7);
		switch (host->xclient.format) {
		case 8:
			memcpy(dest->xclient.data.b, src->xclient.data.b, 20);
			break;
		case 16:
			memswap16(dest->xclient.data.s, src->xclient.data.s,
				10);
			break;
		case 32:
			memswap32(dest->xclient.data.l, src->xclient.data.l, 5);
			break;
		}
		break;
	default:
	{
		DisplayProxy *proxy = src == host ?
			find_display_proxy(src->xany.display)
			: (DisplayProxy *) tswap32(src->xany.display);
		if (type == proxy->xkb_event_base) {
			XkbEvent *xkb_dest = (XkbEvent *) dest;
			const XkbEvent *xkb_src = (XkbEvent *) src;
			int xkb_type = src == host ? xkb_src->any.xkb_type
				: tswap32(xkb_src->any.xkb_type);
			switch (xkb_type) {
			case XkbNewKeyboardNotify:
				memswap32(xkb_dest, xkb_src, 13);
				memcpy(&xkb_dest->new_kbd.req_major,
					&xkb_src->new_kbd.req_major, 2);
				break;
			case XkbStateNotify:
				memswap32(xkb_dest, xkb_src, 17);
				memcpy(&xkb_dest->state.grab_mods,
					&xkb_src->state.grab_mods, 4);
				xkb_dest->state.ptr_buttons
					= tswap32(xkb_src->state.ptr_buttons);
				memcpy(&xkb_dest->state.keycode,
					&xkb_src->state.keycode, 4);
				break;
			default:
				warnx("unhandled XkbEvent type %d", xkb_type);
			}
			break;
		}
		ProxyExtension *target_ext = proxy->target_ext;
		while (target_ext) {
			if (target_ext->swap_event
				&& target_ext->swap_event(dest, src, host,
				target_ext->data))
				break;
			target_ext = target_ext->next;
		}
		if (!target_ext)
			warnx("unhandled XEvent type %d", type);
	}
	}
}

static inline void swap_gcvalues(XGCValues *dest, const XGCValues *src) {
	memswap32(dest, src, size32of(XGCValues) - 1);
	dest->dashes = src->dashes;
}

static void swap_XkbKeyType(XkbKeyTypePtr dest, const XkbKeyTypePtr src) {
	swap_XkbMods(&dest->mods, &src->mods);
	dest->num_levels = src->num_levels;
	unsigned char map_count = src->map_count;
	dest->map_count = map_count;
	if (map_count) {
		XkbKTMapEntryPtr map = src->map;
		XkbKTMapEntryRec *map_proxy = malloc(map_count
			* sizeof(XkbKTMapEntryRec));
		dest->map = tswap32(map_proxy);
		int i;
		for (i = 0; i < map_count; i++) {
			map_proxy[i].active = tswap32(map[i].active);
			map_proxy[i].level = map[i].level;
			swap_XkbMods(&map_proxy[i].mods, &map[i].mods);
		}
	}
	else
		dest->map = NULL;
	if (src->preserve) {
		XkbModsPtr preserve = src->preserve;
		XkbModsRec *preserve_proxy = malloc(map_count
			* sizeof(XkbModsRec));
		dest->preserve = tswap32(preserve_proxy);
		int i;
		for (i = 0; i < map_count; i++)
			swap_XkbMods(preserve_proxy + i, preserve + i);
	}
	else
		dest->preserve = NULL;
	dest->name = tswap32(src->name);
	if (src->level_names)
		warnx("src->level_names not null");
	else
		dest->level_names = NULL;
}

static void swap_XkbMods(XkbModsPtr dest, const XkbModsPtr src) {
	dest->mask = src->mask;
	dest->real_mods = src->real_mods;
	dest->vmods = tswap16(src->vmods);
}

static void swap_XTextProperty(XTextProperty *dest, const XTextProperty *src,
	const XTextProperty *host) {
	memswap32(dest, src, size32of(XTextProperty));
	switch (host->format) {
	case 16:
	case 32:
		warnx("unhandled XTextProperty format");
	}
}

void target_to_host_event(XEvent *host, const XEvent *target) {
	swap_event(host, target, host);
	to_native_display(&host->xany.display);
}

static inline void to_native_display(Display **display) {
	DisplayProxy *proxy = (DisplayProxy *) *display;
	*display = proxy->native;
}

static inline void to_native_key_event(XKeyEvent *dest, XKeyEvent *src) {
	memswap32(dest, src, size32of(XKeyEvent));
	to_native_display(&dest->display);
}

static void update_xkb_desc_proxy(XkbDescRecProxy *desc_proxy,
	unsigned int which) {
	static unsigned int UNSUPPORTED = ~(XkbKeyTypesMask | XkbKeySymsMask
		| XkbModifierMapMask | XkbVirtualModsMask);
	if (which & UNSUPPORTED)
		warnx("Unsupported XkbGetMap flags");

	XkbDescPtr desc = desc_proxy->native;
	if (which & XkbAllClientInfoMask) {
		XkbClientMapPtr map = desc->map;
		XkbClientMapRec *map_proxy;
		if (desc_proxy->map)
			map_proxy = tswap32(desc_proxy->map);
		else {
			map_proxy = malloc(sizeof(XkbClientMapRec));
			map_proxy->types = NULL;
			map_proxy->syms = NULL;
			map_proxy->key_sym_map = NULL;
			desc_proxy->map = tswap32(map_proxy);
		}
		if (which & XkbKeyTypesMask) {
			unsigned char size_types = map->size_types;
			map_proxy->size_types = size_types;
			map_proxy->num_types = map->num_types;
			XkbKeyTypePtr types = map->types;
			XkbKeyTypeRec *types_proxy = tswap32(map_proxy->types);
			types_proxy = realloc(types_proxy,
				size_types * sizeof(XkbKeyTypeRec));
			map_proxy->types = tswap32(types_proxy);
			int i;
			for (i = 0; i < size_types; i++)
				swap_XkbKeyType(types_proxy + i, types + i);
		}
		if (which & XkbKeySymsMask) {
			unsigned short size_syms = map->size_syms;
			map_proxy->size_syms = tswap16(size_syms);
			map_proxy->num_syms = tswap16(map->num_syms);
			KeySym *syms = tswap32(map_proxy->syms);
			syms = realloc(syms, size_syms * sizeof(KeySym));
			map_proxy->syms = tswap32(syms);
			memswap32(syms, map->syms, size_syms);

			int n = desc->max_key_code + 1;
			XkbSymMapPtr key_sym_map = map->key_sym_map;
			XkbSymMapRec *key_sym_map_proxy
				= tswap32(map_proxy->key_sym_map);
			key_sym_map_proxy = realloc(key_sym_map_proxy,
				n * sizeof(XkbSymMapRec));
			map_proxy->key_sym_map = tswap32(key_sym_map_proxy);
			int i;
			for (i = 0; i < n; i++) {
				memcpy(&key_sym_map_proxy[i].kt_index,
					&key_sym_map[i].kt_index,
					XkbNumKbdGroups);
				key_sym_map_proxy[i].group_info
					= key_sym_map[i].group_info;
				key_sym_map_proxy[i].width
					= key_sym_map[i].width;
				key_sym_map_proxy[i].offset
					= tswap16(key_sym_map[i].offset);
			}
		}
		if (which & XkbModifierMapMask)
			map_proxy->modmap = tswap32(map->modmap);
	}
	if (which & XkbAllServerInfoMask) {
		XkbServerMapPtr server = desc->server;
		XkbServerMapRec *server_proxy;
		if (desc_proxy->server)
			server_proxy = tswap32(desc_proxy->server);
		else {
			server_proxy = malloc(sizeof(XkbServerMapRec));
			server_proxy->acts = NULL;
			server_proxy->behaviors = NULL;
			server_proxy->key_acts = NULL;
			server_proxy->explicit = NULL;
			server_proxy->vmodmap = NULL;
			desc_proxy->server = tswap32(server_proxy);
		}
		if (which & XkbVirtualModsMask)
			memcpy(server_proxy->vmods, server->vmods,
				XkbNumVirtualMods);
	}
}

static void XConnectionWatch(Display *dpy, XPointer client_data, int fd,
	Bool opening, XPointer *watch_data) {
warnx("Called connection watch!");
if (dpy || client_data || fd || opening || watch_data) {}
/*    XtInputId* iptr;
    XtAppContext app = XtDisplayToApplicationContext(dpy);

    if (opening) {
	iptr = (XtInputId *) __XtMalloc(sizeof(XtInputId));
	*iptr = XtAppAddInput(app, fd, (XtPointer) XtInputReadMask,
			      ProcessInternalConnection, client_data);
	*watch_data = (XPointer) iptr;
    } else {
	iptr = (XtInputId *) *watch_data;
	XtRemoveInput(*iptr);
        (void) XtFree(*watch_data);
    }*/
}

static Bool xic_destroy(XIC xic, XPointer p, XPointer data) {
warnx("xic_destroy called!");
if (xic || p || data) {}
	return True;
}

static void xim_destroy(XIM xim, XPointer p, XPointer data) {
warnx("xim_destroy called!");
if (xim || p || data) {}
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
static DisplayProxy *g_display_proxy;
static unsigned last_function;
if (g_display_proxy && function != _XGETASYNCREPLY
 && g_display_proxy->request < g_display_proxy->native->request){
warnx("stub %d req %ld", last_function, g_display_proxy->native->request);
exit(0);
}
last_function = function;
	switch (function) {
	case DESTROY_IMAGE:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[2]);
		XImage *ximage = image_proxy->native;
		if (!image_proxy->data)
			ximage->data = NULL;
		assign_result32(args[1], XDestroyImage(ximage));
		free(image_proxy);
		return 0;
	}
	case GET_PIXEL:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[2]);
		XImage *ximage = get_image(image_proxy);
		int x = tswap32(args[3]);
		int y = tswap32(args[4]);
		assign_result32(args[1], XGetPixel(ximage, x, y));
		return 0;
	}
	case LOCK_DISPLAY: {
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		if (dpy->lock_fns)
			dpy->lock_fns->lock_display(dpy);
		proxy->request = tswap32(dpy->request);
		proxy->last_req = tswap32(dpy->last_req);
		proxy->bufptr = tswap32(dpy->bufptr);
		return 0;
	}
	case PUT_PIXEL:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[2]);
		XImage *ximage = get_image(image_proxy);
		int x = tswap32(args[3]);
		int y = tswap32(args[4]);
		unsigned long pixel = tswap32(args[5]);
		assign_result32(args[1], XPutPixel(ximage, x, y, pixel));
		return 0;
	}
	case STUB_INIT: {
		stub_lock_fns = (void *) tswap32(args[1]);
		return 0;
	}
	case UNLOCK_DISPLAY:
	{
		if (!async_handler_data.env) {
			emu_cpu_env *new_env = emu_cpu_clone(env);
			async_handler_data.env = new_env;
			emu_set_return_value(new_env,
				(unsigned) async_handler_data.callback_args);
		}
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		refresh_display_native(proxy);
		if (proxy->async_handlers && !proxy->async_handler_set) {
			_XAsyncHandler *async = malloc(sizeof(_XAsyncHandler));
			async->next = dpy->async_handlers;
			async->handler = async_handler;
			async->data = (XPointer) proxy;
			dpy->async_handlers = async;
			proxy->async_handler_set = True;
		}
		if (dpy->synchandler)
			warnx("unlocking, synchandler not null.");
		if (dpy->lock_fns)
			dpy->lock_fns->unlock_display(dpy);
		return 0;
	}
	case XADDCONNECTIONWATCH:
	{
		Display *dpy = get_display(args[2]);
// XConnectionWatchProc callback
//		void *callback = (void *) tswap32(args[3]);
// XPointer client_data
		void *client_data = (void *) tswap32(args[4]);
//		assign_result32(args[1], XAddConnectionWatch(dpy, callback,
//			client_data));
		assign_result32(args[1], XAddConnectionWatch(dpy,
			XConnectionWatch, client_data));
		return 0;
	}
	case XADDEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		XExtCodes *codes = XAddExtension(display);
		_XExtensionProxy *ext_proxy = malloc(sizeof(_XExtensionProxy));
		memswap32(&ext_proxy->codes, codes, size32of(XExtCodes));
		assign_result32(args[1], &ext_proxy->codes);
		ext_proxy->next = proxy->ext_proxies;
		proxy->ext_proxies = ext_proxy;
		return 0;
	}
	case XALLOCCLASSHINT:
	{
		assign_result32(args[1], XAllocClassHint());
		return 0;
	}
	case XALLOCCOLOR:
	{
warnx("XAllocColor called!");
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Colormap colormap = tswap32(args[3]);
		XColor *target_screen_in_out = (void *) tswap32(args[4]);
		XColor screen_in_out;
		swap_color(&screen_in_out, target_screen_in_out);
		assign_result32(args[1], XAllocColor(display, colormap,
			&screen_in_out));
		swap_color(target_screen_in_out, &screen_in_out);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XALLOCNAMEDCOLOR:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Colormap colormap = tswap32(args[3]);
		void *color_name = (void *) tswap32(args[4]);
		XColor *target_screen_def_return = (void *) tswap32(args[5]);
		XColor screen_def_return;
		XColor *target_exact_def_return = (void *) tswap32(args[6]);
		XColor exact_def_return;
		assign_result32(args[1], XAllocNamedColor(display, colormap,
			color_name, &screen_def_return, &exact_def_return));
		swap_color(target_screen_def_return, &screen_def_return);
		swap_color(target_exact_def_return, &exact_def_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XALLOCSIZEHINTS:
	{
		assign_result32(args[1], XAllocSizeHints());
		return 0;
	}
	case XALLOCWMHINTS:
	{
		assign_result32(args[1], XAllocWMHints());
		return 0;
	}
	case XBELL:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		int percent = tswap32(args[3]);
		assign_result32(args[1], XBell(display, percent));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCHANGEGC:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		unsigned long valuemask = tswap32(args[4]);
		XGCValues *target_values = (void *) tswap32(args[5]);
		XGCValues values;
		swap_gcvalues(&values, target_values);
		assign_result32(args[1], XChangeGC(display, gc, valuemask,
			&values));
		return 0;
	}
	case XCHANGEPROPERTY:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Atom property = tswap32(args[4]);
		Atom type = tswap32(args[5]);
		int format = tswap32(args[6]);
		int mode = tswap32(args[7]);
		void *target_data = (void *) tswap32(args[8]);
		int nelements = tswap32(args[9]);
		void *data = target_data;
		switch (format) {
		case 16:
			data = malloc(nelements * sizeof(short));
			memswap16(data, target_data, nelements);
			break;
		case 32:
			data = malloc(nelements * sizeof(unsigned));
			memswap32(data, target_data, nelements);
			break;
		}
		assign_result32(args[1], XChangeProperty(display, w, property,
			type, format, mode, data, nelements));
		if (data != target_data)
			free(data);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCHANGEWINDOWATTRIBUTES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		unsigned long valuemask = tswap32(args[4]);
		XSetWindowAttributes *target_attributes
			= (void *) tswap32(args[5]);
		XSetWindowAttributes attributes;
		memswap32(&attributes, target_attributes,
			size32of(XSetWindowAttributes));
		assign_result32(args[1], XChangeWindowAttributes(display, w,
			valuemask, &attributes));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCHECKIFEVENT:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		Display *display = get_display(args[2]);
		XEvent *target_event_return = (void *) tswap32(args[3]);
		XEvent event_return;
		Bool (*predicate)(Display *, XEvent *, XPointer)
			= (void *) args[4];
		XPointer arg = (void *) args[5];
		predicate_data pdata;
		pdata.env = new_env;
		pdata.callback_args[0] = (unsigned) predicate;
		pdata.callback_args[3] = (unsigned) arg;
		emu_set_return_value(new_env, (unsigned) pdata.callback_args);
		Bool match = XCheckIfEvent(display, &event_return,
			predicate_handler, (XPointer) &pdata);
		if (match)
			host_to_target_event(target_event_return,
				&event_return);
		assign_result32(args[1], match);
		emu_cpu_free(new_env);
		return 0;
	}
	case XCHECKTYPEDWINDOWEVENT:
	{
		Display *display = get_display(args[2]);
		Window w = tswap32(args[3]);
		int event_type = tswap32(args[4]);
		XEvent *target_event_return = (void *) tswap32(args[5]);
		XEvent event_return;
		Bool found = XCheckTypedWindowEvent(display, w, event_type,
			&event_return);
		assign_result32(args[1], found);
		if (found)
			host_to_target_event(target_event_return,
				&event_return);
		return 0;
	}
	case XCLEARAREA:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		Bool exposures = tswap32(args[8]);
		assign_result32(args[1], XClearArea(display, w, x, y, width,
			height, exposures));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCLEARWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XClearWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCLIPBOX:
	{
		Region r = (void *) args[2];
		XRectangle *target_rect_return = (void *) tswap32(args[3]);
		XRectangle rect_return;
		assign_result32(args[1], XClipBox(r, &rect_return));
		memswap16(target_rect_return, &rect_return,
			size16of(XRectangle));
		return 0;
	}
	case XCLOSEDISPLAY:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XCloseDisplay(display));
g_display_proxy = NULL;
		return 0;
	}
	case XCONFIGUREWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		unsigned int value_mask = tswap32(args[4]);
		XWindowChanges *target_values = (void *) tswap32(args[5]);
		XWindowChanges values;
		memswap32(&values, target_values, size32of(XWindowChanges));
		assign_result32(args[1], XConfigureWindow(display, w,
			value_mask, &values));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCONVERTCASE:
	{
		KeySym sym = tswap32(args[1]);
		KeySym *target_lower = (void *) tswap32(args[2]);
		KeySym lower;
		KeySym *target_upper = (void *) tswap32(args[3]);
		KeySym upper;
		XConvertCase(sym, &lower, &upper);
		*target_lower = tswap32(lower);
		*target_upper = tswap32(upper);
		return 0;
	}
	case XCONVERTSELECTION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Atom selection = tswap32(args[3]);
		Atom target = tswap32(args[4]);
		Atom property = tswap32(args[5]);
		Window requestor = tswap32(args[6]);
		Time time = tswap32(args[7]);
		assign_result32(args[1], XConvertSelection(display, selection,
			target, property, requestor, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCOPYAREA:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable src = tswap32(args[3]);
		Drawable dest = tswap32(args[4]);
		GC gc = (void *) args[5];
		int src_x = tswap32(args[6]);
		int src_y = tswap32(args[7]);
		unsigned int width = tswap32(args[8]);
		unsigned int height = tswap32(args[9]);
		int dest_x = tswap32(args[10]);
		int dest_y = tswap32(args[11]);
		assign_result32(args[1], XCopyArea(display, src, dest, gc,
			src_x, src_y, width, height, dest_x, dest_y));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCOPYPLANE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable src = tswap32(args[3]);
		Drawable dest = tswap32(args[4]);
		GC gc = (void *) args[5];
		int src_x = tswap32(args[6]);
		int src_y = tswap32(args[7]);
		unsigned int width = tswap32(args[8]);
		unsigned int height = tswap32(args[9]);
		int dest_x = tswap32(args[10]);
		int dest_y = tswap32(args[11]);
		unsigned long plane = tswap32(args[12]);
		assign_result32(args[1], XCopyPlane(display, src, dest, gc,
			src_x, src_y, width, height, dest_x, dest_y, plane));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEBITMAPFROMDATA:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		void *data = (void *) tswap32(args[4]);
		unsigned int width = tswap32(args[5]);
		unsigned int height = tswap32(args[6]);
		assign_result32(args[1], XCreateBitmapFromData(display, d, data,
			width, height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATECOLORMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Visual *visual = get_visual(args[4]);
		int alloc = tswap32(args[5]);
		assign_result32(args[1], XCreateColormap(display, w, visual,
			alloc));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEFONTCURSOR:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		unsigned int shape = tswap32(args[3]);
		assign_result32(args[1], XCreateFontCursor(display, shape));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEGC:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		unsigned long valuemask = tswap32(args[4]);
		XGCValues *target_values = (void *) tswap32(args[5]);
		XGCValues values;
		if (target_values)
			swap_gcvalues(&values, target_values);
		GC *result = (void *) tswap32(args[1]);
		*result = XCreateGC(display, d, valuemask, &values);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEIC:
	{
		XIMProxy *im_proxy = (void *) args[2];
		XIM im = im_proxy->native;
		unsigned *vargs = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		if (count != 4)
			warnx("XCreateIC called with %d pairs.", count);
		memswap32(vargs, vargs, 2 * count);
		char *vargs0 = (char *) vargs[0];
		unsigned vargs1 = vargs[1];
		if (strcmp(vargs0, XNInputStyle))
			warnx("XCreatIC unsupported argument 0 %s.",
				vargs0);
		char *vargs2 = (char *) vargs[2];
		unsigned vargs3 = vargs[3];
		if (strcmp(vargs2, XNClientWindow))
			warnx("XCreatIC unsupported argument 2 %s.",
				vargs2);
		char *vargs4 = (char *) vargs[4];
		unsigned vargs5 = vargs[5];
		if (strcmp(vargs4, XNFocusWindow))
			warnx("XCreatIC unsupported argument 4 %s.",
				vargs4);
		char *vargs6 = (char *) vargs[6];
		unsigned vargs7;
		if (!strcmp(vargs6, XNDestroyCallback)) {
			XICCallback *target_destroy = (XICCallback *) vargs[7];
			XICCallback destroy;
			destroy.client_data = target_destroy->client_data;
			destroy.callback = xic_destroy;
			vargs7 = (unsigned) &destroy;
		}
		else {
			warnx("XCreatIC unsupported argument 6 %s.",
				vargs6);
			vargs7 = vargs[7];
		}
		XIC *result = (void *) tswap32(args[1]);
		*result = XCreateIC(im, vargs0, vargs1, vargs2, vargs3, vargs4,
			vargs5, vargs6, vargs7, NULL);
		DisplayProxy *proxy = im_proxy->display;
		proxy->request = tswap32(proxy->native->request);
		return 0;
	}
	case XCREATEIMAGE:
	{
		Display *display = get_display(args[2]);
		Visual *visual = get_visual(args[3]);
		unsigned int depth = tswap32(args[4]);
		int format = tswap32(args[5]);
		int offset = tswap32(args[6]);
		char *data = (void *) tswap32(args[7]);
		unsigned int width = tswap32(args[8]);
		unsigned int height = tswap32(args[9]);
		int bitmap_pad = tswap32(args[10]);
		int bytes_per_line = tswap32(args[11]);
		XImage *image = XCreateImage(display, visual, depth, format,
			offset, data, width, height, bitmap_pad,
			bytes_per_line);
		assign_result32(args[1], create_image_proxy(image));
		return 0;
	}
	case XCREATEPIXMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		unsigned int width = tswap32(args[4]);
		unsigned int height = tswap32(args[5]);
		unsigned int depth = tswap32(args[6]);
		assign_result32(args[1], XCreatePixmap(display, d, width,
			height, depth));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEPIXMAPCURSOR:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Pixmap source = tswap32(args[3]);
		Pixmap mask = tswap32(args[4]);
		XColor *target_foreground_color = (void *) tswap32(args[5]);
		XColor foreground_color;
		swap_color(&foreground_color, target_foreground_color);
		XColor *target_background_color = (void *) tswap32(args[6]);
		XColor background_color;
		swap_color(&background_color, target_background_color);
		unsigned int x = tswap32(args[7]);
		unsigned int y = tswap32(args[8]);
		assign_result32(args[1], XCreatePixmapCursor(display, source,
			mask, &foreground_color, &background_color, x, y));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEREGION:
	{
		Region *result = (void *) tswap32(args[1]);
		*result = XCreateRegion();
		return 0;
	}
	case XCREATESIMPLEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window parent = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		unsigned int border_width = tswap32(args[8]);
		unsigned long border = tswap32(args[9]);
		unsigned long background = tswap32(args[10]);
		assign_result32(args[1], XCreateSimpleWindow(display, parent, x,
			y, width, height, border_width, border, background));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XCREATEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window parent = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		unsigned int border_width = tswap32(args[8]);
		int depth = tswap32(args[9]);
		unsigned int class = tswap32(args[10]);
		Visual *visual = get_visual(args[11]);
		unsigned long valuemask = tswap32(args[12]);
		XSetWindowAttributes *target_attributes
			= (void *) tswap32(args[13]);
		XSetWindowAttributes attributes;
		memswap32(&attributes, target_attributes,
			size32of(XSetWindowAttributes));
		assign_result32(args[1], XCreateWindow(display, parent, x, y,
			width, height, border_width, depth, class, visual,
			valuemask, &attributes));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDEFINECURSOR:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Cursor cursor = tswap32(args[4]);
		assign_result32(args[1], XDefineCursor(display, w, cursor));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDELETECONTEXT:
	{
		Display *display = get_display(args[2]);
		XID rid = tswap32(args[3]);
		XContext context = args[4];
		assign_result32(args[1], XDeleteContext(display, rid, context));
		return 0;
	}
	case XDELETEPROPERTY:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Atom property = tswap32(args[4]);
		assign_result32(args[1], XDeleteProperty(display, w, property));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDESTROYIC:
	{
		XIC ic = (void *) args[1];
		XDestroyIC(ic);
		return 0;
	}
	case XDESTROYWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XDestroyWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDISPLAYKEYCODES:
	{
		Display *display = get_display(args[2]);
		int *target_min_keycodes_return = (void *) tswap32(args[3]);
		int min_keycodes_return;
		int *target_max_keycodes_return = (void *) tswap32(args[4]);
		int max_keycodes_return;
		assign_result32(args[1], XDisplayKeycodes(display,
			&min_keycodes_return, &max_keycodes_return));
		*target_min_keycodes_return = tswap32(min_keycodes_return);
		*target_max_keycodes_return = tswap32(max_keycodes_return);
		return 0;
	}
	case XDISPLAYNAME:
	{
		const char *string = (void *) tswap32(args[2]);
		assign_result32(args[1], XDisplayName(string));
		return 0;
	}
	case XDRAWARC:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		unsigned int width = tswap32(args[7]);
		unsigned int height = tswap32(args[8]);
		int angle1 = tswap32(args[9]);
		int angle2 = tswap32(args[10]);
		assign_result32(args[1], XDrawArc(display, d, gc, x, y, width,
			height, angle1, angle2));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDRAWLINE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x1 = tswap32(args[5]);
		int y1 = tswap32(args[6]);
		int x2 = tswap32(args[7]);
		int y2 = tswap32(args[8]);
		assign_result32(args[1], XDrawLine(display, d, gc, x1, y1, x2,
			y2));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDRAWLINES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		XPoint *target_points = (void *) tswap32(args[5]);
		int npoints = tswap32(args[6]);
		XPoint *points = malloc(npoints * sizeof(XPoint));
		memswap16(points, target_points, npoints * size16of(XPoint));
		int mode = tswap32(args[7]);
		assign_result32(args[1], XDrawLines(display, d, gc, points,
			npoints, mode));
		free(points);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDRAWPOINT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		assign_result32(args[1], XDrawPoint(display, d, gc, x, y));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XDRAWRECTANGLE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		unsigned int width = tswap32(args[7]);
		unsigned int height = tswap32(args[8]);
		assign_result32(args[1], XDrawRectangle(display, d, gc, x, y,
			width, height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XESETCLOSEDISPLAY:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		int target_extension = args[3];
		int extension = tswap32(target_extension);
		_XExtensionProxy *ext_proxy = proxy->ext_proxies;
		while (ext_proxy->codes.extension != target_extension)
			ext_proxy = ext_proxy->next;
		CloseDisplayType proc = (void *) args[4];
		ext_proxy->close_display_data.env = new_env;
		ext_proxy->close_display_data.callback_args[0]
			= (unsigned) proc;
		ext_proxy->close_display_data.callback_args[1]
			= (unsigned) tswap32(proxy);
		ext_proxy->close_display_data.callback_args[2]
			= (unsigned) tswap32(&ext_proxy->codes);
		emu_set_return_value(new_env,
			(unsigned) ext_proxy->close_display_data.callback_args);
		CloseDisplayType previous = XESetCloseDisplay(display,
			extension, close_display);
		if (previous)
			warnx("Previous close display hook not null.");
		assign_result32(args[1], previous);
		return 0;
	}
	case XEVENTSQUEUED:
	{
		Display *display = get_display(args[2]);
		int mode = tswap32(args[3]);
		assign_result32(args[1], XEventsQueued(display, mode));
		return 0;
	}
	case XEXTENDEDMAXREQUESTSIZE:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XExtendedMaxRequestSize(display));
		return 0;
	}
	case XFILLARC:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		unsigned int width = tswap32(args[7]);
		unsigned int height = tswap32(args[8]);
		int angle1 = tswap32(args[9]);
		int angle2 = tswap32(args[10]);
		assign_result32(args[1], XFillArc(display, d, gc, x, y, width,
			height, angle1, angle2));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFILLPOLYGON:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		XPoint *target_points = (void *) tswap32(args[5]);
		int npoints = tswap32(args[6]);
		XPoint *points = malloc(npoints * sizeof(XPoint));
		memswap16(points, target_points, npoints * size16of(XPoint));
		int shape = tswap32(args[7]);
		int mode = tswap32(args[8]);
		assign_result32(args[1], XFillPolygon(display, d, gc, points,
			npoints, shape, mode));
		free(points);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFILLRECTANGLE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		unsigned int width = tswap32(args[7]);
		unsigned int height = tswap32(args[8]);
		assign_result32(args[1], XFillRectangle(display, d, gc, x, y,
			width, height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFILTEREVENT:
	{
		XEvent *target_event = (void *) tswap32(args[2]);
		XEvent event;
		target_to_host_event(&event, target_event);
		Window window = tswap32(args[3]);
		assign_result32(args[1], XFilterEvent(&event, window));
		return 0;
	}
	case XFINDCONTEXT:
	{
// TODO: check returned data, xeyes
		Display *display = get_display(args[2]);
		XID rid = tswap32(args[3]);
		XContext context = args[4];
		XPointer *data_return = (void *) tswap32(args[5]);
		assign_result32(args[1], XFindContext(display, rid, context,
			data_return));
		return 0;
	}
	case XFLUSH:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XFlush(display));
		return 0;
	}
	case XFREE:
	{
		void *data = (void *) tswap32(args[2]);
		assign_result32(args[1], XFree(data));
		return 0;
	}
	case XFREECOLORMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Colormap colormap = tswap32(args[3]);
		assign_result32(args[1], XFreeColormap(display, colormap));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFREECURSOR:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Cursor cursor = tswap32(args[3]);
		assign_result32(args[1], XFreeCursor(display, cursor));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFREEGC:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		GC gc = (void *) args[3];
		assign_result32(args[1], XFreeGC(display, gc));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XFREEMODIFIERMAP:
	{
		XModifierKeymap *modmap = (void *) tswap32(args[2]);
		memswap32(modmap, modmap, size32of(XModifierKeymap));
		assign_result32(args[1], XFreeModifiermap(modmap));
		return 0;
	}
	case XFREEPIXMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Pixmap pixmap = tswap32(args[3]);
		assign_result32(args[1], XFreePixmap(display, pixmap));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case _XGETASYNCREPLY:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		char *replbuf = (void *) tswap32(args[2]);
		xReply *rep = (void *) tswap32(args[3]);
		char *buf = (void *) args[4];
		int len = args[5];
		int extra = tswap32(args[6]);
		Bool discard = tswap32(args[7]);
		char *repl = _XGetAsyncReply(dpy, replbuf, rep, buf, len,
			extra, discard);
		xReply *dest = (xReply *) replbuf;
		xReply *src = (xReply *) repl;
		dest->generic.type = src->generic.type;
		dest->generic.data1 = src->generic.data1;
		dest->generic.sequenceNumber
			= tswap16(src->generic.sequenceNumber);
		sent_request *sent = proxy->sent_requests;
		while (sent && sent->request != dpy->last_request_read)
			sent = sent->next;
		if (!sent)
			warnx("sent request not found!");
		switch (sent->reqType) {
		case X_GetInputFocus:
			dest->inputFocus.focus = tswap32(src->inputFocus.focus);
			break;
		default:
			warnx("unhandled async reply %d", sent->reqType);
		}
		return 0;
	}
	case XGETATOMNAME:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Atom atom = tswap32(args[3]);
		assign_result32(args[1], XGetAtomName(display, atom));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETATOMNAMES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Atom *target_atoms = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		Atom *atoms = malloc(count * sizeof(Atom));
		memswap32(atoms, target_atoms, count);
		char **names_return = (void *) tswap32(args[5]);
		assign_result32(args[1], XGetAtomNames(dpy, atoms, count,
			names_return));
		memswap32(names_return, names_return, count);
		free(atoms);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XGETDEFAULT:
	{
		Display *display = get_display(args[2]);
		const char *program = (void *) tswap32(args[3]);
		const char *option = (void *) tswap32(args[4]);
		assign_result32(args[1], XGetDefault(display, program, option));
		return 0;
	}
	case XGETERRORTEXT:
	{
		Display *display = get_display(args[2]);
		int code = tswap32(args[3]);
		char *buffer_return = (void *) tswap32(args[4]);
		int length = tswap32(args[5]);
		assign_result32(args[1], XGetErrorText(display, code,
			buffer_return, length));
		return 0;
	}
	case XGETGCVALUES:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		unsigned long valuemask = tswap32(args[4]);
		XGCValues *target_values_return = (void *) tswap32(args[5]);
		XGCValues values_return;
		assign_result32(args[1], XGetGCValues(display, gc, valuemask,
			&values_return));
		swap_gcvalues(target_values_return, &values_return);
		return 0;
	}
	case XGETGEOMETRY:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		Window *target_root_return = (void *) tswap32(args[4]);
		Window root_return;
		int *target_x_return = (void *) tswap32(args[5]);
		int x_return;
		int *target_y_return = (void *) tswap32(args[6]);
		int y_return;
		unsigned int *target_width_return = (void *) tswap32(args[7]);
		unsigned int width_return;
		unsigned int *target_height_return = (void *) tswap32(args[8]);
		unsigned int height_return;
		unsigned int *target_border_width_return
			= (void *) tswap32(args[9]);
		unsigned int border_width_return;
		unsigned int *target_depth_return = (void *) tswap32(args[10]);
		unsigned int depth_return;
		assign_result32(args[1], XGetGeometry(display, d, &root_return,
			&x_return, &y_return, &width_return, &height_return,
			&border_width_return, &depth_return));
		*target_root_return = tswap32(root_return);
		*target_x_return = tswap32(x_return);
		*target_y_return = tswap32(y_return);
		*target_width_return = tswap32(width_return);
		*target_height_return = tswap32(height_return);
		*target_border_width_return = tswap32(border_width_return);
		*target_depth_return = tswap32(depth_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETIMAGE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		unsigned long plane_mask = tswap32(args[8]);
		int format = tswap32(args[9]);
		XImage *image = XGetImage(display, d, x, y, width, height,
			plane_mask, format);
		assign_result32(args[1], create_image_proxy(image));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETIMVALUES:
	{
		XIMProxy *im_proxy = (void *) args[2];
		XIM im = im_proxy->native;
		unsigned *vargs = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		if (count != 1)
			warnx("XGetIMValues called with %d pairs.", count);
		memswap32(vargs, vargs, 2 * count);
		char *vargs0 = (char *) vargs[0];
		unsigned vargs1 = vargs[1];
		char *invalid = XGetIMValues(im, vargs0, vargs1, NULL);
		if (invalid)
			warnx("XGetIMValues returned error.");
		assign_result32(args[1], invalid);
		if (!strcmp(vargs0, XNQueryInputStyle)) {
			XIMStyles **target_styles = (XIMStyles **) vargs1;
			XIMStyles *styles = *target_styles;
			*target_styles = tswap32(styles);
			unsigned short count_styles = styles->count_styles;
			styles->count_styles = tswap16(styles->count_styles);
			memswap32(styles->supported_styles,
				styles->supported_styles, count_styles);
			styles->supported_styles
				= tswap32(styles->supported_styles);
		}
		else
			warnx("XGetIMValues unsupported argument %s.",
				vargs0);
		return 0;
	}
	case XGETINPUTFOCUS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window *target_focus_return = (void *) tswap32(args[3]);
		Window focus_return;
		int *target_revert_to_return = (void *) tswap32(args[4]);
		int revert_to_return;
		assign_result32(args[1], XGetInputFocus(display, &focus_return,
			&revert_to_return));
		*target_focus_return = tswap32(focus_return);
		*target_revert_to_return = tswap32(revert_to_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETKEYBOARDMAPPING:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		KeyCode first_keycode = tswap32(args[3]);
		int keycode_count = tswap32(args[4]);
		int *target_keysyms_per_keycode_return
			= (void *) tswap32(args[5]);
		int keysyms_per_keycode_return;
		KeySym *mapping = XGetKeyboardMapping(display, first_keycode,
			keycode_count, &keysyms_per_keycode_return);
		assign_result32(args[1], mapping);
		memswap32(mapping, mapping,
			keycode_count * keysyms_per_keycode_return);
		*target_keysyms_per_keycode_return
			= tswap32(keysyms_per_keycode_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETMODIFIERMAPPING:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		XModifierKeymap *modmap = XGetModifierMapping(display);
		assign_result32(args[1], modmap);
		memswap32(modmap, modmap, size32of(XModifierKeymap));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETSELECTIONOWNER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Atom selection = tswap32(args[3]);
		assign_result32(args[1], XGetSelectionOwner(display,
			selection));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETVISUALINFO:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		long vinfo_mask = tswap32(args[3]);
		XVisualInfo *target_vinfo_template = (void *) tswap32(args[4]);
		XVisualInfo vinfo_template;
		memswap32(&vinfo_template, target_vinfo_template,
			size32of(XVisualInfo));
		int *target_nitems_return = (void *) tswap32(args[5]);
		int nitems_return;
		XVisualInfo *info = XGetVisualInfo(display, vinfo_mask,
			&vinfo_template, &nitems_return);
		int i;
		for (i = 0; i < nitems_return; i++)
			set_visual_proxy(&info[i].visual, proxy);
		memswap32(info, info, nitems_return * size32of(XVisualInfo));
		assign_result32(args[1], info);
		*target_nitems_return = tswap32(nitems_return);
		return 0;
	}
	case XGETWINDOWATTRIBUTES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		XWindowAttributes *target_window_attributes_return
			= (void *) tswap32(args[4]);
		XWindowAttributes window_attributes_return;
		assign_result32(args[1], XGetWindowAttributes(display, w,
			&window_attributes_return));
		set_visual_proxy(&window_attributes_return.visual, proxy);
		set_screen_proxy(&window_attributes_return.screen, proxy);
		memswap32(target_window_attributes_return,
			&window_attributes_return, size32of(XWindowAttributes));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGETWINDOWPROPERTY:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Atom property = tswap32(args[4]);
		long long_offset = tswap32(args[5]);
		long long_length = tswap32(args[6]);
		Bool delete = tswap32(args[7]);
		Atom req_type = tswap32(args[8]);
		Atom *target_actual_type_return = (void *) tswap32(args[9]);
		Atom actual_type_return;
		int *target_actual_format_return = (void *) tswap32(args[10]);
		int actual_format_return;
		unsigned long *target_nitems_return
			= (void *) tswap32(args[11]);
		unsigned long nitems_return;
		unsigned long *target_bytes_after_return
			= (void *) tswap32(args[12]);
		unsigned long bytes_after_return;
		unsigned char **target_prop_return = (void *) tswap32(args[13]);
		unsigned char *prop_return;
		assign_result32(args[1], XGetWindowProperty(display, w,
			property, long_offset, long_length, delete, req_type,
			&actual_type_return, &actual_format_return,
			&nitems_return, &bytes_after_return, &prop_return));
		*target_actual_type_return = tswap32(actual_type_return);
		*target_actual_format_return = tswap32(actual_format_return);
		*target_nitems_return = tswap32(nitems_return);
		*target_bytes_after_return = tswap32(bytes_after_return);
		*target_prop_return = tswap32(prop_return);
		switch (actual_format_return) {
		case 16:
			memswap16(prop_return, prop_return, nitems_return);
			break;
		case 32:
			memswap32(prop_return, prop_return, nitems_return);
			break;
		}
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGRABKEYBOARD:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window grab_window = tswap32(args[3]);
		Bool owner_events = tswap32(args[4]);
		int pointer_mode = tswap32(args[5]);
		int keyboard_mode = tswap32(args[6]);
		Time time = tswap32(args[7]);
		assign_result32(args[1], XGrabKeyboard(display, grab_window,
			owner_events, pointer_mode, keyboard_mode, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGRABPOINTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window grab_window = tswap32(args[3]);
		Bool owner_events = tswap32(args[4]);
		unsigned int event_mask = tswap32(args[5]);
		int pointer_mode = tswap32(args[6]);
		int keyboard_mode = tswap32(args[7]);
		Window confine_to = tswap32(args[8]);
		Cursor cursor = tswap32(args[9]);
		Time time = tswap32(args[10]);
		assign_result32(args[1], XGrabPointer(display, grab_window,
			owner_events, event_mask, pointer_mode, keyboard_mode,
			confine_to, cursor, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XGRABSERVER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		assign_result32(args[1], XGrabServer(display));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XIFEVENT:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		Display *display = get_display(args[2]);
		XEvent *target_event_return = (void *) tswap32(args[3]);
		XEvent event_return;
		Bool (*predicate)(Display *, XEvent *, XPointer)
			= (void *) args[4];
		XPointer arg = (void *) args[5];
		predicate_data pdata;
		pdata.env = new_env;
		pdata.callback_args[0] = (unsigned) predicate;
		pdata.callback_args[3] = (unsigned) arg;
		emu_set_return_value(new_env, (unsigned) pdata.callback_args);
		assign_result32(args[1], XIfEvent(display, &event_return,
			predicate_handler, (XPointer) &pdata));
		host_to_target_event(target_event_return, &event_return);
		emu_cpu_free(new_env);
		return 0;
	}
	case XINITIMAGE: {
		XImage *target_image = (void *) tswap32(args[2]);
		XImage image;
		memswap32(&image, target_image, 12);
		memcpy(&image.red_mask, &target_image->red_mask,
			3 * sizeof(unsigned long));
		assign_result32(args[1], XInitImage(&image));
		target_image->bytes_per_line = tswap32(image.bytes_per_line);
		target_image->obdata = (XPointer) -1;
		return 0;
	}
	case _XINITIMAGEFUNCPTRS:
	{
		XImageProxy *image_proxy = (void *) tswap32(args[1]);
		if (!image_proxy->initial_sync)
			warnx("_XInitImageFuncPtrs already synced.");
		XImage *image = get_image(image_proxy);
		_XInitImageFuncPtrs(image);
		return 0;
	}
	case XINTERNATOM:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		void *atom_name = (void *) tswap32(args[3]);
		Bool only_if_exists = tswap32(args[4]);
		assign_result32(args[1], XInternAtom(display, atom_name,
			only_if_exists));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XINTERNATOMS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		char **target_names = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		char **names = malloc(count * sizeof(char *));
		memswap32(names, target_names, count);
		Bool onlyIfExists = tswap32(args[5]);
		Atom *target_atoms_return = (void *) tswap32(args[6]);
		Atom *atoms_return = malloc(count * sizeof(Atom));
		assign_result32(args[1], XInternAtoms(dpy, names, count,
			onlyIfExists, atoms_return));
		memswap32(target_atoms_return, atoms_return, count);
		free(names);
		free(atoms_return);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XINTERSECTREGION:
	{
		Region sra = (void *) args[2];
		Region srb = (void *) args[3];
		Region dr_return = (void *) args[4];
		assign_result32(args[1], XIntersectRegion(sra, srb, dr_return));
		return 0;
	}
	case XKBBELL:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Window win = tswap32(args[3]);
		int percent = tswap32(args[4]);
		Atom name = tswap32(args[5]);
		assign_result32(args[1], XkbBell(dpy, win, percent, name));
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBGETMAP: {
		DisplayProxy *raw_proxy = (void *) args[2];
		DisplayProxy *proxy = tswap32(raw_proxy);
		Display *dpy = proxy->native;
		unsigned int which = tswap32(args[3]);
		unsigned int deviceSpec = tswap32(args[4]);
		XkbDescPtr desc = XkbGetMap(dpy, which, deviceSpec);
		XkbDescRecProxy *desc_proxy = malloc(sizeof(XkbDescRecProxy));
		desc_proxy->dpy = raw_proxy;
if (dpy != desc->dpy)
warnx("desc display is different");
		desc_proxy->flags = tswap16(desc->flags);
		desc_proxy->device_spec = tswap16(desc->device_spec);
		desc_proxy->min_key_code = desc->min_key_code;
		desc_proxy->max_key_code = desc->max_key_code;
		desc_proxy->server = NULL;
		desc_proxy->map = NULL;
		desc_proxy->native = desc;
		update_xkb_desc_proxy(desc_proxy, which);
		if (desc->ctrls)
			warnx("desc->ctrls not null");
		else
			desc_proxy->ctrls = NULL;
		if (desc->indicators)
			warnx("desc->indicators not null");
		else
			desc_proxy->indicators = NULL;
		if (desc->names)
			warnx("desc->names not null");
		else
			desc_proxy->names = NULL;
		if (desc->compat)
			warnx("desc->compat not null");
		else
			desc_proxy->compat = NULL;
		if (desc->geom)
			warnx("desc->geom not null");
		else
			desc_proxy->geom = NULL;
		assign_result32(args[1], desc_proxy);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBGETNAMES: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned int which = tswap32(args[3]);
		XkbDescRecProxy *desc_proxy = (void *) tswap32(args[4]);
		XkbDescPtr desc = desc_proxy->native;
		Status status = XkbGetNames(dpy, which, desc);
		assign_result32(args[1], status);
		if (status != Success)
			warnx("XkbGetNames failed");
		XkbNamesPtr names = desc->names;
		XkbNamesRec *names_proxy;
		if (!desc_proxy->names && names) {
			names_proxy = calloc(1, sizeof(XkbNamesRec));
			desc_proxy->names = tswap32(names_proxy);
		}
		else
			names_proxy = tswap32(desc_proxy->names);
		static unsigned int UNSUPPORTED = ~(XkbKeycodesNameMask
			| XkbGeometryNameMask | XkbSymbolsNameMask
			| XkbPhysSymbolsNameMask | XkbTypesNameMask
			| XkbCompatNameMask | XkbKeyTypeNamesMask
			| XkbKTLevelNamesMask | XkbIndicatorNamesMask
			| XkbKeyNamesMask | XkbKeyAliasesMask
			| XkbVirtualModNamesMask | XkbGroupNamesMask
			| XkbRGNamesMask);
		if (which & XkbKeycodesNameMask)
			warnx("XkbKeycodesNameMask requested");
		if (which & XkbGeometryNameMask)
			warnx("XkbGeometryNameMask requested");
		if (which & XkbSymbolsNameMask)
			warnx("XkbSymbolsNameMask requested");
		if (which & XkbPhysSymbolsNameMask)
			warnx("XkbPhysSymbolsNameMask requested");
		if (which & XkbTypesNameMask)
			warnx("XkbTypesNameMask requested");
		if (which & XkbCompatNameMask)
			warnx("XkbCompatNameMask requested");
		if (which & XkbKeyTypeNamesMask)
			warnx("XkbKeyTypeNamesMask requested");
		if (which & XkbKTLevelNamesMask)
			warnx("XkbKTLevelNamesMask requested");
		if (which & XkbIndicatorNamesMask)
			warnx("XkbIndicatorNamesMask requested");
		if (which & XkbKeyNamesMask)
			warnx("XkbKeyNamesMask requested");
		if (which & XkbKeyAliasesMask)
			warnx("XkbKeyAliasesMask requested");
		if (which & XkbVirtualModNamesMask)
			memswap32(&names_proxy->vmods, &names->vmods,
				XkbNumVirtualMods);
		if (which & XkbGroupNamesMask)
			memswap32(&names_proxy->groups, &names->groups,
				XkbNumKbdGroups);
		if (which & XkbRGNamesMask)
			warnx("XkbRGNamesMask requested");
		if (which & UNSUPPORTED)
			warnx("Unsupported XkbGetNames flags");
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBGETSTATE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned int deviceSpec = tswap32(args[3]);
		XkbStatePtr rtrnState = (void *) tswap32(args[4]);
		assign_result32(args[1], XkbGetState(dpy, deviceSpec,
			rtrnState));
		rtrnState->base_group = tswap16(rtrnState->base_group);
		rtrnState->latched_group = tswap16(rtrnState->latched_group);
		rtrnState->ptr_buttons = tswap16(rtrnState->ptr_buttons);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBGETUPDATEDMAP: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned int which = tswap32(args[3]);
		XkbDescRecProxy *desc_proxy = (void *) tswap32(args[4]);
		XkbDescPtr desc = desc_proxy->native;
		Status status = XkbGetUpdatedMap(dpy, which, desc);
		assign_result32(args[1], status);
		if (status != Success)
			warnx("XkbGetUpdatedMap failed");
		update_xkb_desc_proxy(desc_proxy, which);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBKEYSYMTOMODIFIERS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		KeySym ks = tswap32(args[3]);
		assign_result32(args[1], XkbKeysymToModifiers(dpy, ks));
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBLIBRARYVERSION:
	{
		int *target_libMajorRtrn = (void *) tswap32(args[2]);
		int libMajorRtrn = tswap32(*target_libMajorRtrn);
		int *target_libMinorRtrn = (void *) tswap32(args[3]);
		int libMinorRtrn = tswap32(*target_libMinorRtrn);
		assign_result32(args[1], XkbLibraryVersion(&libMajorRtrn,
			&libMinorRtrn));
		*target_libMajorRtrn = tswap32(libMajorRtrn);
		*target_libMinorRtrn = tswap32(libMinorRtrn);
		return 0;
	}
	case XKBQUERYEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_opcodeReturn = (void *) tswap32(args[3]);
		int opcodeReturn;
		int *target_eventBaseReturn = (void *) tswap32(args[4]);
		int eventBaseReturn;
		int *target_errorBaseReturn = (void *) tswap32(args[5]);
		int errorBaseReturn;
		int *target_majorRtrn = (void *) tswap32(args[6]);
		int majorRtrn = tswap32(*target_majorRtrn);
		int *target_minorRtrn = (void *) tswap32(args[7]);
		int minorRtrn = tswap32(*target_minorRtrn);
		assign_result32(args[1], XkbQueryExtension(dpy, &opcodeReturn,
			&eventBaseReturn, &errorBaseReturn, &majorRtrn,
			&minorRtrn));
		if (target_opcodeReturn)
			*target_opcodeReturn = tswap32(opcodeReturn);
		*target_eventBaseReturn = tswap32(eventBaseReturn);
		if (target_errorBaseReturn)
			*target_errorBaseReturn = tswap32(errorBaseReturn);
		*target_majorRtrn = tswap32(majorRtrn);
		*target_minorRtrn = tswap32(minorRtrn);
		proxy->xkb_event_base = eventBaseReturn;
		return 0;
	}
	case XKBSELECTEVENTDETAILS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned int deviceID = tswap32(args[3]);
		unsigned int eventType = tswap32(args[4]);
		unsigned long affect = tswap32(args[5]);
		unsigned long details = tswap32(args[6]);
		assign_result32(args[1], XkbSelectEventDetails(dpy, deviceID,
			eventType, affect, details));
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBSELECTEVENTS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		unsigned int deviceID = tswap32(args[3]);
		unsigned int affect = tswap32(args[4]);
		unsigned int values = tswap32(args[5]);
		assign_result32(args[1], XkbSelectEvents(dpy, deviceID, affect,
			values));
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBSETDETECTABLEAUTOREPEAT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		Bool detectable = tswap32(args[3]);
		Bool *target_supported = (void *) tswap32(args[4]);
		Bool supported;
		assign_result32(args[1], XkbSetDetectableAutoRepeat(dpy,
			detectable, &supported));
		if (target_supported)
			*target_supported = tswap32(supported);
		proxy->request = tswap32(dpy->request);
		return 0;
	}
	case XKBTRANSLATEKEYSYM:
	{
		Display *dpy = get_display(args[2]);
		KeySym *target_sym_return = (void *) tswap32(args[3]);
		KeySym sym_return = tswap32(*target_sym_return);
		unsigned int modifiers = tswap32(args[4]);
		char *buffer = (void *) tswap32(args[5]);
		int nbytes = tswap32(args[6]);
		int *target_extra_rtrn = (void *) tswap32(args[7]);
		int extra_rtrn;
		assign_result32(args[1], XkbTranslateKeySym(dpy, &sym_return,
			modifiers, buffer, nbytes, &extra_rtrn));
		*target_sym_return = tswap32(sym_return);
		if (target_extra_rtrn)
			*target_extra_rtrn = tswap32(extra_rtrn);
		return 0;
	}
	case XKBUSEEXTENSION:
	{
		Display *dpy = get_display(args[2]);
		int *target_major_rtrn = (void *) tswap32(args[3]);
		int major_rtrn;
		int *target_minor_rtrn = (void *) tswap32(args[4]);
		int minor_rtrn;
		assign_result32(args[1], XkbUseExtension(dpy, &major_rtrn,
			&minor_rtrn));
		if (target_major_rtrn)
			*target_major_rtrn = tswap32(major_rtrn);
		if (target_minor_rtrn)
			*target_minor_rtrn = tswap32(minor_rtrn);
		return 0;
	}
	case XKEYCODETOKEYSYM:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		KeyCode keycode = tswap32(args[3]);
		int index = tswap32(args[4]);
		assign_result32(args[1], XKeycodeToKeysym(display, keycode,
			index));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XKEYSYMTOKEYCODE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		KeySym keysym = tswap32(args[3]);
		KeyCode *result = (void *) tswap32(args[1]);
		*result = XKeysymToKeycode(display, keysym);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XKEYSYMTOSTRING:
	{
		KeySym keysym = tswap32(args[2]);
		assign_result32(args[1], XKeysymToString(keysym));
		return 0;
	}
	case XLISTDEPTHS:
	{
		Display *display = get_display(args[2]);
		int screen_number = tswap32(args[3]);
		int *target_count_return = (void *) tswap32(args[4]);
		int count_return;
		int *depths = XListDepths(display, screen_number,
			&count_return);
		assign_result32(args[1], depths);
		if (depths) {
			memswap32(depths, depths, count_return);
			*target_count_return = tswap32(count_return);
		}
		return 0;
	}
	case XLISTPIXMAPFORMATS:
	{
		Display *display = get_display(args[2]);
		int *target_count_return = (void *) tswap32(args[3]);
		int count_return;
		XPixmapFormatValues *formats = XListPixmapFormats(display,
			&count_return);
		assign_result32(args[1], formats);
		if (formats) {
			memswap32(formats, formats, count_return
				* size32of(XPixmapFormatValues));
			*target_count_return = tswap32(count_return);
		}
		return 0;
	}
	case XLOCALEOFIM:
	{
warnx("XLocaleOfIM called!");
		XIMProxy *im_proxy = (void *) args[2];
		XIM im = im_proxy->native;
		assign_result32(args[1], XLocaleOfIM(im));
		return 0;
	}
	case XLOOKUPCOLOR:
	{
		Display *display = get_display(args[2]);
		Colormap colormap = tswap32(args[3]);
		void *color_name = (void *) tswap32(args[4]);
		XColor *target_exact_def_return = (void *) tswap32(args[5]);
		XColor exact_def_return;
		XColor *target_screen_def_return = (void *) tswap32(args[6]);
		XColor screen_def_return;
		assign_result32(args[1], XLookupColor(display, colormap,
			color_name, &exact_def_return, &screen_def_return));
		swap_color(target_exact_def_return, &exact_def_return);
		swap_color(target_screen_def_return, &screen_def_return);
		return 0;
	}
	case XLOOKUPKEYSYM:
	{
		XKeyEvent *target_key_event = (void *) tswap32(args[2]);
		XKeyEvent key_event;
		memswap32(&key_event, target_key_event, size32of(XKeyEvent));
		DisplayProxy *proxy = (DisplayProxy *) key_event.display;
		Display *display = proxy->native;
		key_event.display = display;
		int index = tswap32(args[3]);
		assign_result32(args[1], XLookupKeysym(&key_event, index));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XLOOKUPSTRING:
	{
		XKeyEvent *target_event_struct = (void *) tswap32(args[2]);
		XKeyEvent event_struct;
		to_native_key_event(&event_struct, target_event_struct);
		char *buffer_return = (void *) tswap32(args[3]);
		int bytes_buffer = tswap32(args[4]);
		KeySym *target_keysym_return = (void *) tswap32(args[5]);
		KeySym keysym_return;
// XComposeStatus *status_in_out, assuming null
		XComposeStatus *status_in_out = (void *) tswap32(args[6]);
		if (status_in_out != NULL)
			warnx("status_in_out not null");
		assign_result32(args[1], XLookupString(&event_struct,
			buffer_return, bytes_buffer, &keysym_return,
			status_in_out));
		if (target_keysym_return)
			*target_keysym_return = tswap32(keysym_return);
		return 0;
	}
	case XLOWERWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XLowerWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XMAPSUBWINDOWS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XMapSubwindows(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XMAPWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XMapWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XMAXREQUESTSIZE:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XMaxRequestSize(display));
		return 0;
	}
	case XMBLOOKUPSTRING:
	{
		XIC ic = (void *) args[2];
		XKeyPressedEvent *target_event = (void *) tswap32(args[3]);
		XKeyPressedEvent event;
		to_native_key_event(&event, target_event);
		char *buffer_return = (void *) tswap32(args[4]);
		int bytes_buffer = tswap32(args[5]);
		KeySym *target_keysym_return = (void *) tswap32(args[6]);
		KeySym keysym_return = tswap32(*target_keysym_return);
		Status *target_status_return = (void *) tswap32(args[7]);
		Status status_return;
		assign_result32(args[1], XmbLookupString(ic, &event,
			buffer_return, bytes_buffer, &keysym_return,
			&status_return));
		*target_keysym_return = tswap32(keysym_return);
		*target_status_return = tswap32(status_return);
		return 0;
	}
	case XMBSETWMPROPERTIES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window w = tswap32(args[2]);
		const char *window_name = (void *) tswap32(args[3]);
		const char *icon_name = (void *) tswap32(args[4]);
		char **target_argv = (void *) tswap32(args[5]);
		int argc = tswap32(args[6]);
		char **argv = target_argv ? malloc(argc * sizeof(char *))
			: NULL;
		memswap32(argv, target_argv, argc);
		XSizeHints *target_normal_hints = (void *) tswap32(args[7]);
		XSizeHints *normal_hints;
		if (target_normal_hints) {
			normal_hints = alloca(sizeof(XSizeHints));
			memswap32(normal_hints, target_normal_hints,
				size32of(XSizeHints));
		}
		else
			normal_hints = NULL;
		XWMHints *target_wm_hints = (void *) tswap32(args[8]);
		XWMHints *wm_hints;
		if (target_wm_hints) {
			wm_hints = alloca(sizeof(XWMHints));
			memswap32(wm_hints, target_wm_hints,
				size32of(XWMHints));
		}
		else
			wm_hints = NULL;
		XClassHint *target_class_hints = (void *) tswap32(args[9]);
		XClassHint *class_hints;
		if (target_class_hints) {
			class_hints = alloca(sizeof(XClassHint));
			memswap32(class_hints, target_class_hints,
				size32of(XClassHint));
		}
		else
			class_hints = NULL;
		XmbSetWMProperties(display, w, window_name, icon_name, argv,
			argc, normal_hints, wm_hints, class_hints);
		free(argv);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XMBTEXTLISTTOTEXTPROPERTY:
	{
		Display *display = get_display(args[2]);
		void *target_list = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		void *list = malloc(count * sizeof(char *));
		memswap32(list, target_list, count);
		XICCEncodingStyle style = tswap32(args[5]);
		XTextProperty *target_text_prop_return
			= (void *) tswap32(args[6]);
		XTextProperty text_prop_return;
		assign_result32(args[1], XmbTextListToTextProperty(display,
			list, count, style, &text_prop_return));
		swap_XTextProperty(target_text_prop_return, &text_prop_return,
			&text_prop_return);
		free(list);
		return 0;
	}
	case XMOVERESIZEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		unsigned int width = tswap32(args[6]);
		unsigned int height = tswap32(args[7]);
		assign_result32(args[1], XMoveResizeWindow(display, w, x, y,
			width, height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XMOVEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		int x = tswap32(args[4]);
		int y = tswap32(args[5]);
		assign_result32(args[1], XMoveWindow(display, w, x, y));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XNEXTEVENT:
	{
		Display *display = get_display(args[2]);
		XEvent *target_event_return = (void *) tswap32(args[3]);
		XEvent event_return;
		assign_result32(args[1], XNextEvent(display, &event_return));
		host_to_target_event(target_event_return, &event_return);
		return 0;
	}
	case XOPENDISPLAY:
	{
		const char *display_name = (void *) tswap32(args[2]);
		Display *dpy = XOpenDisplay(display_name);
		if (!dpy) {
			assign_result32(args[1], NULL);
			return 0;
		}
		DisplayProxy *proxy = malloc(sizeof(DisplayProxy));
g_display_proxy = proxy;
warnx("opened display %p", proxy);
		XExtCodes *codes = XAddExtension(dpy);
		proxy->extension = codes->extension;
		XEDataObject object = { .display = dpy };
		XExtData **structure = XEHeadOfExtensionList(object);
		XExtData *ext_data = malloc(sizeof(XExtData));
		ext_data->number = codes->extension;
		ext_data->free_private = free_display_proxy;
		ext_data->private_data = (XPointer) proxy;
		XAddToExtensionList(structure, ext_data);

		proxy->fd = tswap32(dpy->fd);
		proxy->vendor = tswap32(dpy->vendor);
		proxy->byte_order = tswap32(dpy->byte_order);
		proxy->bitmap_bit_order = tswap32(dpy->bitmap_bit_order);
		proxy->release = tswap32(dpy->release);
		proxy->request = tswap32(dpy->request);
		proxy->bufmax = tswap32(dpy->bufmax);
		if (dpy->synchandler)
			warnx("synchandler not null.");
		proxy->synchandler = dpy->synchandler;
		proxy->display_name = tswap32(dpy->display_name);
		proxy->default_screen = tswap32(dpy->default_screen);
		proxy->nscreens = tswap32(dpy->nscreens);
		if (dpy->nscreens != 1)
			warnx("more than one screen: %d", dpy->nscreens);

		Screen *screen = dpy->screens;
		ScreenProxy *screen_proxy = malloc(sizeof(ScreenProxy));
		proxy->screens = tswap32(screen_proxy);
		screen_proxy->native = screen;
		screen_proxy->display = tswap32(proxy);
		screen_proxy->root = tswap32(screen->root);
		screen_proxy->width = tswap32(screen->width);
		screen_proxy->height = tswap32(screen->height);
		screen_proxy->ndepths = tswap32(screen->ndepths);

		Depth *depths = screen->depths;
		Depth *target_depths = malloc(screen->ndepths * sizeof(Depth));
		screen_proxy->depths = tswap32(target_depths);
		int i;
		for (i = 0; i < screen->ndepths; i++) {
			target_depths[i].depth = tswap32(depths[i].depth);
			int nvisuals = depths[i].nvisuals;
			target_depths[i].nvisuals = tswap32(nvisuals);
			Visual *visuals = depths[i].visuals;
			VisualProxy *target_visuals = malloc(nvisuals
				* sizeof(VisualProxy));
			target_depths[i].visuals
				= (Visual *) tswap32(target_visuals);
			int j;
			for (j = 0; j < nvisuals; j++)
				create_visual_proxy(target_visuals + j,
					visuals + j, proxy);
		}

		screen_proxy->root_depth = tswap32(screen->root_depth);
		object.screen = screen;
		structure = XEHeadOfExtensionList(object);
		ext_data = malloc(sizeof(XExtData));
		ext_data->number = codes->extension;
		ext_data->free_private = free_screen_proxy;
		ext_data->private_data = (XPointer) screen_proxy;
		XAddToExtensionList(structure, ext_data);

		Visual *visual = screen->root_visual;
		set_visual_proxy(&visual, proxy);
		screen_proxy->root_visual = (VisualProxy *) tswap32(visual);

		screen_proxy->cmap = tswap32(screen->cmap);
		screen_proxy->white_pixel = tswap32(screen->white_pixel);
		screen_proxy->black_pixel = tswap32(screen->black_pixel);

		proxy->async_handlers = NULL;

		struct _XLockPtrs *lock_fns = malloc(sizeof(struct _XLockPtrs));
		memcpy(lock_fns, stub_lock_fns, sizeof(struct _XLockPtrs));
		proxy->lock_fns = tswap32(lock_fns);

		proxy->native = dpy;
		proxy->ext_proxies = NULL;
		proxy->async_handler_set = False;
		proxy->sent_requests = NULL;
		proxy->xkb_event_base = 0;
		proxy->target_ext = NULL;

		assign_result32(args[1], proxy);
		return 0;
	}
	case XOPENIM:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		struct _XrmHashBucketRec *rdb = (void *) args[3];
		if (rdb)
			warnx("XOpenIM rdb is not null.");
		char *res_name = (void *) args[4];
		if (res_name)
			warnx("XOpenIM res_name is not null.");
		char *res_class = (void *) args[5];
		if (res_class)
			warnx("XOpenIM res_class is not null.");
		XIMProxy **result = (void *) tswap32(args[1]);
		XIM im = XOpenIM(dpy, rdb, res_name, res_class);
		if (im) {
			XIMProxy *im_proxy = malloc(sizeof(XIMProxy));
			im_proxy->native = im;
			im_proxy->display = proxy;
			*result = im_proxy;
		}
		else
			*result = NULL;
		return 0;
	}
	case XPENDING:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XPending(display));
		return 0;
	}
	case XPUTIMAGE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Drawable d = tswap32(args[3]);
		GC gc = (void *) args[4];
		XImageProxy *image_proxy = (void *) tswap32(args[5]);
		XImage *image;
		switch ((int) image_proxy->obdata) {
		case 0:
			image = get_image(image_proxy);
			break;
		case -1:
			image = alloca(sizeof(XImage));
			memswap32(image, image_proxy, 12);
			memcpy(&image->red_mask, &image_proxy->red_mask,
				3 * sizeof(unsigned long));
			memset(&image->f, 0, sizeof(struct funcs));
			break;
		default:
			image = NULL;
			warnx("unknown XImage obdata.");
		}
		int src_x = tswap32(args[6]);
		int src_y = tswap32(args[7]);
		int dest_x = tswap32(args[8]);
		int dest_y = tswap32(args[9]);
		unsigned int width = tswap32(args[10]);
		unsigned int height = tswap32(args[11]);
		assign_result32(args[1], XPutImage(display, d, gc, image, src_x,
			src_y, dest_x, dest_y, width, height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XQUERYCOLORS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Colormap colormap = tswap32(args[3]);
		XColor *target_defs_in_out = (void *) tswap32(args[4]);
		int ncolors = tswap32(args[5]);
		XColor *defs_in_out = malloc(ncolors * sizeof(XColor));
		int i;
		for (i = 0; i < ncolors; i++)
			swap_color(defs_in_out + i, target_defs_in_out + i);
		assign_result32(args[1], XQueryColors(display, colormap,
			defs_in_out, ncolors));
		for (i = 0; i < ncolors; i++)
			swap_color(target_defs_in_out + i, defs_in_out + i);
		free(defs_in_out);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XQUERYEXTENSION:
	{
		Display *display = get_display(args[2]);
		const char *name = (void *) tswap32(args[3]);
		int *target_major_opcode_return = (void *) tswap32(args[4]);
		int major_opcode_return;
		int *target_first_event_return = (void *) tswap32(args[5]);
		int first_event_return;
		int *target_first_error_return = (void *) tswap32(args[6]);
		int first_error_return;
		assign_result32(args[1], XQueryExtension(display, name,
			&major_opcode_return, &first_event_return,
			&first_error_return));
		*target_major_opcode_return = tswap32(major_opcode_return);
		*target_first_event_return = tswap32(first_event_return);
		*target_first_error_return = tswap32(first_error_return);
		return 0;
	}
	case XQUERYKEYMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		char *keys_return = (void *) tswap32(args[3]);
		assign_result32(args[1], XQueryKeymap(display, keys_return));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XQUERYPOINTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Window *target_root_return = (void *) tswap32(args[4]);
		Window root_return;
		Window *target_child_return = (void *) tswap32(args[5]);
		Window child_return;
		int *target_root_x_return = (void *) tswap32(args[6]);
		int root_x_return;
		int *target_root_y_return = (void *) tswap32(args[7]);
		int root_y_return;
		int *target_win_x_return = (void *) tswap32(args[8]);
		int win_x_return;
		int *target_win_y_return = (void *) tswap32(args[9]);
		int win_y_return;
		unsigned int *target_mask_return = (void *) tswap32(args[10]);
		unsigned int mask_return;
		assign_result32(args[1], XQueryPointer(display, w, &root_return,
			&child_return, &root_x_return, &root_y_return,
			&win_x_return, &win_y_return, &mask_return));
		*target_root_return = tswap32(root_return);
		*target_child_return = tswap32(child_return);
		*target_root_x_return = tswap32(root_x_return);
		*target_root_y_return = tswap32(root_y_return);
		*target_win_x_return = tswap32(win_x_return);
		*target_win_y_return = tswap32(win_y_return);
		*target_mask_return = tswap32(mask_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XQUERYTREE:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Window *target_root_return = (void *) tswap32(args[4]);
		Window root_return;
		Window *target_parent_return = (void *) tswap32(args[5]);
		Window parent_return;
		Window **target_children_return = (void *) tswap32(args[6]);
		Window *children_return;
		unsigned int *target_nchildren_return
			= (void *) tswap32(args[7]);
		unsigned int nchildren_return;
		assign_result32(args[1], XQueryTree(display, w, &root_return,
			&parent_return, &children_return, &nchildren_return));
		*target_root_return = tswap32(root_return);
		*target_parent_return = tswap32(parent_return);
		memswap32(children_return, children_return, nchildren_return);
		*target_children_return = tswap32(children_return);
		*target_nchildren_return = tswap32(nchildren_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XRAISEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XRaiseWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XRECONFIGUREWMWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		int screen_number = tswap32(args[4]);
		unsigned int mask = tswap32(args[5]);
		XWindowChanges *target_changes = (void *) tswap32(args[6]);
		XWindowChanges changes;
		memswap32(&changes, target_changes, size32of(XWindowChanges));
		assign_result32(args[1], XReconfigureWMWindow(display, w,
			screen_number, mask, &changes));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XREPARENTWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Window parent = tswap32(args[4]);
		int x = tswap32(args[5]);
		int y = tswap32(args[6]);
		assign_result32(args[1], XReparentWindow(display, w, parent, x,
			y));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XRESIZEWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		unsigned int width = tswap32(args[4]);
		unsigned int height = tswap32(args[5]);
		assign_result32(args[1], XResizeWindow(display, w, width,
			height));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XRESOURCEMANAGERSTRING:
	{
		Display *display = get_display(args[2]);
		assign_result32(args[1], XResourceManagerString(display));
		return 0;
	}
	case XRMCOMBINEDATABASE:
	{
		XrmDatabase source_db = (void *) args[1];
		XrmDatabase *target_db = (void *) tswap32(args[2]);
		Bool override = tswap32(args[3]);
		XrmCombineDatabase(source_db, target_db, override);
		return 0;
	}
	case XRMCOMBINEFILEDATABASE:
	{
		void *filename = (void *) tswap32(args[2]);
		XrmDatabase *target = (void *) tswap32(args[3]);
		Bool override = tswap32(args[4]);
		assign_result32(args[1], XrmCombineFileDatabase(filename,
			target, override));
		return 0;
	}
	case XRMDESTROYDATABASE:
	{
		XrmDatabase database = (void *) args[1];
		XrmDestroyDatabase(database);
		return 0;
	}
	case XRMGETDATABASE:
	{
		Display *display = get_display(args[2]);
		XrmDatabase *result = (void *) tswap32(args[1]);
		*result = XrmGetDatabase(display);
		return 0;
	}
	case XRMGETFILEDATABASE:
	{
		void *filename = (void *) tswap32(args[2]);
		XrmDatabase *result = (void *) tswap32(args[1]);
		*result = XrmGetFileDatabase(filename);
		return 0;
	}
	case XRMGETRESOURCE:
	{
warnx("xrmgetresource");
		XrmDatabase database = (void *) args[2];
		void *str_name = (void *) tswap32(args[3]);
		void *str_class = (void *) tswap32(args[4]);
// char **str_type_return
		void *str_type_return = (void *) tswap32(args[5]);
// XrmValue *value_return
		void *value_return = (void *) tswap32(args[6]);
		assign_result32(args[1], XrmGetResource(database, str_name,
			str_class, str_type_return, value_return));
		return 0;
	}
	case XRMGETSTRINGDATABASE:
	{
		void *data = (void *) tswap32(args[2]);
		XrmDatabase *result = (void *) tswap32(args[1]);
		*result = XrmGetStringDatabase(data);
		return 0;
	}
	case XRMINITIALIZE:
	{
		XrmInitialize();
		return 0;
	}
	case XRMMERGEDATABASES:
	{
		XrmDatabase source_db = (void *) args[1];
		XrmDatabase *target_db = (void *) tswap32(args[2]);
		XrmMergeDatabases(source_db, target_db);
		return 0;
	}
	case XRMPARSECOMMAND:
	{
		XrmDatabase *database = (void *) tswap32(args[1]);
		void *target_table = (void *) tswap32(args[2]);
		int table_count = tswap32(args[3]);
		void *table = malloc(table_count * sizeof(XrmOptionDescRec));
		memswap32(table, target_table, table_count
			* size32of(XrmOptionDescRec));
		void *name = (void *) tswap32(args[4]);
		int *target_argc_in_out = (void *) tswap32(args[5]);
		int argc_in_out = tswap32(*target_argc_in_out);
		int old_argc_in_out = argc_in_out;
		void *target_argv_in_out = (void *) tswap32(args[6]);
		void *argv_in_out = malloc(argc_in_out * sizeof(char *));
		memswap32(argv_in_out, target_argv_in_out, argc_in_out);
		XrmParseCommand(database, table, table_count, name,
			&argc_in_out, argv_in_out);
		if (argc_in_out != old_argc_in_out) {
			*target_argc_in_out = tswap32(argc_in_out);
			memswap32(target_argv_in_out, argv_in_out, argc_in_out);
		}
		free(table);
		free(argv_in_out);
		return 0;
	}
	case XRMPERMSTRINGTOQUARK:
	{
		void *string = (void *) tswap32(args[2]);
		XrmQuark *result = (void *) tswap32(args[1]);
		*result = XrmPermStringToQuark(string);
		return 0;
	}
	case XRMQGETRESOURCE:
	{
		XrmDatabase database = (void *) args[2];
		XrmNameList quark_name = (void *) tswap32(args[3]);
		XrmClassList quark_class = (void *) tswap32(args[4]);
		XrmRepresentation *quark_type_return
			= (void *) tswap32(args[5]);
		XrmValue *target_value_return = (void *) tswap32(args[6]);
		XrmValue value_return;
		assign_result32(args[1], XrmQGetResource(database, quark_name,
			quark_class, quark_type_return, &value_return));
		memswap32(target_value_return, &value_return,
			size32of(XrmValue));
		return 0;
	}
	case XRMQGETSEARCHLIST:
	{
		XrmDatabase database = (void *) args[2];
		XrmNameList names = (void *) tswap32(args[3]);
		XrmClassList classes = (void *) tswap32(args[4]);
		void *list_return = (void *) tswap32(args[5]);
		int list_length = tswap32(args[6]);
		assign_result32(args[1], XrmQGetSearchList(database, names,
			classes, list_return, list_length));
		return 0;
	}
	case XRMQGETSEARCHRESOURCE:
	{
		void *list = (void *) tswap32(args[2]);
		XrmName name = args[3];
		XrmClass class = args[4];
		XrmRepresentation *type_return = (void *) tswap32(args[5]);
		XrmValue *target_value_return = (void *) tswap32(args[6]);
		XrmValue value_return;
		assign_result32(args[1], XrmQGetSearchResource(list, name,
			class, type_return, &value_return));
		memswap32(target_value_return, &value_return,
			size32of(XrmValue));
		return 0;
	}
	case XRMQUARKTOSTRING:
	{
		XrmQuark quark = args[2];
		assign_result32(args[1], XrmQuarkToString(quark));
		return 0;
	}
	case XRMSETDATABASE:
	{
		Display *display = get_display(args[1]);
		XrmDatabase database = (void *) args[2];
		XrmSetDatabase(display, database);
		return 0;
	}
	case XRMSTRINGTOQUARK:
	{
		void *string = (void *) tswap32(args[2]);
		XrmQuark *result = (void *) tswap32(args[1]);
		*result = XrmStringToQuark(string);
		return 0;
	}
	case XRMUNIQUEQUARK:
	{
		XrmQuark *result = (void *) tswap32(args[1]);
		*result = XrmUniqueQuark();
		return 0;
	}
	case XSAVECONTEXT:
	{
		Display *display = get_display(args[2]);
		XID rid = tswap32(args[3]);
		XContext context = args[4];
		const char *data = (void *) args[5];
		assign_result32(args[1], XSaveContext(display, rid, context,
			data));
		return 0;
	}
	case XSCREENNUMBEROFSCREEN:
	{
		Screen *screen = get_screen(args[2]);
		assign_result32(args[1], XScreenNumberOfScreen(screen));
		return 0;
	}
	case XSCREENRESOURCESTRING:
	{
		ScreenProxy *proxy = (void *) tswap32(args[2]);
		Screen *screen = proxy->native;
		assign_result32(args[1], XScreenResourceString(screen));
		DisplayProxy *display_proxy = tswap32(proxy->display);
		display_proxy->request
			= tswap32(display_proxy->native->request);
		return 0;
	}
	case XSELECTINPUT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		long event_mask = tswap32(args[4]);
		assign_result32(args[1], XSelectInput(display, w, event_mask));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSENDEVENT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Bool propagate = tswap32(args[4]);
		long event_mask = tswap32(args[5]);
		XEvent *target_event_send = (void *) tswap32(args[6]);
		XEvent event_send;
		swap_event(&event_send, target_event_send, &event_send);
		assign_result32(args[1], XSendEvent(display, w, propagate,
			event_mask, &event_send));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETARCMODE:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		int arc_mode = tswap32(args[4]);
		assign_result32(args[1], XSetArcMode(display, gc, arc_mode));
		return 0;
	}
	case XSETBACKGROUND:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		unsigned long background = tswap32(args[4]);
		assign_result32(args[1], XSetBackground(display, gc,
			background));
		return 0;
	}
	case XSETCLASSHINT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		XClassHint *target_class_hints = (void *) tswap32(args[4]);
		XClassHint class_hints;
		memswap32(&class_hints, target_class_hints,
			size32of(XClassHint));
		assign_result32(args[1], XSetClassHint(display, w,
			&class_hints));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETCLIPMASK:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		GC gc = (void *) args[3];
		Pixmap pixmap = tswap32(args[4]);
		assign_result32(args[1], XSetClipMask(display, gc, pixmap));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETCLIPORIGIN:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		int clip_x_origin = tswap32(args[4]);
		int clip_y_origin = tswap32(args[5]);
		assign_result32(args[1], XSetClipOrigin(display, gc,
			clip_x_origin, clip_y_origin));
		return 0;
	}
	case XSETCLIPRECTANGLES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		GC gc = (void *) args[3];
		int clip_x_origin = tswap32(args[4]);
		int clip_y_origin = tswap32(args[5]);
		XRectangle *target_rectangles = (void *) tswap32(args[6]);
		int n = tswap32(args[7]);
		XRectangle *rectangles = malloc(n * sizeof(XRectangle));
		memswap16(rectangles, target_rectangles,
			n * size16of(XRectangle));
		int ordering = tswap32(args[8]);
		assign_result32(args[1], XSetClipRectangles(display, gc,
			clip_x_origin, clip_y_origin, rectangles, n, ordering));
		free(rectangles);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETDASHES: {
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		GC gc = (void *) args[3];
		int dash_offset = tswap32(args[4]);
		const char *dash_list = (void *) tswap32(args[5]);
		int n = tswap32(args[6]);
		assign_result32(args[1], XSetDashes(display, gc, dash_offset,
			dash_list, n));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETERRORHANDLER:
	{
		if (!ErrorHandler_data.env) {
			emu_cpu_env *new_env = emu_cpu_clone(env);
			ErrorHandler_data.env = new_env;
			emu_set_return_value(new_env,
				(unsigned) ErrorHandler_data.callback_args);
		}
		XErrorHandler target_handler = (void *) args[2];
		XErrorHandler handler = tswap32(target_handler);
		XErrorHandler target_previous
			= (void *) ErrorHandler_data.callback_args[0];
		Dl_info info;
		if (!dladdr(handler, &info)) {
			ErrorHandler_data.callback_args[0]
				= (unsigned) target_handler;
			handler = ErrorHandler;
		}
		XErrorHandler previous = XSetErrorHandler(handler);
		XErrorHandler *result = (void *) tswap32(args[1]);
		*result = previous == ErrorHandler ? target_previous
			: tswap32(previous);
		return 0;
	}
	case XSETFOREGROUND:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		unsigned long foreground = tswap32(args[4]);
		assign_result32(args[1], XSetForeground(display, gc,
			foreground));
		return 0;
	}
	case XSETFUNCTION:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		int function = tswap32(args[4]);
		assign_result32(args[1], XSetFunction(display, gc, function));
		return 0;
	}
	case XSETGRAPHICSEXPOSURES:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		Bool graphics_exposures = tswap32(args[4]);
		assign_result32(args[1], XSetGraphicsExposures(display, gc,
			graphics_exposures));
		return 0;
	}
	case XSETICFOCUS:
	{
		XIC ic = (void *) args[1];
		XSetICFocus(ic);
		return 0;
	}
	case XSETIMVALUES:
	{
		XIMProxy *im_proxy = (void *) args[2];
		XIM im = im_proxy->native;
		unsigned *vargs = (void *) tswap32(args[3]);
		int count = tswap32(args[4]);
		if (count != 1)
			warnx("XSetIMValues called with %d pairs.", count);
		memswap32(vargs, vargs, 2 * count);
		char *vargs0 = (char *) vargs[0];
		unsigned vargs1;
		if (!strcmp(vargs0, XNDestroyCallback)) {
			XIMCallback *target_destroy = (XIMCallback *) vargs[1];
			XIMCallback destroy;
			destroy.client_data = target_destroy->client_data;
			destroy.callback = xim_destroy;
			vargs1 = (unsigned) &destroy;
		}
		else {
			warnx("XSetIMValues unsupported argument %s.",
				vargs0);
			vargs1 = vargs[1];
		}
		char *invalid = XSetIMValues(im, vargs0, vargs1, NULL);
		if (invalid)
			warnx("XSetIMValues returned error.");
		assign_result32(args[1], invalid);
		return 0;
	}
	case XSETINPUTFOCUS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window focus = tswap32(args[3]);
		int revert_to = tswap32(args[4]);
		Time time = tswap32(args[5]);
		assign_result32(args[1], XSetInputFocus(display, focus,
			revert_to, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETIOERRORHANDLER:
	{
//		XIOErrorHandler handler = (void *) tswap32(args[2]);
		XIOErrorHandler previous = XSetIOErrorHandler(IOErrorHandler);
		assign_result32(args[1], previous);
		return 0;
	}
	case XSETLOCALEMODIFIERS:
	{
		void *modifier_list = (void *) tswap32(args[2]);
		assign_result32(args[1], XSetLocaleModifiers(modifier_list));
		return 0;
	}
	case XSETNORMALHINTS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		XSizeHints *target_hints = (void *) tswap32(args[4]);
		XSizeHints hints;
		memswap32(&hints, target_hints, size32of(XSizeHints));
		assign_result32(args[1], XSetNormalHints(display, w, &hints));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETSELECTIONOWNER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Atom selection = tswap32(args[3]);
		Window owner = tswap32(args[4]);
		Time time = tswap32(args[5]);
		assign_result32(args[1], XSetSelectionOwner(display, selection,
			owner, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETSTANDARDPROPERTIES:
	{
		Display *display = get_display(args[2]);
		Window w = tswap32(args[3]);
		void *window_name = (void *) tswap32(args[4]);
		void *icon_name = (void *) tswap32(args[5]);
		Pixmap icon_pixmap = tswap32(args[6]);
// Assuming null
		void *argv = (void *) args[7];
		if (argv != NULL)
			warnx("set standard, argv not null");
// Assuming 0
		int argc = args[8];
		XSizeHints *target_hints = (void *) tswap32(args[9]);
		XSizeHints hints;
		memswap32(&hints, target_hints, size32of(XSizeHints));
		assign_result32(args[1], XSetStandardProperties(display, w,
			window_name, icon_name, icon_pixmap, argv, argc,
			&hints));
		return 0;
	}
	case XSETSUBWINDOWMODE:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		int subwindow_mode = tswap32(args[4]);
		assign_result32(args[1], XSetSubwindowMode(display, gc,
			subwindow_mode));
		return 0;
	}
	case XSETTRANSIENTFORHINT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Window prop_window = tswap32(args[4]);
		assign_result32(args[1], XSetTransientForHint(display, w,
			prop_window));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETTSORIGIN:
	{
		Display *display = get_display(args[2]);
		GC gc = (void *) args[3];
		int ts_x_origin = tswap32(args[4]);
		int ts_y_origin = tswap32(args[5]);
		assign_result32(args[1], XSetTSOrigin(display, gc, ts_x_origin,
			ts_y_origin));
		return 0;
	}
	case XSETWINDOWBACKGROUND:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		unsigned long background_pixel = tswap32(args[4]);
		assign_result32(args[1], XSetWindowBackground(display, w,
			background_pixel));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWINDOWBACKGROUNDPIXMAP:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Pixmap background_pixmap = tswap32(args[4]);
		assign_result32(args[1], XSetWindowBackgroundPixmap(display, w,
			background_pixmap));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMHINTS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		XWMHints *target_wm_hints = (void *) tswap32(args[4]);
		XWMHints wm_hints;
		memswap32(&wm_hints, target_wm_hints, size32of(XWMHints));
		assign_result32(args[1], XSetWMHints(display, w, &wm_hints));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMICONNAME:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window w = tswap32(args[2]);
		XTextProperty *target_text_prop = (void *) tswap32(args[3]);
		XTextProperty text_prop;
		swap_XTextProperty(&text_prop, target_text_prop, &text_prop);
		XSetWMIconName(display, w, &text_prop);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMNAME:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window w = tswap32(args[2]);
		XTextProperty *target_text_prop = (void *) tswap32(args[3]);
		XTextProperty text_prop;
		swap_XTextProperty(&text_prop, target_text_prop, &text_prop);
		XSetWMName(display, w, &text_prop);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMNORMALHINTS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window w = tswap32(args[2]);
		XSizeHints *target_hints = (void *) tswap32(args[3]);
		XSizeHints hints;
		memswap32(&hints, target_hints, size32of(XSizeHints));
		XSetWMNormalHints(display, w, &hints);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMPROPERTIES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *display = proxy->native;
		Window w = tswap32(args[2]);
		XTextProperty *target_window_name = (void *) tswap32(args[3]);
		XTextProperty *window_name;
		if (target_window_name) {
			window_name = alloca(sizeof(XTextProperty));
			swap_XTextProperty(window_name, target_window_name,
				window_name);
		}
		else
			window_name = NULL;
		XTextProperty *target_icon_name = (void *) tswap32(args[4]);
		XTextProperty *icon_name;
		if (target_icon_name) {
			icon_name = alloca(sizeof(XTextProperty));
			swap_XTextProperty(icon_name, target_icon_name,
				icon_name);
		}
		else
			icon_name = NULL;
		void *target_argv = (void *) tswap32(args[5]);
		int argc = tswap32(args[6]);
		void *argv = target_argv ? malloc(argc * sizeof(char *)) : NULL;
		memswap32(argv, target_argv, argc);
		XSizeHints *target_normal_hints = (void *) tswap32(args[7]);
		XSizeHints *normal_hints;
		if (target_normal_hints) {
			normal_hints = alloca(sizeof(XSizeHints));
			memswap32(normal_hints, target_normal_hints,
				size32of(XSizeHints));
		}
		else
			normal_hints = NULL;
		XWMHints *target_wm_hints = (void *) tswap32(args[8]);
		XWMHints *wm_hints;
		if (target_wm_hints) {
			wm_hints = alloca(sizeof(XWMHints));
			memswap32(wm_hints, target_wm_hints,
				size32of(XWMHints));
		}
		else
			wm_hints = NULL;
		XClassHint *target_class_hints = (void *) tswap32(args[9]);
		XClassHint *class_hints;
		if (target_class_hints) {
			class_hints = alloca(sizeof(XClassHint));
			memswap32(class_hints, target_class_hints,
				size32of(XClassHint));
		}
		else
			class_hints = NULL;
		XSetWMProperties(display, w, window_name, icon_name, argv, argc,
			normal_hints, wm_hints, class_hints);
		free(argv);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSETWMPROTOCOLS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		Atom *target_protocols = (void *) tswap32(args[4]);
		int count = tswap32(args[5]);
		Atom *protocols = malloc(count * sizeof(Atom));
		memswap32(protocols, target_protocols, count);
		assign_result32(args[1], XSetWMProtocols(display, w, protocols,
			count));
		free(protocols);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSTRINGTOKEYSYM:
	{
		const char *string = (void *) tswap32(args[2]);
		assign_result32(args[1], XStringToKeysym(string));
		return 0;
	}
	case XSUPPORTSLOCALE:
	{
		assign_result32(args[1], XSupportsLocale());
		return 0;
	}
	case XSYNC:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Bool discard = tswap32(args[3]);
		assign_result32(args[1], XSync(display, discard));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XSYNCHRONIZE:
	{
// assuming result is discarded
		Display *display = get_display(args[2]);
		Bool onoff = tswap32(args[3]);
		assign_result32(args[1], XSynchronize(display, onoff));
		return 0;
	}
	case XTRANSLATECOORDINATES:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window src_w = tswap32(args[3]);
		Window dest_w = tswap32(args[4]);
		int src_x = tswap32(args[5]);
		int src_y = tswap32(args[6]);
		int *target_dest_x_return = (void *) tswap32(args[7]);
		int dest_x_return;
		int *target_dest_y_return = (void *) tswap32(args[8]);
		int dest_y_return;
		Window *target_child_return = (void *) tswap32(args[9]);
		Window child_return;
		assign_result32(args[1], XTranslateCoordinates(display, src_w,
			dest_w, src_x, src_y, &dest_x_return, &dest_y_return,
			&child_return));
		*target_dest_x_return = tswap32(dest_x_return);
		*target_dest_y_return = tswap32(dest_y_return);
		*target_child_return = tswap32(child_return);
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XUNGRABKEYBOARD:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Time time = tswap32(args[3]);
		assign_result32(args[1], XUngrabKeyboard(display, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XUNGRABPOINTER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Time time = tswap32(args[3]);
		assign_result32(args[1], XUngrabPointer(display, time));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XUNGRABSERVER:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		assign_result32(args[1], XUngrabServer(display));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XUNIONRECTWITHREGION:
	{
		XRectangle *target_rectangle = (void *) tswap32(args[2]);
		XRectangle rectangle;
		memswap16(&rectangle, target_rectangle, size16of(XRectangle));
		Region src_region = (void *) args[3];
		Region dest_region_return = (void *) args[4];
		assign_result32(args[1], XUnionRectWithRegion(&rectangle,
			src_region, dest_region_return));
		return 0;
	}
	case XUNMAPWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		assign_result32(args[1], XUnmapWindow(display, w));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XUNSETICFOCUS:
	{
		XIC ic = (void *) args[1];
		XUnsetICFocus(ic);
		return 0;
	}
	case XWINDOWEVENT:
	{
		Display *display = get_display(args[2]);
		Window w = tswap32(args[3]);
		long event_mask = tswap32(args[4]);
		XEvent *target_event_return = (void *) tswap32(args[5]);
		XEvent event_return;
		assign_result32(args[1], XWindowEvent(display, w, event_mask,
			&event_return));
		host_to_target_event(target_event_return, &event_return);
		return 0;
	}
	case XWITHDRAWWINDOW:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *display = proxy->native;
		Window w = tswap32(args[3]);
		int screen_number = tswap32(args[4]);
		assign_result32(args[1], XWithdrawWindow(display, w,
			screen_number));
		proxy->request = tswap32(display->request);
		return 0;
	}
	case XWMGEOMETRY:
	{
		Display *display = get_display(args[2]);
		int screen_number = tswap32(args[3]);
		void *user_geometry = (void *) tswap32(args[4]);
		void *default_geometry = (void *) tswap32(args[5]);
		unsigned int border_width = tswap32(args[6]);
		XSizeHints *target_hints = (void *) tswap32(args[7]);
		XSizeHints hints;
		memswap32(&hints, target_hints, size32of(XSizeHints));
		int *target_x_return = (void *) tswap32(args[8]);
		int x_return;
		int *target_y_return = (void *) tswap32(args[9]);
		int y_return;
		int *target_width_return = (void *) tswap32(args[10]);
		int width_return;
		int *target_height_return = (void *) tswap32(args[11]);
		int height_return;
		int *target_gravity_return = (void *) tswap32(args[12]);
		int gravity_return;
		assign_result32(args[1], XWMGeometry(display, screen_number,
			user_geometry, default_geometry, border_width, &hints,
			&x_return, &y_return, &width_return, &height_return,
			&gravity_return));
		*target_x_return = tswap32(x_return);
		*target_y_return = tswap32(y_return);
		*target_width_return = tswap32(width_return);
		*target_height_return = tswap32(height_return);
		*target_gravity_return = tswap32(gravity_return);
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
