#include <err.h>
#include <emu-stub/handler.h>
#include <emu-stub/handler-api.h>
#include <SDL/SDL.h>

#include "libSDL-1.2_const.h"

typedef struct {
	SDL_Palette *palette;
	Uint8 BitsPerPixel;
	Uint8 BytesPerPixel;
	char pad0[4];
	Uint8 Rshift;
	Uint8 Gshift;
	Uint8 Bshift;
	char pad1[1];
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	SDL_PixelFormat *native;
} SDL_PixelFormatProxy;

typedef struct {
	void *seek;
	void *read;
	char pad0[4];
	void *close;
	SDL_RWops *native;
} SDL_RWopsProxy;

typedef struct {
	Uint32 flags;
	SDL_PixelFormatProxy *format;
	int w;
	int h;
	Uint16 pitch;
	void *pixels;
	int offset;
	SDL_Surface *native;
} SDL_SurfaceProxy;

static SDL_SurfaceProxy *video_proxy;

static void assign_pixel_value(unsigned arg,
	__attribute__((unused)) const SDL_PixelFormat *format, Uint32 pixel) {
	Uint32 *result = (void *) tswap32(arg);
#ifdef BSWAP_NEEDED
	*result = pixel << (4 - format->BytesPerPixel) * 8;
#else
	*result = pixel;
#endif
}

static SDL_SurfaceProxy *create_proxy(SDL_Surface *surface) {
	if (!surface)
		return NULL;
	SDL_SurfaceProxy *proxy = malloc(sizeof(SDL_SurfaceProxy));
	proxy->flags = tswap32(surface->flags);
	SDL_PixelFormatProxy *format_proxy
		= malloc(sizeof(SDL_PixelFormatProxy));
	proxy->format = tswap32(format_proxy);
	proxy->w = tswap32(surface->w);
	proxy->h = tswap32(surface->h);
	proxy->pitch = tswap16(surface->pitch);
	proxy->pixels = tswap32(surface->pixels);
// do swap?
	proxy->offset = surface->offset;
	proxy->native = surface;

	SDL_PixelFormat *format = surface->format;
	SDL_Palette *palette = format->palette;
	if (palette) {
		SDL_Palette *palette_proxy = malloc(sizeof(SDL_Palette));
		format_proxy->palette = tswap32(palette_proxy);
		palette_proxy->ncolors = tswap32(palette->ncolors);
		palette_proxy->colors = tswap32(palette->colors);
	}
	else
		format_proxy->palette = NULL;
	format_proxy->BitsPerPixel = format->BitsPerPixel;
	format_proxy->BytesPerPixel = format->BytesPerPixel;
	format_proxy->Rshift = format->Rshift;
	format_proxy->Gshift = format->Gshift;
	format_proxy->Bshift = format->Bshift;
	format_proxy->Rmask = format->Rmask;
	format_proxy->Gmask = format->Gmask;
	format_proxy->Bmask = format->Bmask;
	format_proxy->Amask = format->Amask;
	format_proxy->native = format;

	return proxy;
}

static void free_proxy(SDL_SurfaceProxy *proxy) {
	SDL_PixelFormatProxy *format_proxy = tswap32(proxy->format);
	if (format_proxy->palette)
		free(tswap32(format_proxy->palette));
	free(format_proxy);
	free(proxy);
}

static inline SDL_RWops *get_context(unsigned arg) {
	SDL_RWopsProxy *proxy = (void *) tswap32(arg);
	return proxy->native;
}

static inline SDL_PixelFormat *get_format(unsigned arg) {
	SDL_PixelFormatProxy *proxy = (void *) tswap32(arg);
	return proxy->native;
}

static Uint32 get_pixel_value(unsigned arg,
	__attribute__((unused)) const SDL_Surface *surface) {
#ifdef BSWAP_NEEDED
	return arg >> (4 - surface->format->BytesPerPixel) * 8;
#else
	return arg;
#endif
}

static inline SDL_Surface *get_surface(unsigned arg) {
	SDL_SurfaceProxy *proxy = (void *) tswap32(arg);
	return proxy->native;
}

static void swap_event(SDL_Event *dest, const SDL_Event *src) {
	dest->type = src->type;
	switch (src->type) {
	case SDL_ACTIVEEVENT:
		dest->active.gain = src->active.gain;
		dest->active.state = src->active.state;
		break;
	case SDL_KEYDOWN:
	case SDL_KEYUP:
		dest->key.which = src->key.which;
		dest->key.state = src->key.state;
		dest->key.keysym.scancode = src->key.keysym.scancode;
		dest->key.keysym.sym = tswap32(src->key.keysym.sym);
		dest->key.keysym.mod = tswap32(src->key.keysym.mod);
		dest->key.keysym.unicode = tswap16(src->key.keysym.unicode);
		break;
	case SDL_MOUSEMOTION:
		dest->motion.which = src->motion.which;
		dest->motion.state = src->motion.state;
		dest->motion.x = tswap16(src->motion.x);
		dest->motion.y = tswap16(src->motion.y);
		dest->motion.xrel = tswap16(src->motion.xrel);
		dest->motion.yrel = tswap16(src->motion.yrel);
		break;
	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
		dest->button.which = src->button.which;
		dest->button.button = src->button.button;
		dest->button.state = src->button.state;
		dest->button.x = tswap16(src->button.x);
		dest->button.y = tswap16(src->button.y);
		break;
	case SDL_QUIT:
	case SDL_VIDEOEXPOSE:
		break;
	default:
		warnx("unhandled event type %d", src->type);
	}
}

struct handler_userdata {
	emu_cpu_env *env;
	unsigned callback_args[4];
};

static void callback(void *userdata, Uint8 *stream, int len) {
	struct handler_userdata *hdata = userdata;
	hdata->callback_args[2] = (unsigned) tswap32(stream);
	hdata->callback_args[3] = tswap32(len);
	emu_cpu_run(hdata->env);
}

struct thread_fn_data {
	emu_cpu_env *env;
	unsigned fn_args[2];
};

static int thread_fn(void *data) {
	struct thread_fn_data *fdata = data;
	emu_cpu_run(fdata->env);
	int status = fdata->fn_args[0];
	emu_cpu_free(fdata->env);
	free(fdata);
	return status;
}

int handle_syscall(unsigned args[], emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case SDL_CLOSEAUDIO:
	{
// free the open audio callback, maybe in SDL_Quit too
		SDL_CloseAudio();
		return 0;
	}
	case SDL_CREATERGBSURFACE:
	{
		Uint32 flags = tswap32(args[2]);
		int width = tswap32(args[3]);
		int height = tswap32(args[4]);
		int depth = tswap32(args[5]);
		Uint32 Rmask = args[6];
		Uint32 Gmask = args[7];
		Uint32 Bmask = args[8];
		Uint32 Amask = args[9];
		SDL_Surface *surface = SDL_CreateRGBSurface(flags, width,
			height, depth, Rmask, Gmask, Bmask, Amask);
		assign_result32(args[1], create_proxy(surface));
		return 0;
	}
	case SDL_CREATERGBSURFACEFROM:
	{
		void *pixels = (void *) tswap32(args[2]);
		int width = tswap32(args[3]);
		int height = tswap32(args[4]);
		int depth = tswap32(args[5]);
		int pitch = tswap32(args[6]);
		Uint32 Rmask = args[7];
		Uint32 Gmask = args[8];
		Uint32 Bmask = args[9];
		Uint32 Amask = args[10];
		SDL_Surface *surface = SDL_CreateRGBSurfaceFrom(pixels, width,
			height, depth, pitch, Rmask, Gmask, Bmask, Amask);
		assign_result32(args[1], create_proxy(surface));
		return 0;
	}
	case SDL_CREATESEMAPHORE:
	{
		Uint32 initial_value = tswap32(args[2]);
		SDL_sem **result = (void *) tswap32(args[1]);
		*result = SDL_CreateSemaphore(initial_value);
		return 0;
	}
	case SDL_CREATETHREAD:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		int (*fn)(void *) = (void *) args[2];
		void *data = (void *) args[3];
		struct thread_fn_data *fdata
			= malloc(sizeof(struct thread_fn_data));
		fdata->env = new_env;
		fdata->fn_args[0] = (unsigned) fn;
		fdata->fn_args[1] = (unsigned) data;
		emu_set_return_value(new_env, (unsigned) fdata->fn_args);
		SDL_Thread **result = (void *) tswap32(args[1]);
		*result = SDL_CreateThread(thread_fn, fdata);
		return 0;
	}
	case SDL_DELAY:
	{
		Uint32 ms = tswap32(args[1]);
		SDL_Delay(ms);
		return 0;
	}
	case SDL_DESTROYSEMAPHORE:
	{
		SDL_sem *sem = (void *) args[1];
		SDL_DestroySemaphore(sem);
		return 0;
	}
	case SDL_DISPLAYFORMAT:
	{
		SDL_Surface *surface = get_surface(args[2]);
		SDL_Surface *new_surface = SDL_DisplayFormat(surface);
		assign_result32(args[1], create_proxy(new_surface));
		return 0;
	}
	case SDL_FILLRECT:
	{
		SDL_Surface *dst = get_surface(args[2]);
		SDL_Rect *target_dstrect = (void *) tswap32(args[3]);
		SDL_Rect *dstrect;
		if (target_dstrect) {
			dstrect = alloca(sizeof(SDL_Rect));
			memswap16(dstrect, target_dstrect, size16of(SDL_Rect));
		}
		else
			dstrect = NULL;
		Uint32 color = get_pixel_value(args[4], dst);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_FillRect(dst, dstrect, color);
		if (target_dstrect)
			memswap16(target_dstrect, dstrect, size16of(SDL_Rect));
		return 0;
	}
	case SDL_FLIP:
	{
		SDL_Surface *screen = get_surface(args[2]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_Flip(screen);
		return 0;
	}
	case SDL_FREESURFACE:
	{
		SDL_SurfaceProxy *proxy = (void *) tswap32(args[1]);
		if (proxy && proxy != video_proxy) {
			SDL_FreeSurface(proxy->native);
			free_proxy(proxy);
		}
		return 0;
	}
	case SDL_GETMODSTATE:
	{
		assign_result32(args[1], SDL_GetModState());
		return 0;
	}
	case SDL_GETMOUSESTATE:
	{
		int *target_x = (void *) tswap32(args[2]);
		int x;
		int *target_y = (void *) tswap32(args[3]);
		int y;
		Uint8 *result = (void *) tswap32(args[1]);
		*result = SDL_GetMouseState(&x, &y);
// target_x or target_y may be null
		*target_x = tswap32(x);
		*target_y = tswap32(y);
		return 0;
	}
	case SDL_GETTICKS:
	{
		assign_result32(args[1], SDL_GetTicks());
		return 0;
	}
	case SDL_GL_SETATTRIBUTE:
	{
		SDL_GLattr attr = tswap32(args[2]);
		int value = tswap32(args[3]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_GL_SetAttribute(attr, value);
		return 0;
	}
	case SDL_GL_SWAPBUFFERS:
	{
		SDL_GL_SwapBuffers();
		return 0;
	}
	case SDL_INIT:
	{
		Uint32 flags = tswap32(args[2]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_Init(flags);
		return 0;
	}
	case SDL_INITSUBSYSTEM:
	{
		Uint32 flags = tswap32(args[2]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_InitSubSystem(flags);
		return 0;
	}
	case SDL_JOYSTICKEVENTSTATE:
	{
		int state = tswap32(args[2]);
		assign_result32(args[1], SDL_JoystickEventState(state));
		return 0;
	}
	case SDL_JOYSTICKOPEN:
	{
		int device_index = tswap32(args[2]);
		SDL_Joystick **result = (void *) tswap32(args[1]);
		*result = SDL_JoystickOpen(device_index);
		return 0;
	}
	case SDL_LOADBMP_RW:
	{
		SDL_RWopsProxy *proxy = (void *) tswap32(args[2]);
		SDL_RWops *src = proxy ? proxy->native : NULL;
		int freesrc = args[3];
		SDL_Surface *surface = SDL_LoadBMP_RW(src, freesrc);
		assign_result32(args[1], create_proxy(surface));
		return 0;
	}
	case SDL_LOCKAUDIO:
	{
		SDL_LockAudio();
		return 0;
	}
	case SDL_LOCKSURFACE:
	{
		SDL_Surface *surface = get_surface(args[2]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_LockSurface(surface);
		return 0;
	}
	case SDL_MAPRGB:
	{
		const SDL_PixelFormat *format = get_format(args[2]);
		Uint8 r = tswap32(args[3]);
		Uint8 g = tswap32(args[4]);
		Uint8 b = tswap32(args[5]);
		Uint32 pixel = SDL_MapRGB(format, r, g, b);
		assign_pixel_value(args[1], format, pixel);
		return 0;
	}
	case SDL_MAPRGBA:
	{
		const SDL_PixelFormat *format = get_format(args[2]);
		Uint8 r = tswap32(args[3]);
		Uint8 g = tswap32(args[4]);
		Uint8 b = tswap32(args[5]);
		Uint8 a = tswap32(args[6]);
		Uint32 pixel = SDL_MapRGBA(format, r, g, b, a);
		assign_pixel_value(args[1], format, pixel);
		return 0;
	}
	case SDL_NUMJOYSTICKS:
	{
		assign_result32(args[1], SDL_NumJoysticks());
		return 0;
	}
	case SDL_OPENAUDIO:
	{
		emu_cpu_env *new_env = emu_cpu_clone(env);
		SDL_AudioSpec *target_desired = (void *) tswap32(args[2]);
		SDL_AudioSpec desired;
		desired.freq = tswap32(target_desired->freq);
		desired.format = tswap16(target_desired->format);
		desired.channels = target_desired->channels;
		desired.samples = tswap16(target_desired->samples);
		desired.callback = callback;
		struct handler_userdata *hdata
			= malloc(sizeof(struct handler_userdata));
		hdata->env = new_env;
		hdata->callback_args[0] = (unsigned) target_desired->callback;
		hdata->callback_args[1] = (unsigned) target_desired->userdata;
		emu_set_return_value(new_env, (unsigned) hdata->callback_args);
		desired.userdata = hdata;

		SDL_AudioSpec *target_obtained = (void *) tswap32(args[3]);
		SDL_AudioSpec *obtained = target_obtained ?
			alloca(sizeof(SDL_AudioSpec)) : NULL;
		int *result = (void *) tswap32(args[1]);
		*result = SDL_OpenAudio(&desired, obtained);
		if (obtained) {
			target_obtained->freq = tswap32(obtained->freq);
			target_obtained->format = tswap16(obtained->format);
			target_obtained->channels = obtained->channels;
			target_obtained->silence = obtained->silence;
			target_obtained->samples = tswap16(obtained->samples);
			target_obtained->size = tswap32(obtained->size);
		}
		else {
			target_desired->silence = desired.silence;
			target_desired->size = tswap32(desired.size);
		}
		return 0;
	}
	case SDL_PAUSEAUDIO:
	{
		int pause_on = tswap32(args[1]);
		SDL_PauseAudio(pause_on);
		return 0;
	}
	case SDL_POLLEVENT:
	{
// target_event may be null
		SDL_Event *target_event = (void *) tswap32(args[2]);
		SDL_Event event;
		int pending = SDL_PollEvent(&event);
		assign_result32(args[1], pending);
		if (pending)
			swap_event(target_event, &event);
		return 0;
	}
	case SDL_PUSHEVENT:
	{
		SDL_Event *target_event = (void *) tswap32(args[2]);
		SDL_Event event;
		swap_event(&event, target_event);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_PushEvent(&event);
		return 0;
	}
	case SDL_QUIT__FUNCTION:
	{
		SDL_Quit();
		if (video_proxy) {
			free_proxy(video_proxy);
			video_proxy = NULL;
		}
		return 0;
	}
	case SDL_QUITSUBSYSTEM:
	{
		Uint32 flags = tswap32(args[1]);
		SDL_QuitSubSystem(flags);
		return 0;
	}
	case SDL_RWFROMFILE:
	{
		const char *file = (void *) tswap32(args[2]);
		const char *mode = (void *) tswap32(args[3]);
		SDL_RWops *context = SDL_RWFromFile(file, mode);
		if (!context) {
			assign_result32(args[1], NULL);
			return 0;
		}
		SDL_RWopsProxy *proxy = malloc(sizeof(SDL_RWopsProxy));
		proxy->native = context;
		assign_result32(args[1], proxy);
		return 0;
	}
	case SDL_RWOPS_CLOSE:
	{
		SDL_RWopsProxy *proxy = (void *) tswap32(args[2]);
		SDL_RWops *context = proxy->native;
		int *result = (void *) tswap32(args[1]);
		*result = context->close(context);
		free(proxy);
		return 0;
	}
	case SDL_RWOPS_READ:
	{
		SDL_RWops *context = get_context(args[2]);
		void *ptr = (void *) tswap32(args[3]);
		int size = tswap32(args[4]);
		int maxnum = tswap32(args[5]);
		assign_result32(args[1], context->read(context, ptr, size,
			maxnum));
		return 0;
	}
	case SDL_RWOPS_SEEK:
	{
		SDL_RWops *context = get_context(args[2]);
		int offset = tswap32(args[3]);
		int whence = tswap32(args[4]);
		assign_result32(args[1], context->seek(context, offset,
			whence));
		return 0;
	}
	case SDL_SEMPOST:
	{
		SDL_sem *sem = (void *) args[2];
		int *result = (void *) tswap32(args[1]);
		*result = SDL_SemPost(sem);
		return 0;
	}
	case SDL_SEMWAITTIMEOUT:
	{
		SDL_sem *sem = (void *) args[2];
		Uint32 timeout = tswap32(args[3]);
		assign_result32(args[1], SDL_SemWaitTimeout(sem, timeout));
		return 0;
	}
	case SDL_SETALPHA:
	{
		SDL_Surface *surface = get_surface(args[2]);
		Uint32 flag = tswap32(args[3]);
		Uint32 alpha = tswap32(args[4]);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_SetAlpha(surface, flag, alpha);
		return 0;
	}
	case SDL_SETCOLORKEY:
	{
		SDL_Surface *surface = get_surface(args[2]);
		Uint32 flag = tswap32(args[3]);
		Uint32 key = get_pixel_value(args[4], surface);
		int *result = (void *) tswap32(args[1]);
		*result = SDL_SetColorKey(surface, flag, key);
		return 0;
	}
	case SDL_SETERROR:
	{
		char *strp = (void *) tswap32(args[1]);
		SDL_SetError("%s", strp);
		return 0;
	}
	case SDL_SETVIDEOMODE:
	{
		int width = tswap32(args[2]);
		int height = tswap32(args[3]);
		int bpp = tswap32(args[4]);
		Uint32 flags = tswap32(args[5]);
		SDL_Surface *surface = SDL_SetVideoMode(width, height, bpp,
			flags);
		SDL_SurfaceProxy *proxy = NULL;
		if (surface) {
			if (video_proxy)
				free_proxy(video_proxy);
			proxy = create_proxy(surface);
			video_proxy = proxy;
		}
		assign_result32(args[1], proxy);
		return 0;
	}
	case SDL_SHOWCURSOR:
	{
		int toggle = tswap32(args[2]);
		assign_result32(args[1], SDL_ShowCursor(toggle));
		return 0;
	}
	case SDL_UNLOCKAUDIO:
	{
		SDL_UnlockAudio();
		return 0;
	}
	case SDL_UNLOCKSURFACE:
	{
		SDL_Surface *surface = get_surface(args[1]);
		SDL_UnlockSurface(surface);
		return 0;
	}
	case SDL_UPDATERECT:
	{
		SDL_Surface *screen = get_surface(args[1]);
		Sint32 x = tswap32(args[2]);
		Sint32 y = tswap32(args[3]);
		Uint32 w = tswap32(args[4]);
		Uint32 h = tswap32(args[5]);
		SDL_UpdateRect(screen, x, y, w, h);
		return 0;
	}
	case SDL_UPPERBLIT:
	{
		SDL_Surface *src = get_surface(args[2]);
		const SDL_Rect *target_srcrect = (void *) tswap32(args[3]);
		SDL_Rect *srcrect;
		if (target_srcrect) {
			srcrect = alloca(sizeof(SDL_Rect));
			memswap16(srcrect, target_srcrect, size16of(SDL_Rect));
		}
		else
			srcrect = NULL;
		SDL_Surface *dst = get_surface(args[4]);
		SDL_Rect *target_dstrect = (void *) tswap32(args[5]);
		SDL_Rect *dstrect;
		if (target_dstrect) {
			dstrect = alloca(sizeof(SDL_Rect));
			memswap16(dstrect, target_dstrect, 2);
		}
		else
			dstrect = NULL;
		assign_result32(args[1], SDL_UpperBlit(src, srcrect, dst,
			dstrect));
		if (target_dstrect)
			memswap16(target_dstrect, dstrect, size16of(SDL_Rect));
		return 0;
	}
	case SDL_WAITEVENT:
	{
// target_event may be null
		SDL_Event *target_event = (void *) tswap32(args[2]);
		SDL_Event event;
		assign_result32(args[1], SDL_WaitEvent(&event));
		swap_event(target_event, &event);
		return 0;
	}
	case SDL_WAITTHREAD:
	{
		SDL_Thread *thread = (void *) args[1];
		int *status = (void *) tswap32(args[2]);
		SDL_WaitThread(thread, status);
		return 0;
	}
	case SDL_WARPMOUSE:
	{
		Uint16 x = tswap32(args[1]);
		Uint16 y = tswap32(args[2]);
		SDL_WarpMouse(x, y);
		return 0;
	}
	case SDL_WM_GRABINPUT:
	{
		SDL_GrabMode mode = tswap32(args[2]);
		assign_result32(args[1], SDL_WM_GrabInput(mode));
		return 0;
	}
	case SDL_WM_SETCAPTION:
	{
		const char *title = (void *) tswap32(args[1]);
		const char *icon = (void *) tswap32(args[2]);
		SDL_WM_SetCaption(title, icon);
		return 0;
	}
	case SDL_WM_SETICON:
	{
		SDL_Surface *icon = get_surface(args[1]);
		Uint8 *mask = (void *) tswap32(args[2]);
		SDL_WM_SetIcon(icon, mask);
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
