#include <dlfcn.h>
#include <err.h>
#include <stdlib.h>
#include <GL/glx.h>

#include "libGL_const.h"
#include "libX11.h"

static DisplayProxy *(*find_display_proxy)(Display *display);
static void (*set_visual_proxy)(Visual **visual, DisplayProxy *display_proxy);

static inline float to_float(unsigned x);

__attribute__((constructor)) static void init_extern_symbols(void) {
	static void *handle;
	handle = dlopen("$ORIGIN/libX11.so.6", RTLD_LAZY);
	find_display_proxy = dlsym(handle, "find_display_proxy");
	set_visual_proxy = dlsym(handle, "set_visual_proxy");

	__attribute__((destructor)) void destructor(void) {
		dlclose(handle);
	}
}

static void set_display_proxy_request(Display *dpy, unsigned long request) {
	DisplayProxy *proxy = find_display_proxy(dpy);
	if (proxy)
		proxy->request = tswap32(request);
}

static inline float swapf(float f) {
	return to_float(tswap32(to_unsigned(f)));
}

static inline float to_float(unsigned x) {
	union {
		unsigned i;
		float f;
	} int_float = { .i = x };
	return int_float.f;
}

int handle_syscall(unsigned args[],
	__attribute__((unused)) struct emu_cpu_env *env) {
	unsigned function = tswap32(args[0]);
	switch (function) {
	case GLBEGIN:
	{
		GLenum mode = tswap32(args[1]);
		glBegin(mode);
		return 0;
	}
	case GLBINDTEXTURE:
	{
		GLenum target = tswap32(args[1]);
		GLuint texture = tswap32(args[2]);
		glBindTexture(target, texture);
		return 0;
	}
	case GLBLENDFUNC:
	{
		GLenum sfactor = tswap32(args[1]);
		GLenum dfactor = tswap32(args[2]);
		glBlendFunc(sfactor, dfactor);
		return 0;
	}
	case GLCALLLIST:
	{
		Display *dpy = glXGetCurrentDisplay();
		unsigned long *reqptr = &((_XPrivDisplay) dpy)->request;
		unsigned long request = *reqptr;
		GLuint list = tswap32(args[1]);
		glCallList(list);
		unsigned long new_request = *reqptr;
		if (new_request != request)
			set_display_proxy_request(dpy, new_request);
		return 0;
	}
	case GLCLEAR:
	{
		Display *dpy = glXGetCurrentDisplay();
		unsigned long *reqptr = &((_XPrivDisplay) dpy)->request;
		unsigned long request = *reqptr;
		GLbitfield mask = tswap32(args[1]);
		glClear(mask);
		unsigned long new_request = *reqptr;
		if (new_request != request)
			set_display_proxy_request(dpy, new_request);
		return 0;
	}
	case GLDELETELISTS:
	{
		GLuint list = tswap32(args[1]);
		GLsizei range = tswap32(args[2]);
		glDeleteLists(list, range);
		return 0;
	}
	case GLDELETETEXTURES:
	{
		GLsizei n = tswap32(args[1]);
		const GLuint *target_textures = (void *) tswap32(args[2]);
		GLuint *textures = malloc(n * sizeof(GLuint));
		memswap32(textures, target_textures, n);
		glDeleteTextures(n, textures);
		free(textures);
		return 0;
	}
	case GLDISABLE:
	{
		GLenum cap = tswap32(args[1]);
		glDisable(cap);
		return 0;
	}
	case GLENABLE:
	{
		GLenum cap = tswap32(args[1]);
		glEnable(cap);
		return 0;
	}
	case GLEND:
	{
		glEnd();
		return 0;
	}
	case GLENDLIST:
	{
		Display *dpy = glXGetCurrentDisplay();
		unsigned long *reqptr = &((_XPrivDisplay) dpy)->request;
		unsigned long request = *reqptr;
		glEndList();
		unsigned long new_request = *reqptr;
		if (new_request != request)
			set_display_proxy_request(dpy, new_request);
		return 0;
	}
	case GLFLUSH:
	{
		glFlush();
		return 0;
	}
	case GLFRUSTUM:
	{
#ifdef BSWAP_NEEDED
		static const int I0 = 1;
		static const int I1 = 0;
#else
		static const int I0 = 0;
		static const int I1 = 1;
#endif
		int_s64 is1 = { .i = { tswap32(args[I0 + 1]),
			tswap32(args[I1 + 1]) } };
		int_s64 is2 = { .i = { tswap32(args[I0 + 3]),
			tswap32(args[I1 + 3]) } };
		int_s64 is3 = { .i = { tswap32(args[I0 + 5]),
			tswap32(args[I1 + 5]) } };
		int_s64 is4 = { .i = { tswap32(args[I0 + 7]),
			tswap32(args[I1 + 7]) } };
		int_s64 is5 = { .i = { tswap32(args[I0 + 9]),
			tswap32(args[I1 + 9]) } };
		int_s64 is6 = { .i = { tswap32(args[I0 + 11]),
			tswap32(args[I1 + 11]) } };
		GLdouble left = is1.d;
		GLdouble right = is2.d;
		GLdouble bottom = is3.d;
		GLdouble top = is4.d;
		GLdouble near_val = is5.d;
		GLdouble far_val = is6.d;
		glFrustum(left, right, bottom, top, near_val, far_val);
		return 0;
	}
	case GLGENLISTS:
	{
		GLsizei range = tswap32(args[2]);
		assign_result32(args[1], glGenLists(range));
		return 0;
	}
	case GLGENTEXTURES:
	{
		GLsizei n = tswap32(args[1]);
		GLuint *target_textures = (void *) tswap32(args[2]);
		GLuint *textures = malloc(n * sizeof(GLuint));
		glGenTextures(n, textures);
		memswap32(target_textures, textures, n);
		free(textures);
		return 0;
	}
	case GLGETSTRING:
	{
		GLenum name = tswap32(args[2]);
		assign_result32(args[1], glGetString(name));
		return 0;
	}
	case GLGETTEXIMAGE: {
		GLenum target = tswap32(args[1]);
		GLint level = tswap32(args[2]);
		GLenum format = tswap32(args[3]);
		GLenum type = tswap32(args[4]);
		GLvoid *pixels = (void *) tswap32(args[5]);
		glGetTexImage(target, level, format, type, pixels);
		return 0;
	}
	case GLLIGHTFV:
	{
		GLenum light = tswap32(args[1]);
		GLenum pname = tswap32(args[2]);
		const GLfloat *target_params = (void *) tswap32(args[3]);
		GLfloat params[4];
		int i;
		for (i = 0; i < 4; i++)
			params[i] = swapf(target_params[i]);
		glLightfv(light, pname, params);
		return 0;
	}
	case GLLOADIDENTITY:
	{
		glLoadIdentity();
		return 0;
	}
	case GLMATERIALFV:
	{
		GLenum face = tswap32(args[1]);
		GLenum pname = tswap32(args[2]);
		const GLfloat *target_params = (void *) tswap32(args[3]);
		GLfloat params[4];
		int i;
		for (i = 0; i < 4; i++)
			params[i] = swapf(target_params[i]);
		glMaterialfv(face, pname, params);
		return 0;
	}
	case GLMATRIXMODE:
	{
		GLenum mode = tswap32(args[1]);
		glMatrixMode(mode);
		return 0;
	}
	case GLNEWLIST:
	{
		GLuint list = tswap32(args[1]);
		GLenum mode = tswap32(args[2]);
		glNewList(list, mode);
		return 0;
	}
	case GLNORMAL3F:
	{
		GLfloat nx = to_float(tswap32(args[1]));
		GLfloat ny = to_float(tswap32(args[2]));
		GLfloat nz = to_float(tswap32(args[3]));
		glNormal3f(nx, ny, nz);
		return 0;
	}
	case GLORTHO:
	{
#ifdef BSWAP_NEEDED
		static const int I0 = 1;
		static const int I1 = 0;
#else
		static const int I0 = 0;
		static const int I1 = 1;
#endif
		int_s64 is1 = { .i = { tswap32(args[I0 + 1]),
			tswap32(args[I1 + 1]) } };
		int_s64 is2 = { .i = { tswap32(args[I0 + 3]),
			tswap32(args[I1 + 3]) } };
		int_s64 is3 = { .i = { tswap32(args[I0 + 5]),
			tswap32(args[I1 + 5]) } };
		int_s64 is4 = { .i = { tswap32(args[I0 + 7]),
			tswap32(args[I1 + 7]) } };
		int_s64 is5 = { .i = { tswap32(args[I0 + 9]),
			tswap32(args[I1 + 9]) } };
		int_s64 is6 = { .i = { tswap32(args[I0 + 11]),
			tswap32(args[I1 + 11]) } };
		GLdouble left = is1.d;
		GLdouble right = is2.d;
		GLdouble bottom = is3.d;
		GLdouble top = is4.d;
		GLdouble near_val = is5.d;
		GLdouble far_val = is6.d;
		glOrtho(left, right, bottom, top, near_val, far_val);
		return 0;
	}
	case GLPIXELSTOREI:
	{
		GLenum pname = tswap32(args[1]);
		GLint param = tswap32(args[2]);
		glPixelStorei(pname, param);
		return 0;
	}
	case GLPOPMATRIX:
	{
		glPopMatrix();
		return 0;
	}
	case GLPUSHMATRIX:
	{
		glPushMatrix();
		return 0;
	}
	case GLROTATEF:
	{
		GLfloat angle = to_float(tswap32(args[1]));
		GLfloat x = to_float(tswap32(args[2]));
		GLfloat y = to_float(tswap32(args[3]));
		GLfloat z = to_float(tswap32(args[4]));
		glRotatef(angle, x, y, z);
		return 0;
	}
	case GLSHADEMODEL:
	{
		GLenum mode = tswap32(args[1]);
		glShadeModel(mode);
		return 0;
	}
	case GLTEXCOORD1F:
	{
		GLfloat s = to_float(tswap32(args[1]));
		glTexCoord1f(s);
		return 0;
	}
	case GLTEXCOORD2F:
	{
		GLfloat s = to_float(tswap32(args[1]));
		GLfloat t = to_float(tswap32(args[2]));
		glTexCoord2f(s, t);
		return 0;
	}
	case GLTEXENVI:
	{
		GLenum target = tswap32(args[1]);
		GLenum pname = tswap32(args[2]);
		GLint param = tswap32(args[3]);
		glTexEnvi(target, pname, param);
		return 0;
	}
	case GLTEXIMAGE1D:
	{
		GLenum target = tswap32(args[1]);
		GLint level = tswap32(args[2]);
		GLint internalFormat = tswap32(args[3]);
		GLsizei width = tswap32(args[4]);
		GLint border = tswap32(args[5]);
		GLenum format = tswap32(args[6]);
		GLenum type = tswap32(args[7]);
		GLvoid *pixels = (void *) tswap32(args[8]);
		glTexImage1D(target, level, internalFormat, width, border,
			format, type, pixels);
		return 0;
	}
	case GLTEXIMAGE2D: {
		GLenum target = tswap32(args[1]);
		GLint level = tswap32(args[2]);
		GLint internalFormat = tswap32(args[3]);
		GLsizei width = tswap32(args[4]);
		GLsizei height = tswap32(args[5]);
		GLint border = tswap32(args[6]);
		GLenum format = tswap32(args[7]);
		GLenum type = tswap32(args[8]);
		GLvoid *pixels = (void *) tswap32(args[9]);
		glTexImage2D(target, level, internalFormat, width, height,
			border, format, type, pixels);
		return 0;
	}
	case GLTEXPARAMETERI:
	{
		GLenum target = tswap32(args[1]);
		GLenum pname = tswap32(args[2]);
		GLint param = tswap32(args[3]);
		glTexParameteri(target, pname, param);
		return 0;
	}
	case GLTEXSUBIMAGE2D:
	{
		GLenum target = tswap32(args[1]);
		GLint level = tswap32(args[2]);
		GLint xoffset = tswap32(args[3]);
		GLint yoffset = tswap32(args[4]);
		GLsizei width = tswap32(args[5]);
		GLsizei height = tswap32(args[6]);
		GLenum format = tswap32(args[7]);
		GLenum type = tswap32(args[8]);
		GLvoid *pixels = (void *) tswap32(args[9]);
		glTexSubImage2D(target, level, xoffset, yoffset, width, height,
			format, type, pixels);
		return 0;
	}
	case GLTRANSLATEF:
	{
		GLfloat x = to_float(tswap32(args[1]));
		GLfloat y = to_float(tswap32(args[2]));
		GLfloat z = to_float(tswap32(args[3]));
		glTranslatef(x, y, z);
		return 0;
	}
	case GLVERTEX2F:
	{
		GLfloat x = to_float(tswap32(args[1]));
		GLfloat y = to_float(tswap32(args[2]));
		glVertex2f(x, y);
		return 0;
	}
	case GLVERTEX3F:
	{
		GLfloat x = to_float(tswap32(args[1]));
		GLfloat y = to_float(tswap32(args[2]));
		GLfloat z = to_float(tswap32(args[3]));
		glVertex3f(x, y, z);
		return 0;
	}
	case GLVIEWPORT:
	{
		GLint x = tswap32(args[1]);
		GLint y = tswap32(args[2]);
		GLsizei width = tswap32(args[3]);
		GLsizei height = tswap32(args[4]);
		glViewport(x, y, width, height);
		return 0;
	}
	case GLXCHOOSEVISUAL:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int screen = tswap32(args[3]);
		int *target_attribList = (void *) tswap32(args[4]);
		int n = 0;
		while (target_attribList[n++]) ;
		int *attribList = malloc(n * sizeof(int));
		memswap32(attribList, target_attribList, n);
		XVisualInfo *info = glXChooseVisual(dpy, screen, attribList);
		set_visual_proxy(&info->visual, proxy);
		memswap32(info, info, size32of(XVisualInfo));
		assign_result32(args[1], info);
		free(attribList);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXCREATECONTEXT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		XVisualInfo *target_vis = (void *) tswap32(args[3]);
		XVisualInfo vis;
		memswap32(&vis, target_vis, size32of(XVisualInfo));
		VisualProxy *visual_proxy = (VisualProxy *) vis.visual;
		vis.visual = visual_proxy->native;
		void *shareList = (void *) args[4];
		Bool direct = tswap32(args[5]);
		GLXContext *result = (void *) tswap32(args[1]);
		*result = glXCreateContext(dpy, &vis, shareList, direct);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXDESTROYCONTEXT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		void *ctx = (void *) args[2];
		glXDestroyContext(dpy, ctx);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXGETCLIENTSTRING:
	{
		Display *dpy = get_display(args[2]);
		int name = tswap32(args[3]);
		assign_result32(args[1], glXGetClientString(dpy, name));
		return 0;
	}
	case GLXGETCONFIG:
	{
		Display *dpy = get_display(args[2]);
		XVisualInfo *target_visual = (void *) tswap32(args[3]);
		XVisualInfo visual;
		memswap32(&visual, target_visual, size32of(XVisualInfo));
		VisualProxy *visual_proxy = (VisualProxy *) visual.visual;
		visual.visual = visual_proxy->native;
		int attrib = tswap32(args[4]);
		int *target_value = (void *) tswap32(args[5]);
		int value;
		assign_result32(args[1], glXGetConfig(dpy, &visual, attrib,
			&value));
		*target_value = tswap32(value);
		return 0;
	}
	case __GLXGETSWAPINTERVALMESA:
	{
		PFNGLXGETSWAPINTERVALMESAPROC pglXGetSwapIntervalMESA
			= (PFNGLXGETSWAPINTERVALMESAPROC) glXGetProcAddressARB(
			(const GLubyte *) "glXGetSwapIntervalMESA");
//static int __glXGetSwapIntervalMESA(void) {
		assign_result32(args[1], pglXGetSwapIntervalMESA());
		return 0;
	}
	case GLXISDIRECT:
	{
		Display *dpy = get_display(args[2]);
		void *ctx = (void *) args[3];
		assign_result32(args[1], glXIsDirect(dpy, ctx));
		return 0;
	}
	case GLXMAKECURRENT:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		GLXDrawable drawable = tswap32(args[3]);
		void *ctx = (void *) args[4];
		assign_result32(args[1], glXMakeCurrent(dpy, drawable, ctx));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXQUERYEXTENSION:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int *target_errorb = (void *) tswap32(args[3]);
		int errorb;
		int *target_event = (void *) tswap32(args[4]);
		int event;
		assign_result32(args[1], glXQueryExtension(dpy, &errorb,
			&event));
		*target_errorb = tswap32(errorb);
		*target_event = tswap32(event);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXQUERYEXTENSIONSSTRING:
	{
		Display *dpy = get_display(args[2]);
		int screen = tswap32(args[3]);
		assign_result32(args[1], glXQueryExtensionsString(dpy, screen));
		return 0;
	}
	case GLXQUERYSERVERSTRING:
	{
		DisplayProxy *proxy = (void *) tswap32(args[2]);
		Display *dpy = proxy->native;
		int screen = tswap32(args[3]);
		int name = tswap32(args[4]);
		assign_result32(args[1], glXQueryServerString(dpy, screen,
			name));
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	case GLXQUERYVERSION:
	{
		Display *dpy = get_display(args[2]);
		int *target_maj = (void *) tswap32(args[3]);
		int maj;
		int *target_min = (void *) tswap32(args[4]);
		int min;
		assign_result32(args[1], glXQueryVersion(dpy, &maj, &min));
		*target_maj = tswap32(maj);
		*target_min = tswap32(min);
		return 0;
	}
	case GLXSWAPBUFFERS:
	{
		DisplayProxy *proxy = (void *) tswap32(args[1]);
		Display *dpy = proxy->native;
		GLXDrawable drawable = tswap32(args[2]);
		glXSwapBuffers(dpy, drawable);
		proxy->request = tswap32(((_XPrivDisplay) dpy)->request);
		return 0;
	}
	default:
		warnx("Unimplemented function #%d.", function);
		return -1;
	}
}
