// Sign with eID

#include <err.h>
#include <stdbool.h>
#include <assert_ops/libc.h>
#include <bestiola/err_wrap.h>
#include <nspr/nspr.h>
#include <nss/cert.h>
#include <nss/hasht.h>
#include <nss/nss.h>
#include <nss/pk11pub.h>
#include <nss/secerr.h>
#include <nss/sechash.h>
#include <nss/secpkcs7.h>
#include <sys/stat.h>

struct buffer {
	char *data;
	size_t len;
};

static void log_PR_error(void) {
	PRErrorCode code = PR_GetError();
	const char *s = PR_ErrorToString(code, PR_LANGUAGE_I_DEFAULT);
	warnx("NSS error: %s (%d)", s, code);
}

static CERTCertificate *find_cert_by_derCert(SECItem *derCert) {
	CERTCertList *cert_list = PK11_ListCerts(PK11CertListUser, NULL);
	if (!cert_list)
		return NULL;

	CERTCertificate *result = NULL;
	for (CERTCertListNode *node = CERT_LIST_HEAD(cert_list);
		!CERT_LIST_END(node, cert_list); node = CERT_LIST_NEXT(node)) {
		CERTCertificate *cert = node->cert;
		if (SECITEM_ItemsAreEqual(derCert, &cert->derCert)) {
			result = CERT_DupCertificate(cert);
			break;
		}
	}

	CERT_DestroyCertList(cert_list);
	return result;
}

static void outputfn(void *arg, const char *buf, unsigned long len) {
	if (!buf) {
		warnx("buf is null in outputfn - library failure?");
		log_PR_error();
		return;
        }
	struct buffer *b = arg;
	b->data = assert_realloc(b->data, b->len + len);
	memcpy(b->data + b->len, buf, len);
	b->len += len;
}

static SECItem *file2item(const char *name) {
	FILE *stream = fopen(name, "rb");
        if (!stream)
		return NULL;

	SECItem *dst = NULL;

	struct stat buf;
	assert_fstat(fileno(stream), &buf);

	dst = SECITEM_AllocItem(NULL, NULL, buf.st_size);
	if (!dst)
		goto cleanup_stream;

	assert_fread_noeof(dst->data, 1, buf.st_size, stream);

cleanup_stream:
	assert_fclose(stream);
	return dst;
}

static char *read_password(const char *configdir) {
	char *filename = assert_aprintf("%s/password", configdir);

	SECItem *pw_der = file2item(filename);
	free(filename);
	if (!pw_der) {
		warnx("Could not read password");
		return NULL;
	}

	unsigned int len = pw_der->len;
	if (len > 0 && pw_der->data[len - 1] == '\n')
		len--;
	char *password = assert_strndup((char *)pw_der->data, len);

	SECITEM_FreeItem(pw_der, PR_TRUE);

	return password;
}

static bool sign_file(const char *configdir, const char *data_filename) {
	char *cert_filename = assert_aprintf("%s/signature-cert.der",
		configdir);

	SECItem *derCert = file2item(cert_filename);
	free(cert_filename);
	if (!derCert) {
		warnx("Could not read signature certificate");
		return true;
	}

	bool error = false;

	CERTCertificate *cert = find_cert_by_derCert(derCert);
	if (!cert) {
		warnx("find_cert_by_derCert failed");
		error = true;
		goto cleanup_derCert;
	}
	{

	PK11SlotList *list = PK11_GetAllSlotsForCert(cert, NULL);
	if (!list) {
		warnx("PK11_GetAllSlotsForCert failed");
		error = true;
		goto cleanup_cert;
	}

	char *pw = read_password(configdir);
	if (pw) {
		SECStatus status = SECSuccess;
		int n_login = 0;
		int n_bad_pwd = 0;
		for (PK11SlotListElement *el = list->head; el; el = el->next) {
			PK11SlotInfo *slot = el->slot;
			if (!PK11_NeedLogin(slot))
				continue;
			n_login++;
			SECStatus rv = PK11_CheckUserPassword(slot, pw);
			if (rv != SECSuccess) {
				warnx("PK11_CheckUserPassword failed");
				if (PR_GetError() != SEC_ERROR_BAD_PASSWORD) {
					status = rv;
					break;
				} else
					n_bad_pwd++;
			}
		}
		free(pw);
		if (status != SECSuccess
			|| (n_login > 0 && n_login == n_bad_pwd)) {
			warnx("Authentication failed");
			error = true;
			goto cleanup_list;
		}
	}
	{

	SECItem *data = file2item(data_filename);
	if (!data) {
		warnx("Could not read data");
		error = true;
		goto cleanup_list;
	}

	unsigned char digest_buf[SHA256_LENGTH];
	SECItem digest = {.type = siBuffer, .data = digest_buf,
		.len = SHA256_LENGTH};
	if (HASH_HashBuf(HASH_AlgSHA256, digest.data, data->data, data->len)
		!= SECSuccess)
		error = true;

	SECITEM_FreeItem(data, PR_TRUE);

	if (error)
		goto cleanup_list;

	SEC_PKCS7ContentInfo *cinfo = SEC_PKCS7CreateSignedData(cert,
		certUsageEmailSigner, NULL, SEC_OID_SHA256, &digest, NULL,
		NULL);
	if (!cinfo) {
		warnx("SEC_PKCS7CreateSignedData failed");
		log_PR_error();
		error = true;
		goto cleanup_list;
	}

	if (SEC_PKCS7IncludeCertChain(cinfo, NULL) != SECSuccess) {
		warnx("SEC_PKCS7IncludeCertChain failed");
		log_PR_error();
		error = true;
		goto cleanup_cinfo;
	}

	if (SEC_PKCS7AddSigningTime(cinfo) != SECSuccess) {
		warnx("SEC_PKCS7AddSigningTime failed");
		log_PR_error();
		error = true;
		goto cleanup_cinfo;
	}
	{

	struct buffer p7 = {NULL, 0};
	if (SEC_PKCS7Encode(cinfo, outputfn, &p7, NULL, NULL, NULL)
		!= SECSuccess) {
		warnx("SEC_PKCS7Encode failed");
		log_PR_error();
		error = true;
		goto cleanup_cinfo;
	}

	char *out_filename = assert_aprintf("%s.p7s", data_filename);
	FILE *out = assert_fopen(out_filename, "wb");
	assert_fwrite(p7.data, 1, p7.len, out);
	assert_fclose(out);
	free(out_filename);
	free(p7.data);

	}
cleanup_cinfo:
	SEC_PKCS7DestroyContentInfo(cinfo);

	}
cleanup_list:
	PK11_FreeSlotList(list);

cleanup_cert:
	CERT_DestroyCertificate(cert);

	}
cleanup_derCert:
	SECITEM_FreeItem(derCert, PR_TRUE);
	return error;
}

static char *detect_configdir(void) {
	const char *envvar = getenv("HOME");
	if (!envvar)
		return NULL;

	return assert_aprintf("%s/.eidsign", envvar);
}

int main(int argc, char **argv) {
	if (argc < 2)
		errx_fail("Filename missing");

	char *configdir = detect_configdir();
	if (!configdir)
		errx_fail("Could not detect profile directory");

	PR_Init(PR_USER_THREAD, PR_PRIORITY_LOW, 0);

	if (NSS_Init(configdir) != SECSuccess)
		errx_fail("NSS_Init failed");

	bool error = false;

	if (sign_file(configdir, argv[1]))
		error = true;

	if (NSS_Shutdown() != SECSuccess)
		error = true;

	if (PR_Cleanup() != PR_SUCCESS)
		error = true;

	free(configdir);

	return error? EXIT_FAILURE: EXIT_SUCCESS;
}
