// Cercant Déu

#include <err.h>
#include <gcrypt.h>
#include <unistd.h>
#include <bestiola/base64.h>
#include <libxml/parser.h>
#include <ocell/atomic.h>
#include <ocell/core.h>
#include <ocell/graphic.h>
#include <ocell/image.h>
#include <ocell/super_cast.h>
#include <ocell/stdproc/fade_channel_out.h>
#include <sys/syscall.h>
#include <system/sendmail.h>

enum {
	FPS = 50,

	KID_FADE_STEPS = FPS / 2
};

#define FADE_CAST(p, t) SUPER_CAST(p, t, fade, process, p)

typedef struct fade fade;
typedef struct {
	int alpha_i;
	bool in;
} fade_private;

typedef struct {
	int junction_count;
	int *junctions;
	int kids;
	int start_count;
	int *starts;
	int y;
} floor;

#define GOD_CAST(p, t) SUPER_CAST(p, t, god, process, p)

typedef struct god god;
typedef struct {
	int alpha_i;
} god_private;

#define KID_CAST(p, t) SUPER_CAST(p, t, kid, process, p)

typedef struct kid kid;
typedef struct {
	int alpha_i;
	floor *f;
	int goal;
	bool leaving;
	int speed;
	char vector[64];
} kid_private;

#define TARGET_CAST(p, t) SUPER_CAST(p, t, target, process, p)

typedef struct target target;
typedef struct {
	int alpha_i;
} target_private;

static graphic *black_graph;
static int floor_count;
static floor *floors;
static char god_vector[64];
static bool in_recess;
static bool kid_hit;
static SDL_Surface *kid_surface;
static bool last_mouse_left;
static Sound_Sample *laugh_sample;
static Sound_Sample *recess_sample;
static Sound_Sample *ring_sample;
static Sound_Sample *success_sample;
static bool mouse_left_down;
static size_t private_offset;
static process *rail_p;
static atomic_pointer *sound_channel;
static process *stage_p;
static bool success;
static process *target_p;
static process *target_spot_p;
static graphic *white_graph;

static void cercant_deu_on_kill(process *p);
static void cercant_deu_run(process *p);
static process *clock_create(process *father);
static void clock_run(process *p);
static fade *fade_create(process *father, graphic *graph, bool in);
static fade_private *fade_get_private(fade *f);
static void fade_run(process *p);
static size_t fade_sizeof(void);
static void floor_create(process *p, xmlNode *node, floor *f);
static god *god_create(process *father);
static god_private *god_get_private(god *g);
static void god_on_kill(process *p);
static void god_run(process *p);
static size_t god_sizeof(void);
static int get_int_prop(xmlNode *node, const char *name);
static bool key_advance(void);
static process *key_handler_create(process *father);
static void key_handler_run(process *p);
static kid_private *kid_get_private(kid *k);
static void kid_on_kill(process *p);
static void kid_run(process *p);
static size_t kid_sizeof(void);
static process *kids_create(process *father);
static void kids_run(process *p);
static xmlDoc *load_stage_desc(const char *file_name);
static process *mouse_monitor_create(process *father);
static void mouse_monitor_run(process *p);
static void no_render(process *p);
static process *rail_create(process *father, xmlNode *node);
static void rail_on_kill(process *p);
static process *recess_create(process *father);
static void recess_run(process *p);
static void send_email(void);
static process *stage_create(process *father);
static void stage_on_kill(process *p);
static target *target_create(process *father);
static target_private *target_get_private(target *t);
static void target_on_kill(process *p);
static void target_run(process *p);
static size_t target_sizeof(void);
static process *target_spot_create(process *father);
static void target_spot_on_kill(process *p);
static void target_spot_run(process *p);

int main(void) {
	private_offset = process_sizeof();
	process *p = alloc(process_sizeof());
	program_init(p);
	p->on_kill = cercant_deu_on_kill;
	p->run = cercant_deu_run;
	run_program();
	return EXIT_SUCCESS;
}

static process *background_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->x = get_screen()->w / 2;
	p->y = get_screen()->h / 2;
	p->z = 100;
	p->graph = white_graph;
	p->run(p);
	return p;
}

static void cercant_deu_on_kill(__attribute__((unused)) process *p) {
	if (black_graph)
		graphic_destroy(black_graph);
	if (white_graph)
		graphic_destroy(white_graph);

	if (kid_surface)
		SDL_FreeSurface(kid_surface);

	remove_all_channels();
	if (laugh_sample)
		Sound_FreeSample(laugh_sample);
	if (recess_sample)
		Sound_FreeSample(recess_sample);
	if (ring_sample)
		Sound_FreeSample(ring_sample);
	if (success_sample)
		Sound_FreeSample(success_sample);

	if (sound_channel)
		atomic_destroy(sound_channel);
}

static void cercant_deu_run(process *p) {
	if (p->state)
		goto *p->state;

	set_title("Cercant Déu");
	full_screen = true;

	key_handler_create(p);
	set_fps(FPS, 0);
	set_mode(640, 480);

	black_graph = graphic_from_file("black.png");
	white_graph = graphic_from_file("white.png");

	kid_surface = load_image("kid.png");

	laugh_sample = load_sample("laugh.wav");
	recess_sample = load_sample("recess.ogg");
	ring_sample = load_sample("ring.ogg");
	success_sample = load_sample("success.ogg");

	sound_channel = atomic_create();

	background_create(p);
	stage_p = stage_create(p);
	fade_create(p, black_graph, true);
	FRAME(p);

	mouse_monitor_create(p);
	target_create(p);
	while (target_p->alpha != OPAQUE)
		FRAME(p);

	static process *clock_p;
	clock_p = clock_create(p);
	static process *kids_p;
	kids_p = kids_create(p);

	while (target_p)
		FRAME(p);

	fade_channel_out_create(p, sound_channel, 500);
	while (get(sound_channel))
		FRAME(p);

	fade_create(p, white_graph, false);
	FRAME(p);

	send_signal_tree(stage_p, S_KILL, true);
	send_signal_tree(clock_p, S_KILL, true);
	send_signal_tree(kids_p, S_KILL, true);

	remove_all_channels();

	god_create(p);
	send_email();

	play_sample(success_sample, 0, sound_channel);
	while (get(sound_channel))
		FRAME(p);

	while (!key_advance())
		FRAME(p);

	fade_create(p, black_graph, false);
	FRAME(p);

	let_me_alone(p);
	send_signal(p, S_KILL);
}

static process *clock_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->priority = 1;
	p->run = clock_run;
	p->run(p);
	return p;
}

static void clock_run(process *p) {
	static int class_time;
	if (class_time)
		class_time--;
	else {
		class_time = 30 * 60 * FPS - 1;
		recess_create(p);
	}
	frame_done(p);
}

static void create_stage_objects(process *p) {
	xmlDoc *document = load_stage_desc("stage.xml");
	xmlNode *root = xmlDocGetRootElement(document);
	floor_count = 0;
	for (xmlNode *node = root->children; node; node = node->next) {
		if (xmlStrEqual(node->name, xmlCast("floor")))
			floor_count++;
	}
	floors = alloc(floor_count * sizeof(floor));
	floor *f = floors;
	for (xmlNode *node = root->children; node; node = node->next) {
		if (xmlStrEqual(node->name, xmlCast("floor")))
			floor_create(p, node, f++);
	}
	xmlFreeDoc(document);
}

static void evaluate_vector(const char *vector) {
	char digest[32];
	gcry_md_hash_buffer(GCRY_MD_SHA256, &digest, vector, 64);
	for (int i = 0; i < 32; i++) {
		if (digest[i])
			return;
	}
	success = true;
	memcpy(god_vector, vector, sizeof god_vector);
}

static fade *fade_create(process *father, graphic *graph, bool in) {
	fade *f_p = alloc(fade_sizeof());
	process *p = FADE_CAST(f_p, process);
	process_init(p, father);
	p->x = get_screen()->w / 2;
	p->y = get_screen()->h / 2;
	p->z = -100;
	p->graph = graph;
	p->run = fade_run;
	fade_private *f = fade_get_private(f_p);
	f->in = in;

	p->run(p);
	return f_p;
}

static fade_private *fade_get_private(fade *f) {
	void *p = f;
	return p + private_offset;
}

static void fade_run(process *p) {
	fade *f_p = (fade *) p;
	fade_private *f = fade_get_private(f_p);
	if (p->state)
		goto *p->state;

	send_signal(p->father, S_FREEZE);

	const int STEPS = FPS / 2;
	if (f->in) {
		for (f->alpha_i = STEPS; f->alpha_i >= 0; f->alpha_i--) {
			p->alpha = OPAQUE * f->alpha_i / STEPS;
			FRAME(p);
		}
	}
	else {
		for (f->alpha_i = 0; f->alpha_i <= STEPS; f->alpha_i++) {
			p->alpha = OPAQUE * f->alpha_i / STEPS;
			FRAME(p);
		}
	}

	send_signal(p->father, S_WAKEUP);
	send_signal(p, S_KILL);
}

static size_t fade_sizeof(void) {
	static size_t size;
	if (!size)
		size = private_offset + sizeof(fade_private);
	return size;
}

static void floor_create(process *p, xmlNode *node, floor *f) {
	f->kids = 0;
	f->y = get_int_prop(node, "y");
	f->junction_count = 0;
	f->start_count = 0;
	for (xmlNode *child = node->children; child; child = child->next) {
		if (xmlStrEqual(child->name, xmlCast("junction")))
			f->junction_count++;
		else if (xmlStrEqual(child->name, xmlCast("start")))
			f->start_count++;
	}
	f->junctions = alloc(f->junction_count * sizeof(int));
	f->starts = alloc(f->start_count * sizeof(int));
	int ji = 0;
	int si = 0;
	for (xmlNode *child = node->children; child; child = child->next) {
		if (xmlStrEqual(child->name, xmlCast("junction")))
			f->junctions[ji++] = get_int_prop(child, "x");
		else if (xmlStrEqual(child->name, xmlCast("rail")))
			rail_create(p, child);
		else if (xmlStrEqual(child->name, xmlCast("start")))
			f->starts[si++] = get_int_prop(child, "x");
	}
}

static SDL_Surface *generate_kid_surface(const char *vector) {
	static const char HAIR_TEMPLATE[3] = {0xff, 0x00, 0x00};
	static const char SKIN_TEMPLATE[3] = {0xff, 0xff, 0xff};
	static const char ROBE_TEMPLATE[3] = {0x00, 0xff, 0x00};
	static const char SHOE_TEMPLATE[3] = {0x00, 0x00, 0xff};

	static const char HAIR_COLORS[4][3] = {{0xd0, 0xd0, 0xc0},
		{0x90, 0x50, 0x20}, {0x00, 0x00, 0x10}, {0x70, 0x30, 0x20}};
	char hair_color[3];
	memcpy(hair_color, HAIR_COLORS[vector[0] >> 4 & 0x03], 3);
	hair_color[0] |= vector[0] & 0x0f;
	hair_color[1] |= vector[1] >> 4 & 0x0f;
	hair_color[2] |= vector[1] & 0x0f;

	static const char SKIN_COLORS[4][3] = {{0xe0, 0xb0, 0xa0},
		{0x50, 0x30, 0x20}, {0xc0, 0x90, 0x60}, {0xa0, 0x60, 0x30}};
	char skin_color[3];
	memcpy(skin_color, SKIN_COLORS[vector[2] >> 4 & 0x03], 3);
	skin_color[0] |= vector[2] & 0x0f;
	skin_color[1] |= vector[3] >> 4 & 0x0f;
	skin_color[2] |= vector[3] & 0x0f;

	char robe_color[3] = {vector[4], vector[5], vector[6]};
	char shoe_color[3] = {vector[7], vector[8], vector[9]};

	int width = kid_surface->w;
	int height = kid_surface->h;
	SDL_Surface *surface = new_map(width, height);
	if (SDL_BlitSurface(kid_surface, NULL, surface, NULL)) {
		warnx("SDL_BlitSurface: %s", SDL_GetError());
		fatal_error();
	}

	for (int y = 0; y < height; y++) {
		for (int x = 0; x < width; x++) {
			char *pixel = surface->pixels + y * surface->pitch
				+ x * 4;
			if (pixel[3] == 0x00)
				continue;
			if (!memcmp(pixel, HAIR_TEMPLATE, 3))
				memcpy(pixel, hair_color, 3);
			else if (!memcmp(pixel, SKIN_TEMPLATE, 3))
				memcpy(pixel, skin_color, 3);
			else if (!memcmp(pixel, ROBE_TEMPLATE, 3))
				memcpy(pixel, robe_color, 3);
			else if (!memcmp(pixel, SHOE_TEMPLATE, 3))
				memcpy(pixel, shoe_color, 3);
		}
	}

	return surface;
}

static int get_int_prop(xmlNode *node, const char *name) {
	char *value = xmlUncast(xmlGetProp(node, xmlCast(name)));
	int prop = atoi(value);
	free(value);
	return prop;
}

static god *god_create(process *father) {
	god *g_p = alloc(god_sizeof());
	process *p = GOD_CAST(g_p, process);
	process_init(p, father);
	p->x = get_screen()->w / 2;
	p->y = get_screen()->h / 2;
	p->size = 400;
	p->on_kill = god_on_kill;
	p->run = god_run;

	SDL_Surface *surface = generate_kid_surface(god_vector);
	SDL_Surface *surface1 = new_map(surface->w + 2, surface->h + 2);
	for (int y = 0; y < surface->h; y++) {
		char *pixels = surface->pixels + y * surface->pitch;
		char *pixels1 = surface1->pixels + (y + 1) * surface1->pitch
			+ 4;
		memcpy(pixels1, pixels, 4 * surface->w);
	}
	p->graph = graphic_from_surface(surface1);

	god_private *g = god_get_private(g_p);
	g->alpha_i = 0;

	p->run(p);
	return g_p;
}

static god_private *god_get_private(god *g) {
	void *p = g;
	return p + private_offset;
}

static void god_on_kill(process *p) {
	graphic_destroy(p->graph);
}

static void god_run(process *p) {
	const int STEPS = FPS;
	god *g_p = (god *) p;
	god_private *g = god_get_private(g_p);
	if (g->alpha_i < STEPS) {
		g->alpha_i++;
		p->alpha = OPAQUE * g->alpha_i / STEPS;
	}
	frame_done(p);
}

static size_t god_sizeof(void) {
	static size_t size;
	if (!size)
		size = private_offset + sizeof(god_private);
	return size;
}

static bool key_advance(void) {
	return key_down(_SPACE) || key_down(_ENTER) || mouse_left_down;
}

static process *key_handler_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->run = key_handler_run;
	p->run(p);
	return p;
}

static void key_handler_run(process *p) {
	if (p->state)
		goto *p->state;

	for (;;) {
		if ((key(_ALT) && key_down(_F4)) || key_down(_ESC)
			|| quit_requested) {
			let_me_alone(p);
			break;
		}
		if (key(_ALT) && key_down(_ENTER)) {
			full_screen = !full_screen;
			set_mode(640, 480);
		}

		FRAME(p);
	}
	send_signal(p, S_KILL);
}

static kid *kid_create(process *father, int max_per_floor) {
	kid *k_p = alloc(kid_sizeof());
	process *p = KID_CAST(k_p, process);
	process_init(p, father);
	p->z = random_between(0, 1)? 25: 75;
	p->on_kill = kid_on_kill;
	p->run = kid_run;

	kid_private *k = kid_get_private(k_p);

	int floor_i = random_between(0, floor_count - 1);
	while (floors[floor_i].kids >= max_per_floor)
		floor_i = (floor_i + 1) % floor_count;

	floor *f = floors + floor_i;
	k->f = f;
	f->kids++;

	int start_i = random_between(0, f->start_count - 1);
	p->x = f->starts[start_i];
	p->y = f->y;

	k->alpha_i = KID_FADE_STEPS;
	k->goal = 0;
	k->leaving = false;
	k->speed = 1;

	if (syscall(SYS_getrandom, k->vector, sizeof k->vector, 0) < 0) {
		warnx("getrandom failed");
		fatal_error();
	}
	SDL_Surface *surface = generate_kid_surface(k->vector);
	p->graph = graphic_from_surface(surface);

	p->run(p);
	return k_p;
}

static kid_private *kid_get_private(kid *k) {
	void *p = k;
	return p + private_offset;
}

static void kid_on_kill(process *p) {
	graphic_destroy(p->graph);
}

static void kid_run(process *p) {
	kid *k_p = (kid *) p;
	kid_private *k = kid_get_private(k_p);

	if (k->leaving) {
		k->alpha_i--;
		if (!k->alpha_i) {
			k->f->kids--;
			evaluate_vector(k->vector);
			send_signal(p, S_KILL);
			return;
		}
	}
	else if (mouse_left_down && !kid_hit && !success
		&& collision(p, target_spot_p)
		&& !collision(target_spot_p, rail_p) && (p->z == 25
		|| !collision(target_spot_p, stage_p))) {
		kid_hit = true;
		k->leaving = true;
		k->alpha_i = KID_FADE_STEPS - 1;
		play_sample(laugh_sample, 0, NULL);
	}
	else {
		for (int i = 0; i < k->speed; i++) {
			for (int j = 0; j < k->f->junction_count; j++) {
				if (p->x == k->f->junctions[j]) {
					p->z = random_between(0, 1)? 25: 75;
					break;
				}
			}
			if (k->goal > 0) {
				k->goal--;
				if (p->x < 670)
					p->x++;
			} else if (k->goal < 0) {
				k->goal++;
				if (p->x > -30)
					p->x--;
			}
			else {
				k->goal = random_between(-320, 320);
				k->speed = random_between(2, 6);
				break;
			}
		}
	}

	p->alpha = OPAQUE * k->alpha_i / KID_FADE_STEPS;
	frame_done(p);
}

static size_t kid_sizeof(void) {
	static size_t size;
	if (!size)
		size = private_offset + sizeof(kid_private);
	return size;
}

static process *kids_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->run = kids_run;
	p->run(p);
	return p;
}

static int kids_in_stage(void) {
	int kids = 0;
	for (int i = 0; i < floor_count; i++)
		kids += floors[i].kids;
	return kids;
}

static void kids_run(process *p) {
	static int wait_time;
	if (wait_time) {
		wait_time--;
		if (in_recess && wait_time > FPS)
			wait_time = FPS;
	}
	else if (in_recess) {
		if (kids_in_stage() < 8 * floor_count)
			kid_create(p, 8);
		wait_time = random_between(FPS / 2, FPS);
	}
	else {
		if (kids_in_stage() < 2 * floor_count)
			kid_create(p, 2);
		wait_time = random_between(2 * FPS, 3 * FPS);
	}

	frame_done(p);
}

static xmlDoc *load_stage_desc(const char *file_name) {
	char *full = full_name(file_name);
	xmlDoc *document = xmlParseFile(full);
	free(full);
	if (!document)
		fatal_error();
	return document;
}

static process *mouse_monitor_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->priority = 2;
	p->run = mouse_monitor_run;
	p->run(p);
	return p;
}

static void mouse_monitor_run(process *p) {
	mouse_left_down = mouse.left && !last_mouse_left;
	last_mouse_left = mouse.left;
	frame_done(p);
}

static void no_render(__attribute__((unused)) process *p) {
}

static process *rail_create(process *father, xmlNode *node) {
	process *p = alloc(process_sizeof());
	rail_p = p;
	process_init(p, father);
	p->x = get_screen()->w / 2;
	p->y = get_int_prop(node, "y");
	p->z = 10;
	p->graph = graphic_from_file("rail.png");
	p->on_kill = rail_on_kill;

	p->run(p);
	return p;
}

static void rail_on_kill(process *p) {
	graphic_destroy(p->graph);
}

static process *recess_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->run = recess_run;
	p->run(p);
	return p;
}

static void recess_run(process *p) {
	if (p->state)
		goto *p->state;

	if (!success) {
		play_sample(ring_sample, 0, sound_channel);
		while (get(sound_channel))
			FRAME(p);
	}

	in_recess = true;

	if (!success) {
		play_sample(recess_sample, 0, sound_channel);
		while (get(sound_channel))
			FRAME(p);
	}

	in_recess = false;
	send_signal(p, S_KILL);
}

static void send_email(void) {
	const char *logname = getenv("LOGNAME");
	char *b64 = base64_encode_buffer(god_vector, 64);
	if (!b64)
		fatal_error();

	char b64_1[76 + 1];
	char b64_2[12 + 1];
	memcpy(b64_1, b64, 76);
	b64_1[76] = '\0';
	memcpy(b64_2, b64 + 76, 12);
	b64_2[12] = '\0';
	free(b64);

	char *message = ocell_asprintf("\
From javier@jasp.net Thu Jan  1 00:00:00 UTC 1970\n\
Subject: You have found God!\n\
From: Javier Serrano Polo <javier@jasp.net>\n\
To: %s <%s@localhost>\n\
Content-Type: multipart/mixed; boundary=\"-:\"\n\
Mime-Version: 1.0\n\
\n\
---:\n\
Content-Type: text/plain\n\
Content-Transfer-Encoding: 7bit\n\
\n\
Congratulations!\n\
\n\
Your God vector is attached.\n\
---:\n\
Content-Type: application/octet-stream; name=\"vector\"\n\
Content-Disposition: attachment; filename=\"vector\"\n\
Content-Transfer-Encoding: base64\n\
\n\
%s\n\
%s\n\
---:--\n\
", logname, logname, b64_1, b64_2);
	system_sendmail(message);
	free(message);
}

static process *stage_create(process *father) {
	process *p = alloc(process_sizeof());
	process_init(p, father);
	p->x = get_screen()->w / 2;
	p->y = get_screen()->h / 2;
	p->z = 50;
	p->graph = graphic_from_file("stage.png");
	p->on_kill = stage_on_kill;

	create_stage_objects(p);

	p->run(p);
	return p;
}

static void stage_on_kill(process *p) {
	graphic_destroy(p->graph);

	for (int i = 0; i < floor_count; i++) {
		floor *f = floors + i;
		free(f->junctions);
		free(f->starts);
	}
	free(floors);
	floor_count = 0;
	floors = NULL;
}

static target *target_create(process *father) {
	target *t_p = alloc(target_sizeof());
	process *p = TARGET_CAST(t_p, process);
	target_p = p;
	process_init(p, father);
	p->priority = 1;
	p->z = -50;
	p->graph = graphic_from_file("target.png");
	p->on_kill = target_on_kill;
	p->run = target_run;

	target_private *t = target_get_private(t_p);
	t->alpha_i = 0;

	p->run(p);

	target_spot_create(p);

	return t_p;
}

static target_private *target_get_private(target *t) {
	void *p = t;
	return p + private_offset;
}

static void target_on_kill(process *p) {
	graphic_destroy(p->graph);
	target_p = NULL;
}

static void target_run(process *p) {
	const int STEPS = FPS / 10;
	target *t_p = (target *) p;
	target_private *t = target_get_private(t_p);
	p->x = mouse.x;
	p->y = mouse.y;

	if (success) {
		t->alpha_i--;
		if (!t->alpha_i) {
			send_signal_tree(p, S_KILL, true);
			return;
		}
	}
	else if (t->alpha_i < STEPS)
		t->alpha_i++;

	p->alpha = OPAQUE * t->alpha_i / STEPS;
	frame_done(p);
}

static size_t target_sizeof(void) {
	static size_t size;
	if (!size)
		size = private_offset + sizeof(target_private);
	return size;
}

static process *target_spot_create(process *father) {
	process *p = alloc(process_sizeof());
	target_spot_p = p;
	process_init(p, father);
	p->priority = father->priority;
	p->on_kill = target_spot_on_kill;
	p->render = no_render;
	p->run = target_spot_run;

	SDL_Surface *surface = new_map(1, 1);
	char *pixels = surface->pixels;
	for (int i = 0; i < 4; i++)
		pixels[i] = 0xff;
	p->graph = graphic_from_surface(surface);

	p->run(p);
	return p;
}

static void target_spot_on_kill(process *p) {
	graphic_destroy(p->graph);
	target_spot_p = NULL;
}

static void target_spot_run(process *p) {
	p->x = mouse.x;
	p->y = mouse.y;
	kid_hit = false;
	frame_done(p);
}
