// Print a file with expected attributes to standard ouput

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert_ops/libc.h>
#include <bestiola/err_wrap.h>
#include <sys/stat.h>

static void check_mode(FILE *stream, mode_t mode);
static FILE *open_regular_path(const char *path);

int main(int argc, char **argv) {
	if (argc < 4)
		errx_fail("Missing arguments");

	const char *path = argv[1];
	FILE *stream = open_regular_path(path);

	mode_t mode = assert_strtoul(argv[2], 8);
	check_mode(stream, mode);

	size_t size = assert_strtoul(argv[3], 10);
	static char buffer[BUFSIZ];
	for (size_t remaining = size; remaining;) {
		size_t n = remaining < BUFSIZ ? remaining : BUFSIZ;
		assert_fread_noeof(buffer, 1, n, stream);
		assert_fwrite(buffer, 1, n, stdout);
		remaining -= n;
	}

	assert_fclose(stream);

	return EXIT_SUCCESS;
}

static void check_fd(int fd) {
	if (fd == -1)
		errx_fail("Path error");
}

static void check_mode(FILE *stream, mode_t mode) {
	struct stat buf;
	assert_fstat(fileno(stream), &buf);
	if ((buf.st_mode & 0777) != mode)
		errx_fail("Unexpected mode");
}

static FILE *open_regular_path(const char *path) {
	char *components = assert_strdup(path);
	int dirfd = AT_FDCWD;
	char *comp = components;
	if (*comp == '/') {
		comp++;
		dirfd = assert_open("/", O_RDONLY | O_DIRECTORY | O_NOFOLLOW);
	}
	for (;;) {
		char *slash = strchr(comp, '/');
		if (!slash)
			break;
		*slash = '\0';
		int fd = openat(dirfd, comp,
			O_RDONLY | O_DIRECTORY | O_NOFOLLOW);
		check_fd(fd);
		if (dirfd != AT_FDCWD)
			assert_close(dirfd);
		dirfd = fd;
		comp = slash + 1;
	}
	int fd = openat(dirfd, comp, O_RDONLY | O_NOFOLLOW);
	check_fd(fd);
	if (dirfd != AT_FDCWD)
		assert_close(dirfd);
	free(components);
	return assert_fdopen(fd, "r");
}
